/**********************************************************************************
 *
 * @file    main.c
 * @brief   main C file
 *
 * @date    9 Jan. 2023
 * @author  AE Team
 * @note
 *          Change Logs:
 *          Date            Author          Notes
 *          9 Jan. 2023     Shiwa           the first version
 *
 * Copyright (C) Shanghai Eastsoft Microelectronics Co. Ltd. All rights reserved.
 *
 * SPDX-License-Identifier: Apache-2.0
 *
 * Licensed under the Apache License, Version 2.0 (the License); you may
 * not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an AS IS BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 *
 **********************************************************************************
 */
/* Includes ------------------------------------------------------------------ */
#include "main.h"
#include "wm_i2s.h"
#include "wm8978.h"
/* Private Variables --------------------------------------------------------- */
int16_t g_audio_buf[1024];
static int16_t *const s_send_buf = g_audio_buf;
static int16_t *const s_recv_buf = g_audio_buf + 512;
static int16_t *s_send_ptr = s_send_buf;
static int16_t *s_recv_ptr = s_recv_buf;
/* Private Constants --------------------------------------------------------- */
/* Private function prototypes ----------------------------------------------- */
void dma_test_m2m(uint32_t ch, void *src, void *dst, uint32_t len, uint32_t datawidth);
/* Private Macros ------------------------------------------------------------ */

/** @addtogroup Projects_Examples_MD
  * @{
  */

/** @addtogroup Examples_SPI
  * @{
  */
/* Public Variables ---------------------------------------------------------- */
struct wm_i2s_info g_wm_info;
/* Private Function ---------------------------------------------------------- */
/**
  * @brief  callback to set I2S send data
  * @retval No meaning
  */
uint16_t feed_data_copy(void *buffer, uint16_t length)
{
    s_send_ptr = buffer;

    for (uint32_t i = 0; i < length; i++)
        s_send_ptr[i] = s_recv_ptr[i];

    return length;
}
/**
  * @brief  callback to get received data
  * @retval No meaning
  */
uint16_t recv_data_copy(void *buffer, uint16_t length)
{
    s_recv_ptr = buffer;
    return length;
}
/**
  * @brief  Initializate i2c pin
  * @retval None.
  */
static void i2c_pin_init(void)
{
    md_gpio_init_t l_gpio;

    l_gpio.type  = MD_GPIO_TYPE_CMOS;
    l_gpio.odos  = MD_GPIO_OPEN_DRAIN;
    l_gpio.pupd  = MD_GPIO_PUSH_UP;
    l_gpio.odrv  = MD_GPIO_OUT_DRIVE_NORMAL;
    l_gpio.flt   = MD_GPIO_FILTER_DISABLE;
    l_gpio.func  = MD_GPIO_FUNC_1;
    l_gpio.mode  = MD_GPIO_MODE_OUTPUT;
    md_gpio_init(GPIOB, MD_GPIO_PIN_2, &l_gpio);   /*Initialize PB2 for soft I2C SCL pins*/
    md_gpio_init(GPIOB, MD_GPIO_PIN_3, &l_gpio);   /*Initialize PB3 for soft I2C SDA pin*/
}
/**
  * @brief  Initializate i2s pin
  * @retval None.
  */
static void i2s_pin_init(void)
{
    md_gpio_init_t l_gpio;

    l_gpio.type  = MD_GPIO_TYPE_CMOS;
    l_gpio.odos  = MD_GPIO_PUSH_PULL;
    l_gpio.pupd  = MD_GPIO_PUSH_DOWN;
    l_gpio.odrv  = MD_GPIO_OUT_DRIVE_STRONG;
    l_gpio.flt   = MD_GPIO_FILTER_DISABLE;
    l_gpio.func  = MD_GPIO_FUNC_2;
    l_gpio.mode  = MD_GPIO_MODE_OUTPUT;
    md_gpio_init(GPIOA, MD_GPIO_PIN_4, &l_gpio);   /*Initialize PA4 9  WS  for NSS pins*/
    md_gpio_init(GPIOA, MD_GPIO_PIN_5, &l_gpio);   /*Initialize PA5 10 CK  for SCK pin*/
    md_gpio_init(GPIOA, MD_GPIO_PIN_7, &l_gpio);   /*Initialize PA7 12 SDO for MOSI pin*/

    l_gpio.mode  = MD_GPIO_MODE_INPUT;
    md_gpio_init(GPIOA, MD_GPIO_PIN_6, &l_gpio);   /*Initialize PA6 11 SDI for MISO pins*/

    l_gpio.func  = MD_GPIO_FUNC_7;
    l_gpio.mode  = MD_GPIO_MODE_OUTPUT;
    md_gpio_init(GPIOA, MD_GPIO_PIN_8, &l_gpio);   /*Initialize PA8 5  MCK for MCLK pin*/
}
/**
  * @brief  Initialize WM8978 and I2S module
  * @retval None.
  */
void wm8978_i2s_init(void)
{
    i2c_pin_init();

    if (WM8978_Init(I2C0))
    {
        printf("WM8978 I2C init failed\r\n");
        return;
    }

    printf("WM8978 Init ok\r\n");
    i2s_pin_init();
    g_wm_info.spi_x = (void *)SPI0;
    g_wm_info.buffers[0] = g_audio_buf;
    g_wm_info.buffers[1] = g_audio_buf + 512;
    g_wm_info.buffer_length = 512;
    g_wm_info.feed_data = feed_data_copy;
    g_wm_info.recv_data = recv_data_copy;
    g_wm_info.dma_tx_ch = 0;
    g_wm_info.dma_rx_ch = 1;
    g_wm_info.sample_rate = 16000;
    wm_i2s_init(&g_wm_info);
    printf("I2S init ok\r\n");

    for (uint32_t i = 0; i < 1024; i++)
        g_audio_buf[i] = 0;
}

/**
  * @brief  Test main function
  * @retval Status.
  */
int main(void)
{
    /* Configure system clock */
    md_cmu_pll_config(MD_CMU_PLL_INPUT_HOSC8M, MD_CMU_PLL_OUTPUT_72M);
    md_cmu_clock_config(MD_CMU_CLOCK_PLL, 72000000);

    /* Enable ALL peripheral */
    MD_SYSCFG_UNLOCK();
    md_cmu_enable_perh_all();
    MD_SYSCFG_LOCK();

    csi_vic_enable_sirq(DMA_IRQn);
    __enable_irq();
    mcu_uart_init();    /*To initialize the UART module*/
    wm8978_i2s_init();     /*Initialize WM8978 and I2S module*/
    wm_set_mode(WM_SENDRECV);
    wm_set_vol(0xF, 30); /* Max volume=0x3F(63) */
    wm_start();

    while (1);
}

/**
  * @} Examples_SPI
  */

/**
  * @} Projects_Examples_MD
  */

/************* (C) COPYRIGHT Eastsoft Microelectronics *****END OF FILE****/
