/**********************************************************************************
 *
 * @file    main.c
 * @brief   main C file
 *
 * @date    22 Dec. 2023
 * @author  AE Team
 * @note
 *          Change Logs:
 *          Date            Author          Notes
 *          22 Dec. 2023    Lisq            the first version
 *
 * Copyright (C) Shanghai Eastsoft Microelectronics Co. Ltd. All rights reserved.
 *
 * SPDX-License-Identifier: Apache-2.0
 *
 * Licensed under the Apache License, Version 2.0 (the License); you may
 * not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an AS IS BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 *
 **********************************************************************************
 */
/* Includes ------------------------------------------------------------------ */
#include "main.h"

/* Private Macros ------------------------------------------------------------ */
#define TEST_ADDR       0x10000
#define TEST_ADDR_INV   0x2FFFF
#define TEST_LEN    0x20

/* Private Variables --------------------------------------------------------- */
uint32_t g_iap_flag = FALSE;

/* Public Variables ---------------------------------------------------------- */
/* Private Constants --------------------------------------------------------- */
/* Private function prototypes ----------------------------------------------- */
void iap_exe(void)__attribute((section(".irq_iap_section")));

/* Private Function ---------------------------------------------------------- */

/**
  * @brief  Initialize TOGGLE IO.
  * @param  None
  * @retval None
  */
static void init_gpio(void)
{
    md_gpio_init_t gpio_init;
    /* Initialize TOGGLE IO as output */
    md_gpio_init_struct(&gpio_init);      /* initialize the gpio_init */
    gpio_init.mode = MD_GPIO_MODE_OUTPUT; /* output */
    gpio_init.odos = MD_GPIO_PUSH_PULL;   /* push pull */
    gpio_init.pupd = MD_GPIO_FLOATING;    /* no pull up and pull down */
    gpio_init.type = MD_GPIO_TYPE_CMOS;
    gpio_init.func = MD_GPIO_FUNC_1;      /* GPIO function 1 */
    md_gpio_init(LED_PORT, LED_PIN, &gpio_init);

    /* Initialize TOGGLE IO as output */
    md_gpio_init_struct(&gpio_init);      /* initialize the gpio_init */
    gpio_init.mode = MD_GPIO_MODE_OUTPUT; /* output */
    gpio_init.odos = MD_GPIO_PUSH_PULL;   /* push pull */
    gpio_init.pupd = MD_GPIO_FLOATING;    /* no pull up and pull donw */
    gpio_init.type = MD_GPIO_TYPE_CMOS;
    gpio_init.func = MD_GPIO_FUNC_1;      /* GPIO function 1 */
    md_gpio_init(TOGGLE_PORT, TOGGLE_PIN, &gpio_init);
}

/**
  * @brief:  Initialize AD16C4T to timely overflow funtion
  * @param:  None
  * @retval: None
  */
static void init_timer(void)
{
    md_timer_base_init_t ad16c4t_init;

    md_timer_base_struct_init(&ad16c4t_init);   /* initialize the ad16c4t_init  */
    ad16c4t_init.prescaler = 72 - 1;           /* clk_count: 1MHz */
    ad16c4t_init.clk_div = MD_TIMER_CLOCK_DIV1; /* working clock of dead time and filter */
    ad16c4t_init.mode = MD_TIMER_CNT_MODE_UP;   /* count up */
    ad16c4t_init.period = 10 - 1;             /* period is 10 count clock */
    ad16c4t_init.re_cnt = 1 - 1;               /* 0 repeat count */
    md_timer_base_set_config(AD16C4T, &ad16c4t_init);

    md_timer_enable_uit_interrupt(AD16C4T);    /* enable updata interrupt */

    md_mcu_irq_config(AD16C4T_UP_TRIG_COM_IRQn, 0, ENABLE);

    /* enable AD16C4T */
    md_timer_enable_counter_cnten(AD16C4T);
}

/**
  * @brief:  IAP function
  * @param:  None
  * @retval: None
  */
void iap_exe(void)
{
    uint32_t data_buf[TEST_LEN];

    do
    {
        g_iap_flag = FALSE;

        while (READ_BIT(MSC->FLASHSR, MSC_FLASHSR_BUSY_MSK) == 1);

        FLASH_REG_UNLOCK();
        SET_BIT(MSC->FLASHCR, MSC_FLASHCR_IAPEN_MSK);
        SET_BIT(MSC->FLASHCR, MSC_FLASHCR_FLASHREQ_MSK);
        MODIFY_REG(MSC->FLASHADDR, MSC_FLASHADDR_ADDR_MSK, TEST_ADDR << MSC_FLASHADDR_ADDR_POSS);
        MODIFY_REG(MSC->FLASHADDINV, MSC_FLASHADDINV_ADDRINV_MSK, TEST_ADDR_INV << MSC_FLASHADDINV_ADDRINV_POSS);

        WRITE_REG(MSC->FLASHCMD, 0x00005EA1);

        while (READ_BIT(MSC->FLASHSR, MSC_FLASHSR_BUSY_MSK) == 1);

        while (READ_BIT(MSC->FLASHSR, MSC_FLASHSR_SERA_MSK) == 0);

        CLEAR_BIT(MSC->FLASHCR, MSC_FLASHCR_FLASHREQ_MSK);
        CLEAR_BIT(MSC->FLASHCR, MSC_FLASHCR_IAPEN_MSK);
        FLASH_REG_LOCK();

        while (READ_BIT(MSC->FLASHSR, MSC_FLASHSR_BUSY_MSK) == 1);

        FLASH_REG_UNLOCK();
        SET_BIT(MSC->FLASHCR, MSC_FLASHCR_IAPEN_MSK);
        SET_BIT(MSC->FLASHCR, MSC_FLASHCR_FLASHREQ_MSK);
        MODIFY_REG(MSC->FLASHADDR, MSC_FLASHADDR_ADDR_MSK, TEST_ADDR << MSC_FLASHADDR_ADDR_POSS);
        MODIFY_REG(MSC->FLASHADDINV, MSC_FLASHADDINV_ADDRINV_MSK, TEST_ADDR_INV << MSC_FLASHADDINV_ADDRINV_POSS);
        WRITE_REG(MSC->FLASHDR, 0x12345678);

        WRITE_REG(MSC->FLASHCMD, 0x00005DA2);

        while (READ_BIT(MSC->FLASHSR, MSC_FLASHSR_BUSY_MSK) == 1)
		{
		}

		while (READ_BIT(MSC->FLASHSR, MSC_FLASHSR_PROG_MSK) == 0)
		{
		}

        CLEAR_BIT(MSC->FLASHCR, MSC_FLASHCR_FLASHREQ_MSK);
        CLEAR_BIT(MSC->FLASHCR, MSC_FLASHCR_IAPEN_MSK);
        FLASH_REG_LOCK();

        /* Read data from the flash address */
        memcpy((void *)data_buf, (void *)TEST_ADDR, 4);

        if (data_buf[0] != 0x12345678)
            break;

        g_iap_flag = TRUE;
        break;
    }
    while (0);
}

/**
  * @brief:  Main program.
  * @param:  None
  * @retval: None
  */
int main(void)
{
    /* Configure system clock */
    md_cmu_pll_config(MD_CMU_PLL_INPUT_HOSC8M, MD_CMU_PLL_OUTPUT_72M);
    md_cmu_clock_config(MD_CMU_CLOCK_PLL, 72000000);

    /* Enable ALL peripheral */
    MD_SYSCFG_UNLOCK();
    md_cmu_enable_perh_all();
    MD_SYSCFG_LOCK();

    /* Initialize pin */
    init_gpio();
    md_gpio_set_pin_high(LED_PORT, LED_PIN);

    __enable_irq();

    /* Initialize timer_AD16C4T */
    init_timer();

    while (1)
    {
        iap_exe();

        if (g_iap_flag == TRUE)                          /* If iap is successful */
            md_gpio_set_pin_low(LED_PORT, LED_PIN);    /* Turn on the led */
        else
            md_gpio_set_pin_high(LED_PORT, LED_PIN);
    }
}

/**
  * @}
  */
/**
  * @}
  */

/************* (C) COPYRIGHT Eastsoft Microelectronics *****END OF FILE****/
