/**********************************************************************************
 *
 * @file    main.c
 * @brief   main C file
 *
 * @date    28 Apr. 2023
 * @author  AE Team
 * @note
 *          Change Logs:
 *          Date            Author          Notes
 *          28 Apr. 2023    Lisq            the first version
 *
 * Copyright (C) Shanghai Eastsoft Microelectronics Co. Ltd. All rights reserved.
 *
 * SPDX-License-Identifier: Apache-2.0
 *
 * Licensed under the Apache License, Version 2.0 (the License); you may
 * not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an AS IS BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 *
 **********************************************************************************
 */
/* Includes ------------------------------------------------------------------ */
#include <stdio.h>
#include "main.h"

/* Public Variables ---------------------------------------------------------- */
/* Private Macros ------------------------------------------------------------ */
/* Private Variables --------------------------------------------------------- */
static uint32_t s_crc_buf[8] = {1U, 2U, 3U, 4U, 5U, 6U, 7U, 8U};
static unsigned int s_crc_result = 0x0U;

/* Private Constants --------------------------------------------------------- */
/* Private function prototypes ----------------------------------------------- */
void gpio_init(void);
void uart_init(void);

/* Private Function ---------------------------------------------------------- */

/**
  * @brief:  Main program.
  * @param:  None
  * @retval: None
  */
int main(void)
{
    /* Configure system clock */
    md_cmu_clock_config(MD_CMU_CLOCK_HRC48M, 48000000);
    /* Initialize SysTick Interrupt */
    md_init_1ms_tick();

    /* Enable ALL peripheral */
    MD_SYSCFG_UNLOCK();
    md_cmu_enable_perh_all();
    MD_SYSCFG_LOCK();

    __enable_irq();

    /* Init GPIO */
    gpio_init();

    /* Init UART */
    uart_init();

    /* Init CRC32 */
    init_crc32_module();

    /* Calculate CRC32 value for array crc_buf */
    s_crc_result = get_crc32_calculation_result(s_crc_buf, (sizeof(s_crc_buf) / 4));

    /* Delay 1S and output value */
    md_delay_1ms(1000);
    printf("CRC32 calculation result for 0x01 ~ 0x08 is: 0x%08X\r\n", s_crc_result);

    while (1)
    {
        md_gpio_toggle_pin_output(LED_PORT, LED_PIN);
        md_delay_1ms(500);
    }
}

/**
  * @brief  Initialize gpio.
  * @retval None
  */
void gpio_init(void)
{
    md_gpio_init_t gpio_init;

    md_gpio_init_struct(&gpio_init);
    /* Initialize led pin */
    gpio_init.mode  = MD_GPIO_MODE_OUTPUT;
    gpio_init.odos  = MD_GPIO_PUSH_PULL;
    gpio_init.pupd  = MD_GPIO_PUSH_UP;
    gpio_init.odrv  = MD_GPIO_OUT_DRIVE_NORMAL;
    gpio_init.flt   = MD_GPIO_FILTER_DISABLE;
    gpio_init.type  = MD_GPIO_TYPE_CMOS;
    gpio_init.func  = MD_GPIO_FUNC_1;
    md_gpio_init(LED_PORT, LED_PIN, &gpio_init);
    md_gpio_toggle_pin_output(LED_PORT, LED_PIN);

    md_gpio_init_struct(&gpio_init);
    /* Initialize tx pin */
    gpio_init.mode  = MD_GPIO_MODE_OUTPUT;
    gpio_init.odos  = MD_GPIO_PUSH_PULL;
    gpio_init.pupd  = MD_GPIO_PUSH_UP;
    gpio_init.odrv  = MD_GPIO_OUT_DRIVE_NORMAL;
    gpio_init.flt   = MD_GPIO_FILTER_DISABLE;
    gpio_init.type  = MD_GPIO_TYPE_CMOS;
    gpio_init.func  = MD_GPIO_FUNC_2;
    md_gpio_init(UART_TX_PORT, UART_TX_PIN, &gpio_init);

    return;
}

/**
* @brief  Initialize uart
  * @retval None
  */
void uart_init(void)
{
    md_uart_init_t uart_init;
    md_uart_init_struct(&uart_init);

    /* Initialize UART */
    uart_init.baud        = 115200;
    uart_init.word_length = MD_UART_WORD_LENGTH_8B;
    uart_init.stop_bits   = MD_UART_STOP_BITS_1;
    uart_init.parity      = MD_UART_PARITY_NONE;
    uart_init.fctl        = MD_UART_FLOW_CTL_DISABLE;
    uart_init.mode        = MD_UART_MODE;
    md_uart_init(UARTX, &uart_init);
}

int _write(int file, const void *ptr, int len)
{
    uint16_t cnt;
    (void)file;
    uint8_t *ptr_char = (uint8_t *)ptr;

    int i = 0;

    for (i = 0 ; i < len ; i++)
    {
        cnt = 4000;

        while ((READ_BIT(UARTX->STAT, UART_STAT_TFEMPTY_MSK)) != UART_STAT_TFEMPTY_MSK
                && (--cnt))
        {
        }

        md_uart_set_send_data8(UARTX, *ptr_char++);
    }

    return len;
}


/**
  * @}
  */
/**
  * @}
  */

/************* (C) COPYRIGHT Eastsoft Microelectronics *****END OF FILE****/
