/**********************************************************************************
 *
 * @file    main.c
 * @brief   main C file
 *
 * @date    27 Dec. 2022
 * @author  AE Team
 * @note
 *          Change Logs:
 *          Date            Author          Notes
 *          27 Dec. 2022    Lisq            the first version
 *
 * Copyright (C) Shanghai Eastsoft Microelectronics Co. Ltd. All rights reserved.
 *
 * SPDX-License-Identifier: Apache-2.0
 *
 * Licensed under the Apache License, Version 2.0 (the License); you may
 * not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an AS IS BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 *
 **********************************************************************************
 */
/* Includes ------------------------------------------------------------------ */
#include "main.h"

/* Private Macros ------------------------------------------------------------ */
#define TEST_ADDR   0x3000
#define TEST_LEN    0x20

/* Private Variables --------------------------------------------------------- */
/* Public Variables ---------------------------------------------------------- */
/* Private Constants --------------------------------------------------------- */
/* Private function prototypes ----------------------------------------------- */
/* Private Function ---------------------------------------------------------- */

/**
  * @brief  Initialize TOGGLE IO.
  * @param  None
  * @retval None
  */
static void init_gpio(void)
{
    md_gpio_init_t gpio_init;
    /* Initialize TOGGLE IO as output */
    md_gpio_init_struct(&gpio_init);      /* initialize the gpio_init */
    gpio_init.mode = MD_GPIO_MODE_OUTPUT; /* output */
    gpio_init.odos = MD_GPIO_PUSH_PULL;   /* push pull */
    gpio_init.pupd = MD_GPIO_FLOATING;    /* no pull up and pull down */
    gpio_init.type = MD_GPIO_TYPE_CMOS;
    gpio_init.func = MD_GPIO_FUNC_1;      /* GPIO function 1 */
    md_gpio_init(LED_PORT, LED_PIN, &gpio_init);
}

/**
  * @brief:  Main program.
  * @param:  None
  * @retval: None
  */
int main(void)
{
    uint32_t data_buf[TEST_LEN];
    uint32_t iap_flag;

    /* Configure system clock */
    md_cmu_pll_config(MD_CMU_PLL_INPUT_HOSC8M, MD_CMU_PLL_OUTPUT_72M);
    md_cmu_clock_config(MD_CMU_CLOCK_PLL, 72000000);

    /* Enable ALL peripheral */
    MD_SYSCFG_UNLOCK();
    md_cmu_enable_perh_all();
    MD_SYSCFG_LOCK();

    /* Initialize pin */
    init_gpio();
    md_gpio_set_pin_high(LED_PORT, LED_PIN);

    __disable_irq();

    do
    {
        iap_flag = FALSE;

        FLASH_REG_UNLOCK();

        if (IAPROM_PAGE_ERASE(TEST_ADDR, ~TEST_ADDR, 0) == RESET)
            break;

        if (IAPROM_WORD_PROGRAM(TEST_ADDR, ~TEST_ADDR, 0x12345678) == RESET)
            break;

        FLASH_REG_LOCK();

        /* Read data from the flash address */
        memcpy((void *)data_buf, (void *)TEST_ADDR, 4);

        if (data_buf[0] != 0x12345678)
            break;

        /* Multiple words write testing */
        memset((void *)data_buf, 0x55, TEST_LEN * 4);

        FLASH_REG_UNLOCK();

        if (IAPROM_WORDS_PROGRAM(TEST_ADDR, ~TEST_ADDR, data_buf, TEST_LEN * 4, AUTO_ERASE_TRUE) == RESET)
            break;

        FLASH_REG_LOCK();

        if (memcmp((void *)TEST_ADDR, (void *)data_buf, TEST_LEN * 4) != 0)
            break;

        iap_flag = TRUE;
        break;
    }
    while (0);

    __enable_irq();

    while (1)
    {
        if (iap_flag == TRUE)                          /* If iap is successful */
            md_gpio_set_pin_low(LED_PORT, LED_PIN);    /* Turn on the led */
    }
}

/**
  * @}
  */
/**
  * @}
  */

/************* (C) COPYRIGHT Eastsoft Microelectronics *****END OF FILE****/
