/**********************************************************************************
 *
 * @file    main.c
 * @brief   main C file
 *
 * @date    16 June 2023
 * @author  AE Team
 * @note
 *          Change Logs:
 *          Date            Author          Notes
 *          16 June 2023    Lisq            the first version
 *
 * Copyright (C) Shanghai Eastsoft Microelectronics Co. Ltd. All rights reserved.
 *
 * SPDX-License-Identifier: Apache-2.0
 *
 * Licensed under the Apache License, Version 2.0 (the License); you may
 * not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an AS IS BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 *
 **********************************************************************************
 */
/* Includes ------------------------------------------------------------------ */
#include "main.h"

/* Private Macros ------------------------------------------------------------ */
/* Private Variables --------------------------------------------------------- */
uint32_t g_D_10C;
uint32_t g_T_M40;
uint32_t g_TVAL_real;  /* actual conversion value of temperature sense */
uint32_t g_TVAL_5v;    /* conversion value corresponding to the 5V reference voltage */
uint32_t g_Dx;
uint32_t g_Cx;
uint32_t g_Tcal;
uint8_t  g_Index1, g_Index2;

/* Public Variables ---------------------------------------------------------- */
md_adc_init_t g_adc_init;
uint32_t g_adc_result;
uint8_t g_temp = 0U;

/* Private Constants --------------------------------------------------------- */
const int8_t TRIMTAB[12][18] = {{11, 9, 7, 6, 4, 2, 1, 0, -1, -2, -4, -7, -10, -13, -16, -20, -24, -28},
    {8, 6, 5, 4, 3, 2, 1, 0, -1, -2, -3, -5, -7, -9, -12, -15, -18, -21},
    {3, 3, 2, 2, 2, 1, 1, 0, -1, -1, -2, -2, -3, -4, -5, -5, -6, -7},
    {0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0},
    {-6, -5, -4, -3, -2, -2, -1, 0, 1, 1, 2, 2, 2, 3, 3, 3, 4, 4},
    {-14, -10, -7, -5, -4, -2, -1, 0, 1, 2, 3, 4, 5, 6, 8, 9, 11, 12},
    {-18, -14, -11, -8, -5, -3, -1, 0, 1, 2, 4, 6, 7, 9, 10, 12, 14, 16},
    {-24, -19, -15, -11, -8, -5, -2, 0, 2, 3, 5, 7, 9, 11, 13, 16, 19, 21},
    {-30, -24, -19, -14, -10, -6, -3, 0, 2, 4, 7, 10, 12, 14, 17, 19, 21, 24},
    {-36, -29, -23, -17, -12, -7, -3, 0, 3, 6, 9, 12, 15, 18, 21, 24, 26, 29},
    {-41, -34, -27, -20, -14, -8, -4, 0, 4, 7, 10, 14, 17, 20, 24, 27, 30, 34},
    {-48, -39, -31, -23, -16, -10, -5, 0, 4, 8, 12, 16, 20, 24, 28, 32, 36, 40}
};

/* Private function prototypes ----------------------------------------------- */
/* Private Function ---------------------------------------------------------- */
/** @addtogroup Projects_Examples_MD
  * @{
  */

/** @addtogroup Examples
  * @{
  */

/** @addtogroup Projects_Examples_MD
  * @{
  */

/** @addtogroup Examples
  * @{
  */
/**
  * @brief  HRC4M calibration init.
  * @param  None
  * @retval None
  */
void hrc4m_tcal_init(void)
{
    g_D_10C = (ADCTEMPC1 - ADCTEMPC4) * 10 / 55;

    if ((ADCTEMPC1 - ADCTEMPC4) % 55 > 28)
        g_D_10C = g_D_10C + 1;

    g_T_M40  = ADCTEMPC4 - 7 * g_D_10C;
    g_Dx     = (HRC4MFT / 4 - 1000);
    g_Index1 = (ADCFREQC1 - ADCFREQC4 - 32) / 8 + 3;
}

/**
  * @brief  HRC4M calibration function.
  * @param  None
  * @retval None
  */
void hrc4m_tcal_append(void)
{
    g_TVAL_real    = g_adc_result;
    g_TVAL_5v      = g_TVAL_real * ADCREFP / 5000;
    g_Index2       = (g_TVAL_5v - g_T_M40) / g_D_10C;

    if ((g_TVAL_5v - g_T_M40) % g_D_10C > (g_D_10C / 2))
        g_Index2 = g_Index2 + 1;

    g_Cx           = TRIMTAB[g_Index1][g_Index2];
    g_Tcal         = g_Cx + g_Dx;
    SYSCFG->PROT = 0x55AA6996;
    SYSCFG1_PROT = 0x5A962814;
    SYSCFG1_PROT = 0xE7CB69A5;
    HRC4MCALN    = 0x55AA0000 + (HRC4MCALO) - g_Tcal;
    SYSCFG1_PROT = 0x0;
    SYSCFG->PROT = 0x0;
}

/**
  * @brief  Configure the ADC.
  * @param  None
  * @retval None
  */
void adc_base_config(void)
{
    md_adc_struct_init(&g_adc_init);

    /* Set ADC Base Configuration */
    g_adc_init.align    = MD_ADC_DATAALIGN_RIGHT;     /* Specifies ADC data alignment */
    g_adc_init.data_bit = MD_ADC_CONV_BIT_12;
    g_adc_init.div      = MD_ADC_POSDIV_16;           /* ADCCLK divider */
    g_adc_init.n_ref    = MD_ADC_NEG_REF_VSS;
    g_adc_init.p_ref    = MD_ADC_POS_REF_VDD;         /* The positive reference voltage*/
    md_adc_init(ADC, &g_adc_init);

    MD_ADC_SPEED_HIGH_ENABLE(ADC);
    /* Enable temperature sense */
    MD_ADC_TEMP_SENSE_ENABLE(ADC);
    /* Disable ADC sequencers scan mode */
    md_adc_scan_mode_disable_scanen(ADC);
    /* Set ADC group REG sequencer length and scan direction, ADC conversion 1 channel */
    md_adc_set_insert_channel_length_isl(ADC, MD_ADC_ICH_NR_1);
    /* Set ADC continuous conversion mode on ADC group REG.*/
    md_adc_continuous_conversion_disable_cm(ADC);
    /* Set ADC group REG sequence: channel on the selected scan sequence rank. */
    md_adc_set_insert_1st_conv_is1(ADC, MD_ADC_CHANNEL_17);
    md_adc_set_smpt1_cht(ADC, MD_ADC_SAMPLETIME_4, MD_ADC_CHANNEL_17);

    /* ADC insert trig polarity */
    md_adc_set_insert_conv_extern_polarity(ADC, MD_ADC_ETS_RISE);

    /* Enable interruption ADC group insert end of sequence conversions. */
    md_adc_inserted_channel_interrupt_enable_icheie(ADC);
}

/**
  * @brief  Initialize TIMER.
  * @param  None
  * @retval None
  */
void init_timer(void)
{
    md_timer_base_init_t init;
    md_timer_base_struct_init(&init);
    init.prescaler = 480 - 1;
    init.period = 50000;  /* 500ms */
    md_timer_base_set_config(AD16C4T, &init);
}

/**
  * @brief  Initialize PIS.
  * @param  None
  * @retval None
  */
void init_pis(void)
{
    md_pis_init_t pis_init;

    /* Initialize PIS */
    memset(&pis_init, 0, sizeof(md_pis_init_t));
    pis_init.p_src     = MD_PIS_TIMER0_UPDATA;
    pis_init.p_clk     = MD_PIS_CLK_PCLK;
    pis_init.p_edge    = MD_PIS_EDGE_NONE;
    pis_init.p_output  = MD_PIS_OUT_PULSE;
    pis_init.c_trig    = MD_PIS_CH5_ADC0_INSERT;
    pis_init.c_clk     = MD_PIS_CLK_PCLK;
    md_pis_init(&pis_init);
}

/**
  * @brief  Initializate pin of HSCO.
  * @retval None
  */
void hsco_pin_init(void)
{
    md_gpio_init_t gpio_init;

    /* initialize the HSCO */
    md_gpio_init_struct(&gpio_init);
    gpio_init.mode  = MD_GPIO_MODE_OUTPUT;
    gpio_init.odos  = MD_GPIO_PUSH_PULL;
    gpio_init.pupd  = MD_GPIO_FLOATING;
    gpio_init.odrv  = MD_GPIO_OUT_DRIVE_NORMAL;
    gpio_init.flt   = MD_GPIO_FILTER_DISABLE;
    gpio_init.type  = MD_GPIO_TYPE_CMOS;
    gpio_init.func  = MD_GPIO_FUNC_4;
    md_gpio_init(HSCO_PORT, HSCO_PIN, &gpio_init);

    return;
}

static void gpio_init(void)
{
    md_gpio_init_t gpio_init;

    /* Initialize LED pin */
    md_gpio_init_struct(&gpio_init);      /* initialize the gpio_init */
    gpio_init.mode  = MD_GPIO_MODE_OUTPUT;
    gpio_init.odos  = MD_GPIO_PUSH_PULL;
    gpio_init.pupd  = MD_GPIO_FLOATING;
    gpio_init.odrv  = MD_GPIO_OUT_DRIVE_NORMAL;
    gpio_init.flt   = MD_GPIO_FILTER_DISABLE;
    gpio_init.type  = MD_GPIO_TYPE_CMOS;
    gpio_init.func  = MD_GPIO_FUNC_1;
    md_gpio_init(LED_PORT, LED_PIN, &gpio_init);

    return;
}

/**
  * @brief:  Main program.
  * @param:  None
  * @retval: None
  */
int main(void)
{
    /* Configure system clock */
    md_cmu_pll_config(MD_CMU_PLL_INPUT_HRC4M, MD_CMU_PLL_OUTPUT_48M);
    md_cmu_clock_config(MD_CMU_CLOCK_PLL, 48000000);

    MD_SYSCFG_UNLOCK();
    md_cmu_enable_perh_all(); /* Enable ALL peripheral */
    md_cmu_set_hsco_div(7);   /* 7: HSC / 128 */
    md_cmu_set_hsco_type(4);  /* 4: SYSCLK */
    md_cmu_enable_hsco();     /* Enable HSC output */
    MD_SYSCFG_LOCK();

    gpio_init();

    hrc4m_tcal_init();

    hsco_pin_init();

    /* Enable the selected ADC instance */
    md_adc_converter_enable_adcen(ADC);
    /* Disable the selected ADC instance */
    md_adc_converter_disable_adcen(ADC);

    /* Initialize PIS */
    init_pis();

    /* Initialize TIMER */
    init_timer();

    __enable_irq();
    md_mcu_irq_config(ADC_IRQn, 2, ENABLE);

    /* ADC Base Feature Configuration */
    adc_base_config();

    /* Enable the selected ADC instance.*/
    md_adc_converter_enable_adcen(ADC);

    /* Start TIMER */
    md_timer_enable_counter_cnten(AD16C4T);

    while (1)
    {
        /* complete temperature transition */
        if (g_temp == 1)
        {
            g_temp = 0;

            md_gpio_toggle_pin_output(LED_PORT, LED_PIN);

            g_adc_result = md_adc_get_insert_channel1_val(ADC);

            hrc4m_tcal_append();
        }
    }
}

/**
  * @}
  */
/**
  * @}
  */

/************* (C) COPYRIGHT Eastsoft Microelectronics *****END OF FILE****/
