/**********************************************************************************
 *
 * @file    main.c
 * @brief   main C file
 *
 * @date    9 Jan. 2023
 * @author  AE Team
 * @note
 *          Change Logs:
 *          Date            Author          Notes
 *          9 Jan. 2023     Lisq            the first version
 *
 * Copyright (C) Shanghai Eastsoft Microelectronics Co. Ltd. All rights reserved.
 *
 * SPDX-License-Identifier: Apache-2.0
 *
 * Licensed under the Apache License, Version 2.0 (the License); you may
 * not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an AS IS BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 *
 **********************************************************************************
 */
/* Includes ------------------------------------------------------------------ */
#include "main.h"

/* Private Macros ------------------------------------------------------------ */
/* Private Variables --------------------------------------------------------- */
/* Public Variables ---------------------------------------------------------- */
md_adc_init_t g_adc_init;
uint16_t g_adc_result_normal_temp[16] = {0U};
uint16_t g_adc_result_normal[16] = {0U};
uint16_t g_adc_result_insert[16] = {0U};
uint8_t g_dma_flag = 0U;

/* Private Constants --------------------------------------------------------- */
/* Private function prototypes ----------------------------------------------- */
/* Private Function ---------------------------------------------------------- */
/** @addtogroup Projects_Examples_MD
  * @{
  */

/** @addtogroup Examples
  * @{
  */

/** @addtogroup Projects_Examples_MD
  * @{
  */

/** @addtogroup Examples
  * @{
  */
/**
  * @brief  Configure the ADC Pins.
  * @param  None
  * @retval None
  */
void gpio_pin_config(void)
{
    md_gpio_init_t gpio_init;
    md_gpio_init_struct(&gpio_init);

    /* Initialize adc gpio pin */
    gpio_init.mode  = MD_GPIO_MODE_CLOSE;
    gpio_init.pupd  = MD_GPIO_PUSH_UP;
    gpio_init.odos  = MD_GPIO_PUSH_PULL;
    gpio_init.odrv  = MD_GPIO_OUT_DRIVE_NORMAL;
    gpio_init.flt   = MD_GPIO_FILTER_DISABLE;
    gpio_init.type  = MD_GPIO_TYPE_CMOS;
    gpio_init.func  = MD_GPIO_FUNC_0;
    md_gpio_init(ADC_CHANNEL6_GPIO_PORT, ADC_CHANNEL6_PIN, &gpio_init);
    md_gpio_init(ADC_CHANNEL7_GPIO_PORT, ADC_CHANNEL7_PIN, &gpio_init);
}

/**
  * @brief  Configure the ADC.
  * @param  None
  * @retval None
  */
void adc_base_config(void)
{
    md_adc_struct_init(&g_adc_init);

    /* Set ADC Base Configuration:  */
    g_adc_init.align    = MD_ADC_DATAALIGN_RIGHT;     /* Specifies ADC data alignment */
    g_adc_init.data_bit = MD_ADC_CONV_BIT_12;
    g_adc_init.div      = MD_ADC_POSDIV_16;           /* ADCCLK divider */
    g_adc_init.n_ref    = MD_ADC_NEG_REF_VSS;
    g_adc_init.p_ref    = MD_ADC_POS_REF_VDD;         /* The positive reference voltage*/
    md_adc_init(ADC, &g_adc_init);

    MD_ADC_SPEED_HIGH_ENABLE(ADC);
    /* Set ADC group injected conversion trigger:independent or from ADC group REG */
    md_adc_auto_inserted_conversion_enable_iauto(ADC);
    /* Trigger the NCHE FALG mode */
    md_adc_set_eoc_selection_nchesel(ADC, MD_ADC_NCHESEL_MODE_ALL);
    /*Enable ADC sequencers scan mode*/
    md_adc_scan_mode_enable_scanen(ADC);
    /* Set ADC group REG sequencer length and scan direction, ADC conversion 1 channel */
    md_adc_set_normal_channel_length_nsl(ADC, MD_ADC_NCH_NR_1);
    /* Set ADC group INJ sequencer length and scan direction,ADC conversion 1 channel */
    md_adc_set_insert_channel_length_isl(ADC, MD_ADC_ICH_NR_1);
    /* Set ADC continuous conversion mode on ADC group REG.*/
    md_adc_continuous_conversion_enable_cm(ADC);
    /* Enable DMA access */
    md_adc_set_overflow_detection_state_ovrdis(ADC, ENABLE);
    /* Set ADC group sequence: channel on the selected scan sequence rank */
    md_adc_set_normal_1st_conv_ns1(ADC, MD_ADC_CHANNEL_6);
    md_adc_set_smpt1_cht(ADC, MD_ADC_SAMPLETIME_4, MD_ADC_CHANNEL_6);
    md_adc_set_insert_1st_conv_is1(ADC, MD_ADC_CHANNEL_7);
    md_adc_set_smpt1_cht(ADC, MD_ADC_SAMPLETIME_4, MD_ADC_CHANNEL_7);

    /* Enable interruption ADC group insert end of sequence conversions */
    md_adc_inserted_channel_interrupt_enable_icheie(ADC);
}

/**
  * @brief  Configure the DMA.
  * @param  None
  * @retval None
  */
void dma_init(void)
{
    md_dma_config_t dma_config;

    md_dma_enable_it_tc(MD_DMA_CH_0);

    memset(&dma_config, 0x0, sizeof(md_dma_config_t));
    dma_config.src            = (void *)(&ADC->NCHDR);
    dma_config.dst            = (void *)g_adc_result_normal_temp;
    dma_config.size           = 16;
    dma_config.src_data_width = MD_DMA_DATA_SIZE_HALFWORD;
    dma_config.dst_data_width = MD_DMA_DATA_SIZE_HALFWORD;
    dma_config.src_inc        = DISABLE;
    dma_config.dst_inc        = ENABLE;
    dma_config.R_power        = MD_DMA_R_POWER_1;
    dma_config.priority       = MD_DMA_LOW_PRIORITY;
    dma_config.mem_to_mem     = DISABLE;
    dma_config.circle_mode    = ENABLE;
    dma_config.msel           = MD_DMA_MSEL_ADC;
    dma_config.msigsel        = MD_DMA_MSIGSEL_ADC;
    md_dma_init(MD_DMA_CH_0, &dma_config);

    md_dma_enable_channel(MD_DMA_CH_0);
}

/**
  * @brief:  Main program.
  * @param:  None
  * @retval: None
  */
int main(void)
{
    uint8_t i;

    /* Configure system clock */
    md_cmu_clock_config(MD_CMU_CLOCK_HRC48M, 48000000);
    md_init_1ms_tick();

    /* Enable ALL peripheral */
    MD_SYSCFG_UNLOCK();
    md_cmu_enable_perh_all();
    MD_SYSCFG_LOCK();

    /* Enable the selected ADC instance */
    md_adc_converter_enable_adcen(ADC);
    /* Disable the selected ADC instance */
    md_adc_converter_disable_adcen(ADC);

    dma_init();

    /* Pin Configuration */
    gpio_pin_config();
    /* ADC Base Feature Configuration */
    adc_base_config();

    md_mcu_irq_config(ADC_IRQn, 0, ENABLE);
    md_mcu_irq_config(DMA_IRQn, 0, ENABLE);
    __enable_irq();

    /* Enable the selected ADC instance.*/
    md_adc_converter_enable_adcen(ADC);

    /* Enable ADC group REG conversion trigger source internal (SW start)*/
    md_adc_set_normal_channel_conv_start_nchtrg(ADC);

    while (1)
    {
        if (g_dma_flag)
        {
            g_dma_flag = 0;

            /* calculate value */
            for (i = 0; i < 16; i++)
                g_adc_result_normal[i] = g_adc_result_normal_temp[i] * VDD / 4095;
        }
    }
}

/**
  * @}
  */
/**
  * @}
  */

/************* (C) COPYRIGHT Eastsoft Microelectronics *****END OF FILE****/
