/**********************************************************************************
 *
 * @file    main.c
 * @brief   Main file for DEMO
 *
 * @date    13 Mar. 2023
 * @author  AE Team
 * @note
 *          Change Logs:
 *          Date            Author          Notes
 *          13 Mar. 2023    Lisq            the first version
 *
 * Copyright (C) Shanghai Eastsoft Microelectronics Co. Ltd. All rights reserved.
 *
 * SPDX-License-Identifier: Apache-2.0
 *
 * Licensed under the Apache License, Version 2.0 (the License); you may
 * not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an AS IS BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 *
 **********************************************************************************
 */

/* Includes ------------------------------------------------------------------ */

#include <string.h>
#include "main.h"

/* Private Macros ------------------------------------------------------------ */

/* Private Variables --------------------------------------------------------- */

/* Public Variables ---------------------------------------------------------- */
ald_uart_handle_t g_uart2_init;
ald_uart_handle_t g_uart1_init;

char g_tx2_buf[256] = "CUART2 : ";
char g_tx1_buf[256] = "CUART1 : ";
uint8_t g_tx2_len ;
uint8_t g_tx1_len ;

uint8_t g_rx2_buf[256];
uint32_t g_rx2_len = sizeof(g_rx2_buf);
uint8_t g_rx1_buf[256];
uint32_t g_rx1_len = sizeof(g_rx1_buf);

/* Private Constants --------------------------------------------------------- */

/* Private function prototypes ----------------------------------------------- */

/* Private Function ---------------------------------------------------------- */

/**
  * @brief  Initializate pin of uart module.
  * @retval None
  */
void init_uart_pin(void)
{
    ald_gpio_init_t gpio_init;
    memset(&gpio_init, 0x00, sizeof(ald_gpio_init_t));

    /* Initialize CUART2 tx pin: PB10 */
    gpio_init.mode = ALD_GPIO_MODE_OUTPUT;
    gpio_init.od   = ALD_GPIO_PUSH_PULL;
    gpio_init.pupd = ALD_GPIO_PUSH_UP;
    gpio_init.odrv = ALD_GPIO_OUT_DRIVE_NORMAL;
    gpio_init.flt  = ALD_GPIO_FILTER_DISABLE;
    gpio_init.type = ALD_GPIO_TYPE_CMOS;
    gpio_init.func = ALD_GPIO_FUNC_3;
    ald_gpio_init(CUART2_TX_PORT, CUART2_TX_PIN, &gpio_init);

    /* Initialize CUART2 rx pin: PB11 */
    gpio_init.mode = ALD_GPIO_MODE_INPUT;
    gpio_init.od   = ALD_GPIO_PUSH_PULL;
    gpio_init.pupd = ALD_GPIO_PUSH_UP;
    gpio_init.odrv = ALD_GPIO_OUT_DRIVE_NORMAL;
    gpio_init.flt  = ALD_GPIO_FILTER_DISABLE;
    gpio_init.type = ALD_GPIO_TYPE_CMOS;
    gpio_init.func = ALD_GPIO_FUNC_3;
    ald_gpio_init(CUART2_RX_PORT, CUART2_RX_PIN, &gpio_init);

    /* Initialize CUART1 tx pin: PA2 */
    gpio_init.mode = ALD_GPIO_MODE_OUTPUT;
    gpio_init.od   = ALD_GPIO_PUSH_PULL;
    gpio_init.pupd = ALD_GPIO_PUSH_UP;
    gpio_init.odrv = ALD_GPIO_OUT_DRIVE_NORMAL;
    gpio_init.flt  = ALD_GPIO_FILTER_DISABLE;
    gpio_init.type = ALD_GPIO_TYPE_CMOS;
    gpio_init.func = ALD_GPIO_FUNC_2;
    ald_gpio_init(CUART1_TX_PORT, CUART1_TX_PIN, &gpio_init);

    /* Initialize CUART1 rx pin: PA3 */
    gpio_init.mode = ALD_GPIO_MODE_INPUT;
    gpio_init.od   = ALD_GPIO_PUSH_PULL;
    gpio_init.pupd = ALD_GPIO_PUSH_UP;
    gpio_init.odrv = ALD_GPIO_OUT_DRIVE_NORMAL;
    gpio_init.flt  = ALD_GPIO_FILTER_DISABLE;
    gpio_init.type = ALD_GPIO_TYPE_CMOS;
    gpio_init.func = ALD_GPIO_FUNC_2;
    ald_gpio_init(CUART1_RX_PORT, CUART1_RX_PIN, &gpio_init);

    return;
}

/**
  * @brief  CUART2 receive a message complete.
  * @param  arg: Pointer to uart_handle_t structure.
  * @retval None.
  */
void cuart2_recv_complete(ald_uart_handle_t *arg)
{
    static uint8_t s_i = 0;

    if ((s_i > 1) && (*(g_uart2_init.rx_buf - 2) == 0x0D) && (*(g_uart2_init.rx_buf  - 1) == 0x0A))
    {
        memcpy(g_tx2_buf + g_tx2_len, g_rx2_buf, s_i);
        ald_uart_send_by_it(&g_uart2_init, (uint8_t *)g_tx2_buf, g_tx2_len + s_i);
        ald_uart_recv_by_it(&g_uart2_init, g_rx2_buf, 1);
        s_i = 0;
    }
    else
    {
        if (s_i >= g_rx2_len)
            s_i = 0;

        ald_uart_recv_by_it(&g_uart2_init, g_rx2_buf + s_i, 1);
        s_i++;
    }

    return;
}

/**
  * @brief  CUART1 receive a message complete.
  * @param  arg: Pointer to uart_handle_t structure.
  * @retval None.
  */
void cuart1_recv_complete(ald_uart_handle_t *arg)
{
    static uint8_t s_i = 0;

    if ((s_i > 1) && (*(g_uart1_init.rx_buf - 2) == 0x0D) && (*(g_uart1_init.rx_buf - 1) == 0x0A))
    {
        memcpy(g_tx1_buf + g_tx1_len, g_rx1_buf, s_i);
        ald_uart_send_by_it(&g_uart1_init, (uint8_t *)g_tx1_buf, g_tx1_len + s_i);
        s_i = 0;
        ald_uart_recv_by_it(&g_uart1_init, g_rx1_buf, 1);

    }
    else
    {
        if (s_i >= g_rx1_len)
            s_i = 0;

        ald_uart_recv_by_it(&g_uart1_init, g_rx1_buf + s_i, 1);
        s_i++;
    }

    return;
}

/**
  * @brief  Initializate uart module.
  * @retval None
  */
void init_uart(void)
{
    ald_uart_rs485_config_t rs485_config2, rs485_config1;

    memset(&g_uart2_init, 0x00, sizeof(ald_uart_handle_t));
    /* Initialize cuart2 */
    g_uart2_init.perh             = CUART2;
    g_uart2_init.init.baud        = 115200;
    g_uart2_init.init.word_length = ALD_UART_WORD_LENGTH_8B;
    g_uart2_init.init.stop_bits   = ALD_UART_STOP_BITS_1;
    g_uart2_init.init.parity      = ALD_UART_PARITY_NONE;
    g_uart2_init.init.mode        = ALD_UART_MODE_RS485;
    g_uart2_init.init.fctl        = ALD_UART_HW_FLOW_CTL_DISABLE;
    g_uart2_init.rx_cplt_cbk      = cuart2_recv_complete;
    ald_uart_init(&g_uart2_init);

    rs485_config2.normal = DISABLE;
    rs485_config2.dir    = DISABLE;
    rs485_config2.invert = DISABLE;
    rs485_config2.addr   = 0xF;
    ald_uart_rs485_config(&g_uart2_init, &rs485_config2);

    memset(&g_uart1_init, 0x00, sizeof(ald_uart_handle_t));
    /* Initialize uart1 */
    g_uart1_init.perh             = CUART1;
    g_uart1_init.init.baud        = 115200;
    g_uart1_init.init.word_length = ALD_UART_WORD_LENGTH_8B;
    g_uart1_init.init.stop_bits   = ALD_UART_STOP_BITS_1;
    g_uart1_init.init.parity      = ALD_UART_PARITY_NONE;
    g_uart1_init.init.mode        = ALD_UART_MODE_RS485;
    g_uart1_init.init.fctl        = ALD_UART_HW_FLOW_CTL_DISABLE;
    g_uart1_init.rx_cplt_cbk      = cuart1_recv_complete;
    ald_uart_init(&g_uart1_init);

    rs485_config1.normal = DISABLE;
    rs485_config1.dir    = DISABLE;
    rs485_config1.invert = DISABLE;
    rs485_config1.addr   = 0x1;
    ald_uart_rs485_config(&g_uart1_init, &rs485_config1);
}

/**
  * @brief  Test main function
  * @retval Status.
  */
int main(void)
{
    /* Initialize ALD */
    ald_cmu_init();

    /* Configure system clock */
    ald_cmu_pll_config(ALD_CMU_PLL_INPUT_HOSC8M, ALD_CMU_PLL_OUTPUT_72M);
    ald_cmu_clock_config(ALD_CMU_CLOCK_PLL, 72000000);

    /* Enable all peripherals */
    ald_cmu_perh_clock_config(ALD_CMU_PERH_ALL, ENABLE);

    ald_mcu_irq_config(CUART2_IRQn, 4, ENABLE);
    ald_mcu_irq_config(CUART1_IRQn, 4, ENABLE);
    __enable_irq();

    /* Initialize pin */
    init_uart_pin();
    /* Initialize uart */
    init_uart();

    ald_uart_recv_by_it(&g_uart2_init, g_rx2_buf, 1);
    ald_uart_recv_by_it(&g_uart1_init, g_rx1_buf, 1);

    g_tx2_len = strlen(g_tx2_buf);
    g_tx1_len = strlen(g_tx1_buf);

    while (1);
}

/**
  * @}
  */
/**
  * @}
  */

/************* (C) COPYRIGHT Eastsoft Microelectronics *****END OF FILE****/
