/**********************************************************************************
 *
 * @file    main.c
 * @brief   main C file
 *
 * @date    15 Mar. 2023
 * @author  AE Team
 * @note
 *          Change Logs:
 *          Date            Author          Notes
 *          15 Mar. 2023    Lisq            the first version
 *
 * Copyright (C) Shanghai Eastsoft Microelectronics Co. Ltd. All rights reserved.
 *
 * SPDX-License-Identifier: Apache-2.0
 *
 * Licensed under the Apache License, Version 2.0 (the License); you may
 * not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an AS IS BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 *
 **********************************************************************************
 */
/* Includes ------------------------------------------------------------------ */
#include "main.h"

/* Private Macros ------------------------------------------------------------ */
/* Private Variables --------------------------------------------------------- */
/* Public Variables ---------------------------------------------------------- */
ald_timer_handle_t g_ad16c4t_init;
ald_timer_oc_init_t g_oc_init;
ald_gpio_init_t g_gpio_init;
ald_timer_clock_config_t g_ad16c4t_clock_config;
ald_timer_slave_config_t g_slave_config;

/* Private Constants --------------------------------------------------------- */
/* Private function prototypes ----------------------------------------------- */
/* Private Function ---------------------------------------------------------- */
/** @addtogroup Projects_Examples_ALD
  * @{
  */

/** @addtogroup Examples
  * @{
  */
/**
  * @brief  ald timer compare elapsed callback
  * @param  arg: Pointer to timer_handle_t structure.
  * @retval None.
  */
void ald_timer_compare_callback(struct ald_timer_handle_s *arg)
{
    switch (arg->ch)
    {
        case ALD_TIMER_ACTIVE_CHANNEL_1:
            __NOP();
            break;

        case ALD_TIMER_ACTIVE_CHANNEL_2:
            __NOP();
            break;

        case ALD_TIMER_ACTIVE_CHANNEL_3:
            __NOP();
            break;

        case ALD_TIMER_ACTIVE_CHANNEL_4:
            __NOP();
            break;

        default:
            break;
    }
}

/**
  * @brief  Initializate pin of module.
  * @retval None
  */
static void init_gpio(void)
{
    ald_gpio_init_t gpio_init;
    /* initialize the gpio_init */
    memset(&gpio_init, 0x0, sizeof(gpio_init));

    gpio_init.mode = ALD_GPIO_MODE_INPUT;  /* input */
    gpio_init.od   = ALD_GPIO_PUSH_PULL;   /* push pull */
    gpio_init.pupd = ALD_GPIO_PUSH_DOWN;   /* pull down */
    gpio_init.type = ALD_GPIO_TYPE_CMOS;
    gpio_init.func = ALD_GPIO_FUNC_1;      /* GPIO function 1 */
    ald_gpio_init(TRIGER_PORT, TRIGER_PIN, &gpio_init);

    gpio_init.mode = ALD_GPIO_MODE_OUTPUT; /* output */
    gpio_init.od   = ALD_GPIO_PUSH_PULL;   /* push pull */
    gpio_init.pupd = ALD_GPIO_FLOATING;    /* no pull up and pull down */
    gpio_init.type = ALD_GPIO_TYPE_CMOS;
    gpio_init.func = ALD_GPIO_FUNC_3;      /* GPIO function 3 */
    ald_gpio_init(PULSE_PORT, PULSE_PIN, &gpio_init);

    return;
}

/**
  * @brief  Initialize PIS for IO triger AD16C4T1 enable
  * @param  None
  * @retval None
  */
void init_pis(void)
{
    ald_pis_handle_t pis_init;

    memset(&pis_init, 0, sizeof(ald_pis_handle_t));
    pis_init.perh = PIS;
    pis_init.init.producer_src    = ALD_PIS_GPIO_PIN9;       /* production signal */
    pis_init.init.producer_clk    = ALD_PIS_CLK_HCLK;        /* production clock zone */
    pis_init.init.producer_edge   = ALD_PIS_EDGE_UP;         /* production effective edge */
    pis_init.init.producer_signal = ALD_PIS_OUT_LEVEL;       /* production signal type */
    pis_init.init.consumer_trig   = ALD_PIS_CH4_TIMER0_ITR0; /* comsumer signal */
    pis_init.init.consumer_clk    = ALD_PIS_CLK_PCLK;        /* comsumer clock zone */
    ald_pis_create(&pis_init);
}

/**
  * @brief:  Initialize AD16C4T to timely overflow funtion
  * @param:  None
  * @retval: None
  */
static void init_timer(void)
{
    /* Initialize AD16C4T */
    memset(&g_ad16c4t_init, 0x0, sizeof(g_ad16c4t_init));
    g_ad16c4t_init.perh              = AD16C4T;
    g_ad16c4t_init.init.prescaler    = 1;                      /* clk_count: 36MHz */
    g_ad16c4t_init.init.mode         = ALD_TIMER_CNT_MODE_UP;      /* count mode is center1 */
    g_ad16c4t_init.init.period       = 36000 - 1;              /* period is 36000 count clock */
    g_ad16c4t_init.init.clk_div      = ALD_TIMER_CLOCK_DIV1;       /* working clock of dead time and filter */
    g_ad16c4t_init.init.re_cnt       = 0;                      /* no repeat count */
    g_ad16c4t_init.delay_elapse_cbk  = ald_timer_compare_callback; /* compare elapsed callback */
    ald_timer_oc_init(&g_ad16c4t_init);

    /* Initialize clock source */
    g_ad16c4t_clock_config.source = ALD_TIMER_SRC_INTER;   /* clock sources is internal */
    ald_timer_config_clock_source(&g_ad16c4t_init, &g_ad16c4t_clock_config);

    /* compare output configuration for channel1 */
    memset(&g_oc_init, 0x0, sizeof(g_oc_init));
    g_oc_init.oc_mode      = ALD_TIMER_OC_MODE_PWM2;      /* compare output mode is PWM2 */
    g_oc_init.oc_polarity  = ALD_TIMER_OC_POLARITY_HIGH;  /* CHxO compare output polarity is high */
    g_oc_init.ocn_polarity = ALD_TIMER_OCN_POLARITY_HIGH; /* CHxON compare output polarity is high */
    g_oc_init.ocn_idle     = ALD_TIMER_OCN_IDLE_RESET;    /* CHxO idle is low */
    g_oc_init.oc_idle      = ALD_TIMER_OC_IDLE_RESET;     /* CHxON idle is low */
    /* Set the compare value for channel 1 */
    g_oc_init.pulse = 27000 - 1;
    ald_timer_oc_config_channel(&g_ad16c4t_init, &g_oc_init, ALD_TIMER_CHANNEL_1);
    ald_timer_one_pulse_init(&g_ad16c4t_init, ALD_TIMER_OP_MODE_SINGLE);

    /* config slave mode */
    memset(&g_slave_config, 0x0, sizeof(g_slave_config));
    g_slave_config.input = ALD_TIMER_TS_ITR0;  /* TI singel is IT0 */
    g_slave_config.mode = ALD_TIMER_MODE_TRIG; /* slave mode is triger */
    ald_timer_slave_config_sync(&g_ad16c4t_init, &g_slave_config);

    /* Start compare output from channel 1 */
    ald_timer_oc_start_by_it(&g_ad16c4t_init, ALD_TIMER_CHANNEL_1);

    /* Enable NVIC AD16C4T CC interrupt */
    ald_mcu_irq_config(AD16C4T_CC_IRQn, 3, ENABLE);
    /* Enable NVIC AD16C4T UP interrupt */
    ald_mcu_irq_config(AD16C4T_UP_TRIG_COM_IRQn, 4, ENABLE);
}

/**
  * @brief:  Main program.
  * @param:  None
  * @retval: None
  */
int main(void)
{
    /* Initialize ALD */
    ald_cmu_init();
    /* Configure system clock */
    ald_cmu_pll_config(ALD_CMU_PLL_INPUT_HOSC8M, ALD_CMU_PLL_OUTPUT_72M);
    ald_cmu_clock_config(ALD_CMU_CLOCK_PLL, 72000000);

    /* Enable ALL peripheral */
    ald_cmu_perh_clock_config(ALD_CMU_PERH_ALL, ENABLE);

    __enable_irq();

    /* Initialize pin */
    init_gpio();

    /* Initialize pis */
    init_pis();

    /* Initialize timer */
    init_timer();

    while (1)
    {
        ald_delay_1ms(1000);
    }
}

/**
  * @}
  */
/**
  * @}
  */

/************* (C) COPYRIGHT Eastsoft Microelectronics *****END OF FILE****/
