/**********************************************************************************
 *
 * @file    spi_flash.c
 * @brief   SPI Flash file for DEMO
 *
 * @date    14 Mar. 2023
 * @author  AE Team
 * @note
 *          Change Logs:
 *          Date            Author          Notes
 *          14 Mar. 2023    Lisq            the first version
 *
 * Copyright (C) Shanghai Eastsoft Microelectronics Co. Ltd. All rights reserved.
 *
 * SPDX-License-Identifier: Apache-2.0
 *
 * Licensed under the Apache License, Version 2.0 (the License); you may
 * not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an AS IS BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 *
 **********************************************************************************
 */

/* Includes ------------------------------------------------------------------ */

#include "spi_flash.h"

/* Private Macros ------------------------------------------------------------ */

/* Private Variables --------------------------------------------------------- */

static ald_spi_handle_t s_spi;

/* Public Variables ---------------------------------------------------------- */

/* Private Constants --------------------------------------------------------- */

/* Private function prototypes ----------------------------------------------- */

static void delay(int i);
static void init_spi_pin(void);

/* Private Function ---------------------------------------------------------- */

/**
  * @brief  flash sector erase function
  * @param  addr: Specific address which sector to be erase.
  * @retval Status.
  */
ald_status_t flash_sector_erase(uint32_t addr)
{
    uint8_t cmd_buf[4];
    uint8_t i = 0U;

    cmd_buf[0] = FLASH_ERASE;       /*Flash sector erase command*/
    cmd_buf[1] = (addr >> 16) & 0xff;   /*24 bit Flash address*/
    cmd_buf[2] = (addr >> 8) & 0xff;
    cmd_buf[3] = addr & 0xff;

    FLASH_CS_CLR();     /*Choose lower, the selected Flash*/

    if (ald_spi_send_byte_fast(&s_spi, FLASH_WRITE_ENABLE) != ALD_OK)  /*First write send enabling instruction*/
    {
        FLASH_CS_SET();
        return ALD_ERROR;
    }

    FLASH_CS_SET();    /*Pick up and release of Flash*/

    delay(100);
    FLASH_CS_CLR();

    for (i = 0; i < sizeof(cmd_buf); i++)     /*Send the sector erase instructions and Flash address three bytes*/
    {
        if (ald_spi_send_byte_fast(&s_spi, cmd_buf[i]) != ALD_OK)
        {
            FLASH_CS_SET();
            return ALD_ERROR;
        }
    }

    FLASH_CS_SET();

    return ALD_OK;
}

/**
  * @brief  transmit an amount of data in blocking mode.
  * @param  addr: Specific address which to be write.
  * @param  buf: Pointer to data buffer
  * @param  size: Amount of data to be sent
  * @retval Status, see @ref ald_status_t.
  */
ald_status_t flash_write(uint32_t addr, char *buf, uint16_t size)
{
    uint8_t cmd_buf[4];
    uint16_t i = 0U;

    if (buf == NULL)
        return ALD_ERROR;

    cmd_buf[0] = FLASH_PROGRAM;
    cmd_buf[1] = (addr >> 16) & 0xff;
    cmd_buf[2] = (addr >> 8) & 0xff;
    cmd_buf[3] = addr & 0xff;

    FLASH_CS_CLR();     /*Choose lower, the selected Flash*/

    if (ald_spi_send_byte_fast(&s_spi, FLASH_WRITE_ENABLE) != ALD_OK)
    {
        FLASH_CS_SET();
        return ALD_ERROR;
    }

    FLASH_CS_SET();    /*Pick up and release of Flash*/

    delay(100);
    FLASH_CS_CLR();

    for (i = 0; i < sizeof(cmd_buf); i++)     /*Send the programming instructions and Flash address three bytes*/
    {
        if (ald_spi_send_byte_fast(&s_spi, cmd_buf[i]) != ALD_OK)
        {
            FLASH_CS_SET();
            return ALD_ERROR;
        }
    }

    for (i = 0; i < size; i++)  /*To write the data sent to the Flash*/
    {
        if (ald_spi_send_byte_fast(&s_spi, buf[i]) != ALD_OK)
        {
            FLASH_CS_SET();
            return ALD_ERROR;
        }
    }

    FLASH_CS_SET();

    return ALD_OK;
}

/**
  * @brief  Receive an amount of data in blocking mode.
  * @param  addr: address of flash where want to read.
  * @param  buf: Pointer to data buffer
  * @param  size: Amount of data to be received
  * @retval Status, see @ref ald_status_t.
  */
ald_status_t flash_read(uint32_t addr, char *buf, uint16_t size)
{
    uint8_t cmd_buf[4];
    uint16_t i = 0U;
    int r_flag = 0;

    if (buf == NULL)
        return ALD_ERROR;

    cmd_buf[0] = FLASH_READ;
    cmd_buf[1] = (addr >> 16) & 0xff;
    cmd_buf[2] = (addr >> 8) & 0xff;
    cmd_buf[3] = addr & 0xff;

    FLASH_CS_CLR();     /*Choose lower, the selected Flash*/

    for (i = 0; i < sizeof(cmd_buf); i++)   /*Send the editor & reader instructions and Flash address three bytes*/
    {
        if (ald_spi_send_byte_fast(&s_spi, cmd_buf[i]) != ALD_OK)
        {
            FLASH_CS_SET();     /*Pick up and release of Flash*/
            return ALD_ERROR;
        }
    }

    for (i = 0; i < size; i++)  /*Send the programming instructions and Flash address three bytes*/
    {
        buf[i] = ald_spi_recv_byte_fast(&s_spi, &r_flag);

        if (r_flag != ALD_OK)
        {
            FLASH_CS_SET();
            return ALD_ERROR;
        }
    }

    FLASH_CS_SET();

    return ALD_OK;
}

/**
  * @brief  wait until flash unbusy.
  * @retval Status, see @ref ald_status_t.
  */
ald_status_t flash_wait_unbusy(void)
{
    uint8_t status;
    int r_flag = 0;

    FLASH_CS_CLR(); /*Choose lower, the selected Flash*/

    if (ald_spi_send_byte_fast(&s_spi, (uint8_t)FLASH_STATUS) != ALD_OK)   /*Send to read status command*/
    {
        FLASH_CS_SET();     /*Pick up and release of Flash*/
        return ALD_ERROR;
    }

    do
    {
        status = ald_spi_recv_byte_fast(&s_spi, &r_flag);

        if (r_flag != ALD_OK)
        {
            FLASH_CS_SET();
            return ALD_ERROR;
        }
    }
    while (status & 0x01);

    FLASH_CS_SET();

    return ALD_OK;
}

/**
  * @brief  Read flash id in blocking mode.
  * @retval flash id.
  */
uint32_t flash_read_id(void)
{
    uint8_t i;
    int r_flag = 0;
    uint8_t flash_id[4] = {0U};

    flash_id[0] = FLASH_ID;

    FLASH_CS_CLR(); /* Choose lower, the selected Flash */

    for (i = 0; i < sizeof(flash_id); i++)
    {
        if (ald_spi_send_byte_fast(&s_spi, flash_id[i]) != ALD_OK)
        {
            FLASH_CS_SET();     /* Pick up and release of Flash */
            return ALD_ERROR;
        }
    }

    for (i = 0; i < 3; i++)
    {
        flash_id[i] = ald_spi_recv_byte_fast(&s_spi, &r_flag);

        if (r_flag != ALD_OK)
        {
            FLASH_CS_SET();
            return ALD_ERROR;
        }
    }

    FLASH_CS_SET();

    return ((flash_id[0] << 16) | (flash_id[1] << 8) | (flash_id[2]));  /* Manufacturer ID flash_id [0] and device ID flash_id [1] */
}

/**
  * @brief spi function
  * @retval None.
  */
void init_mcu_spi(void)
{
    init_spi_pin();

    s_spi.perh           = SPI0;               /*Using SPI0*/
    s_spi.init.mode      = ALD_SPI_MODE_MASTER;    /*SPI host mode*/
    s_spi.init.baud      = ALD_SPI_BAUD_8;         /*clock / 8*/
    s_spi.init.data_size = ALD_SPI_DATA_SIZE_8;    /*8 bit pattern*/
    s_spi.init.polarity  = ALD_SPI_CPOL_HIGH;      /*Free high level*/
    s_spi.init.phase     = ALD_SPI_CPHA_SECOND;    /*The second edge receiving data*/
    s_spi.init.first_bit = ALD_SPI_FIRSTBIT_MSB;   /*Send the MSB first*/
    s_spi.init.dir       = ALD_SPI_DIRECTION_2LINES;
    s_spi.init.ss_en     = DISABLE;
    s_spi.init.crc_calc  = DISABLE;

    ald_spi_init(&s_spi);  /*According to the parameter initialization SPI peripherals*/
}

/**
  * @brief  Initializate spi flash pin
  * @retval None.
  */
static void init_spi_pin(void)
{
    ald_gpio_init_t gpio_init;

    gpio_init.mode = ALD_GPIO_MODE_OUTPUT;
    gpio_init.od   = ALD_GPIO_PUSH_PULL;
    gpio_init.pupd = ALD_GPIO_PUSH_UP;
    gpio_init.flt  = ALD_GPIO_FILTER_DISABLE;
    gpio_init.odrv = ALD_GPIO_OUT_DRIVE_STRONG;
    gpio_init.type = ALD_GPIO_TYPE_CMOS;
    gpio_init.func = ALD_GPIO_FUNC_1;

    ald_gpio_init(SPI_NSS_PORT, SPI_NSS_PIN, &gpio_init);  /*Initialize the selected pin*/
    FLASH_CS_SET();   /*Choose the output high, release of Flash*/

    gpio_init.func = ALD_GPIO_FUNC_2;
    ald_gpio_init(SPI_SCK_PORT, SPI_SCK_PIN, &gpio_init);  /*Initialize the clock output pin*/
    ald_gpio_init(SPI_MOSI_PORT, SPI_MOSI_PIN, &gpio_init);  /*Initialize the MOSI pin*/

    gpio_init.mode = ALD_GPIO_MODE_INPUT;
    ald_gpio_init(SPI_MISO_PORT, SPI_MISO_PIN, &gpio_init);  /*Initializes the MISO pins*/
}

/**
  * @brief  delay some time.
  * @retval None.
  */
static void delay(int i)
{
    while (i--) ;
}

/**
  * @}
  */
/**
  * @}
  */

/************* (C) COPYRIGHT Eastsoft Microelectronics *****END OF FILE****/
