/**********************************************************************************
 *
 * @file    main.c
 * @brief   Main file for DEMO
 *
 * @date    14 Feb. 2023
 * @author  AE Team
 * @note
 *          Change Logs:
 *          Date            Author          Notes
 *          14 Feb. 2023    Lisq            the first version
 *
 * Copyright (C) Shanghai Eastsoft Microelectronics Co. Ltd. All rights reserved.
 *
 * SPDX-License-Identifier: Apache-2.0
 *
 * Licensed under the Apache License, Version 2.0 (the License); you may
 * not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an AS IS BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 *
 **********************************************************************************
 */

/* Includes ------------------------------------------------------------------ */

#include <string.h>
#include "main.h"

/* Private Macros ------------------------------------------------------------ */

/* Private Variables --------------------------------------------------------- */
uint8_t g_send_buf[20] = {0U};
uint8_t g_recv_buf[20] = {0U};
uint8_t g_send_len = sizeof(g_send_buf);
uint8_t g_recv_len = sizeof(g_recv_buf);
uint8_t g_send_ind = 0U;
uint8_t g_recv_ind = 0U;

/* Public Variables ---------------------------------------------------------- */
ald_spi_handle_t g_spi_config;

/* Private Constants --------------------------------------------------------- */

/* Private function prototypes ----------------------------------------------- */

/* Private Function ---------------------------------------------------------- */

/**
  * @brief  Initializate pin of spi module.
  * @retval None
  */
void init_spi_pin(void)
{
    ald_gpio_init_t gpio_init;

    /* Initialize nss pin */
    gpio_init.mode  = ALD_GPIO_MODE_INPUT;
    gpio_init.od    = ALD_GPIO_PUSH_PULL;
    gpio_init.pupd  = ALD_GPIO_PUSH_UP;
    gpio_init.odrv  = ALD_GPIO_OUT_DRIVE_STRONG;
    gpio_init.flt   = ALD_GPIO_FILTER_DISABLE;
    gpio_init.type  = ALD_GPIO_TYPE_CMOS;
    gpio_init.func  = ALD_GPIO_FUNC_2;
    ald_gpio_init(SPI_NSS_PORT, SPI_NSS_PIN, &gpio_init);
    ald_gpio_write_pin(SPI_NSS_PORT, SPI_NSS_PIN, 1);

    /* Initialize sck pin */
    gpio_init.mode  = ALD_GPIO_MODE_INPUT;
    gpio_init.od    = ALD_GPIO_PUSH_PULL;
    gpio_init.pupd  = ALD_GPIO_PUSH_UP;
    gpio_init.odrv =  ALD_GPIO_OUT_DRIVE_STRONG;
    gpio_init.flt   = ALD_GPIO_FILTER_DISABLE;
    gpio_init.type  = ALD_GPIO_TYPE_CMOS;
    gpio_init.func  = ALD_GPIO_FUNC_2;
    ald_gpio_init(SPI_SCK_PORT, SPI_SCK_PIN, &gpio_init);

    /* Initialize miso pin */
    gpio_init.mode  = ALD_GPIO_MODE_OUTPUT;
    gpio_init.od    = ALD_GPIO_PUSH_PULL;
    gpio_init.pupd  = ALD_GPIO_PUSH_UP;
    gpio_init.odrv  = ALD_GPIO_OUT_DRIVE_STRONG;
    gpio_init.flt   = ALD_GPIO_FILTER_DISABLE;
    gpio_init.type  = ALD_GPIO_TYPE_CMOS;
    gpio_init.func  = ALD_GPIO_FUNC_2;
    ald_gpio_init(SPI_MISO_PORT, SPI_MISO_PIN, &gpio_init);

    /* Initialize mosi pin */
    gpio_init.mode  = ALD_GPIO_MODE_INPUT;
    gpio_init.od    = ALD_GPIO_PUSH_PULL;
    gpio_init.pupd  = ALD_GPIO_PUSH_UP;
    gpio_init.odrv  = ALD_GPIO_OUT_DRIVE_STRONG;
    gpio_init.flt   = ALD_GPIO_FILTER_DISABLE;
    gpio_init.type  = ALD_GPIO_TYPE_CMOS;
    gpio_init.func  = ALD_GPIO_FUNC_2;
    ald_gpio_init(SPI_MOSI_PORT, SPI_MOSI_PIN, &gpio_init);

    return;
}

/**
  * @brief  Send and receive a message complete.
  * @param  arg: Pointer to spi_handle_t structure.
  * @retval None.
  */
void spi_send_recv_complete(ald_spi_handle_t *arg)
{
    ald_spi_send_recv_by_dma(&g_spi_config, g_recv_buf, g_recv_buf, g_recv_len, ALD_DMA_CH_0, ALD_DMA_CH_1);

    return;
}

/**
  * @brief  Test main function
  * @retval Status.
  */
int main(void)
{
    /* Initialize ALD */
    ald_cmu_init();

    /* Configure system clock */
    ald_cmu_pll_config(ALD_CMU_PLL_INPUT_HOSC8M, ALD_CMU_PLL_OUTPUT_72M);
    ald_cmu_clock_config(ALD_CMU_CLOCK_PLL, 72000000);

    /* Enable all peripherals */
    ald_cmu_perh_clock_config(ALD_CMU_PERH_ALL, ENABLE);

    __enable_irq();

    /* Initialize pin */
    init_spi_pin();

    /* clear spi_handle_t structure */
    memset(&g_spi_config, 0x0, sizeof(g_spi_config));
    /* Initialize spi */
    g_spi_config.perh           = SPI0;
    g_spi_config.init.mode      = ALD_SPI_MODE_SLAVER;
    g_spi_config.init.dir       = ALD_SPI_DIRECTION_2LINES;
    g_spi_config.init.data_size = ALD_SPI_DATA_SIZE_8;
    g_spi_config.init.baud      = ALD_SPI_BAUD_128;
    g_spi_config.init.phase     = ALD_SPI_CPHA_SECOND;
    g_spi_config.init.polarity  = ALD_SPI_CPOL_HIGH;
    g_spi_config.init.first_bit = ALD_SPI_FIRSTBIT_MSB;
    g_spi_config.init.ss_en     = DISABLE;
    g_spi_config.init.crc_calc  = DISABLE;
    g_spi_config.tx_rx_cplt_cbk = spi_send_recv_complete;
    ald_spi_init(&g_spi_config);

    /* send and receive data by it */
    ald_spi_send_recv_by_dma(&g_spi_config, g_recv_buf, g_recv_buf, g_recv_len, ALD_DMA_CH_0, ALD_DMA_CH_1);

    while (1)
    {
        ald_delay_1ms(1000);
    }
}

/**
  * @}
  */
/**
  * @}
  */

/************* (C) COPYRIGHT Eastsoft Microelectronics *****END OF FILE****/
