/*
 * Copyright (c) 2006-2021, RT-Thread Development Team
 *
 * SPDX-License-Identifier: Apache-2.0
 *
 * Change Logs:
 * Date           Author       Notes
 * 2018-08-24     yangjie      the first version
 * 2024-01-17     shiwa        update comments
 */

/*
 * 程序清单：信号量例程
 *
 * 该例程创建了一个动态信号量，初始化两个线程，线程1在count每计数10次时，
 * 发送一个信号量，线程2在接收信号量后，对number进行加1操作
 */
/* Includes ------------------------------------------------------------------ */
#include <rtthread.h>

/* Private Macros ------------------------------------------------------------ */
#define THREAD_PRIORITY         25
#define THREAD_TIMESLICE        5

/* Private Constants --------------------------------------------------------- */

/* Private function prototypes ----------------------------------------------- */

/* Private Variables --------------------------------------------------------- */
/* 指向信号量的指针 */
static rt_sem_t s_dynamic_sem = RT_NULL;

ALIGN(RT_ALIGN_SIZE)
static char s_thread1_stack[256];
static struct rt_thread s_thread1;

ALIGN(RT_ALIGN_SIZE)
static char s_thread2_stack[256];
static struct rt_thread s_thread2;

/* Public Variables ---------------------------------------------------------- */

/* Private Function ---------------------------------------------------------- */
static void rt_thread1_entry(void *parameter)
{
    static rt_uint8_t s_count = 0;

    while (1)
    {
        if (s_count <= 100)
        {
            s_count++;
        }
        else
            return;

        /* count每计数10次，就释放一次信号量 */
        if (0 == (s_count % 10))
        {
            rt_kprintf("s_thread1 release a dynamic semaphore.\n");
            rt_sem_release(s_dynamic_sem);
        }
    }
}

static void rt_thread2_entry(void *parameter)
{
    static rt_err_t s_result;
    static rt_uint8_t s_number = 0;

    while (1)
    {
        /* 永久方式等待信号量，获取到信号量，则执行number自加的操作 */
        s_result = rt_sem_take(s_dynamic_sem, RT_WAITING_FOREVER);

        if (s_result != RT_EOK)
        {
            rt_kprintf("s_thread2 take a dynamic semaphore, failed.\n");
            rt_sem_delete(s_dynamic_sem);
            return;
        }
        else
        {
            s_number++;
            rt_kprintf("s_thread2 take a dynamic semaphore. number = %d\n", s_number);
        }
    }
}

/* 信号量示例的初始化 */
int semaphore_sample(void)
{
    /* 创建一个动态信号量，初始值是0 */
    s_dynamic_sem = rt_sem_create("dsem", 0, RT_IPC_FLAG_PRIO);

    if (s_dynamic_sem == RT_NULL)
    {
        rt_kprintf("create dynamic semaphore failed.\n");
        return -1;
    }
    else
    {
        rt_kprintf("create done. dynamic semaphore value = 0.\n");
    }

    rt_thread_init(&s_thread1,
                   "s_thread1",
                   rt_thread1_entry,
                   RT_NULL,
                   &s_thread1_stack[0],
                   sizeof(s_thread1_stack),
                   THREAD_PRIORITY, THREAD_TIMESLICE);
    rt_thread_startup(&s_thread1);

    rt_thread_init(&s_thread2,
                   "s_thread2",
                   rt_thread2_entry,
                   RT_NULL,
                   &s_thread2_stack[0],
                   sizeof(s_thread2_stack),
                   THREAD_PRIORITY - 1, THREAD_TIMESLICE);
    rt_thread_startup(&s_thread2);

    return 0;
}

