/*
 * Copyright (c) 2006-2021, RT-Thread Development Team
 *
 * SPDX-License-Identifier: Apache-2.0
 *
 * Change Logs:
 * Date           Author       Notes
 * 2018-08-24     yangjie      the first version
 * 2024-01-17     shiwa        update comments
 */

/*
 * 程序清单：互斥锁例程
 *
 * 互斥锁是一种保护共享资源的方法。当一个线程拥有互斥锁的时候，
 * 可以保护共享资源不被其他线程破坏。线程1对2个number分别进行加1操作
 * 线程2也会对2个number分别进行加1操作。使用互斥量保证2个number值保持一致
 */
/* Includes ------------------------------------------------------------------ */
#include <rtthread.h>

/* Private Macros ------------------------------------------------------------ */
#define THREAD_PRIORITY         5
#define THREAD_TIMESLICE        5

/* Private Constants --------------------------------------------------------- */

/* Private function prototypes ----------------------------------------------- */

/* Private Variables --------------------------------------------------------- */
/* 指向互斥量的指针 */
static rt_mutex_t s_dynamic_mutex = RT_NULL;
static rt_uint8_t s_number1, s_number2 = 0;

ALIGN(RT_ALIGN_SIZE)
static char s_thread1_stack[256];
static struct rt_thread s_thread1;

ALIGN(RT_ALIGN_SIZE)
static char s_thread2_stack[256];
static struct rt_thread s_thread2;
/* Public Variables ---------------------------------------------------------- */

/* Private Function ---------------------------------------------------------- */
static void rt_thread_entry1(void *parameter)
{
    while (1)
    {
        /* 线程1获取到互斥量后，先后对number1、number2进行加1操作，然后释放互斥量 */
        rt_mutex_take(s_dynamic_mutex, RT_WAITING_FOREVER);
        s_number1++;
        rt_thread_mdelay(10);
        s_number2++;
        rt_mutex_release(s_dynamic_mutex);
    }
}

static void rt_thread_entry2(void *parameter)
{
    while (1)
    {
        /* 线程2获取到互斥量后，检查number1、number2的值是否相同，相同则表示mutex起到了锁的作用 */
        rt_mutex_take(s_dynamic_mutex, RT_WAITING_FOREVER);

        if (s_number1 != s_number2)
        {
            rt_kprintf("not protect.s_number1 = %d, mumber2 = %d \n", s_number1, s_number2);
        }
        else
        {
            rt_kprintf("mutex protect ,s_number1 = mumber2 is %d\n", s_number1);
        }

        s_number1++;
        s_number2++;
        rt_mutex_release(s_dynamic_mutex);

        if (s_number1 >= 50)
            return;
    }
}

/* 互斥量示例的初始化 */
int mutex_sample(void)
{
    /* 创建一个动态互斥量 */
    s_dynamic_mutex = rt_mutex_create("dmutex", RT_IPC_FLAG_PRIO);

    if (s_dynamic_mutex == RT_NULL)
    {
        rt_kprintf("create dynamic mutex failed.\n");
        return -1;
    }

    rt_thread_init(&s_thread1,
                   "s_thread1",
                   rt_thread_entry1,
                   RT_NULL,
                   &s_thread1_stack[0],
                   sizeof(s_thread1_stack),
                   THREAD_PRIORITY, THREAD_TIMESLICE);
    rt_thread_startup(&s_thread1);

    rt_thread_init(&s_thread2,
                   "s_thread2",
                   rt_thread_entry2,
                   RT_NULL,
                   &s_thread2_stack[0],
                   sizeof(s_thread2_stack),
                   THREAD_PRIORITY - 1, THREAD_TIMESLICE);
    rt_thread_startup(&s_thread2);
    return 0;
}

