/*
 * Copyright (c) 2006-2019, RT-Thread Development Team
 *
 * SPDX-License-Identifier: Apache-2.0
 *
 * Change Logs:
 * Date           Author       Notes
 * 2017-07-24     Tanek        the first version
 * 2018-11-12     Ernest Chen  modify copyright
 * 2024-01-17     shiwa        Update comments
 */

/* Includes ------------------------------------------------------------------ */
#include <stdint.h>
#include <rthw.h>
#include <rtthread.h>
#include "drv_uart.h"


/* Private Macros ------------------------------------------------------------ */
#if defined(RT_USING_USER_MAIN) && defined(RT_USING_HEAP)
#ifndef RT_HEAP_SIZE
#define RT_HEAP_SIZE 1024
#endif /* RT_HEAP_SIZE */
static uint32_t s_rt_heap[RT_HEAP_SIZE];     /* heap default size: 4K(1024 * 4) */
#endif /* RT_USING_USER_MAIN && RT_USING_HEAP */

/* Private Constants --------------------------------------------------------- */

/* Private function prototypes ----------------------------------------------- */

/* Private Variables --------------------------------------------------------- */

/* Public Variables ---------------------------------------------------------- */

/* Private Function ---------------------------------------------------------- */

/*
Updates the variable SystemCoreClock and must be called
whenever the core clock is changed during program execution.

Holds the system core clock, which is the system clock
frequency supplied to the SysTick timer and the processor
core clock.
md_cmu_get_sys_clock();
*/


/*
 * Config systick
 */
void SysTick_Configuration(void)
{
    /* ticks = sysclk / RT_TICK_PER_SECOND */
	csi_coret_config(md_cmu_get_sys_clock() / RT_TICK_PER_SECOND, CLINT_IRQn);
	csi_vic_set_prio(CLINT_IRQn, 3);
    csi_vic_enable_sirq(CLINT_IRQn);
}

#if defined(RT_USING_USER_MAIN) && defined(RT_USING_HEAP)
RT_WEAK void *rt_heap_begin_get(void)
{
    return s_rt_heap;
}

RT_WEAK void *rt_heap_end_get(void)
{
    return s_rt_heap + RT_HEAP_SIZE;
}
#endif /* RT_USING_USER_MAIN && RT_USING_HEAP */

/**
 * This function will initial your board.
 */
void rt_hw_board_init()
{
    
    /* Configure system clock */
    md_cmu_pll_config(MD_CMU_PLL_INPUT_HOSC8M, MD_CMU_PLL_OUTPUT_72M);
    md_cmu_clock_config(MD_CMU_CLOCK_PLL, 72000000);
	
    /* Enable ALL peripheral */
    MD_SYSCFG_UNLOCK();
    md_cmu_enable_perh_all();
    MD_SYSCFG_LOCK();
	
	SysTick_Configuration();
	md_mcu_irq_config(MACHINE_MODE_SOFT_IRQn, 3, ENABLE);
	CLIC->CLICINT[MACHINE_MODE_SOFT_IRQn].ATTR |= (3); 
	
    uart_pin_init();
    uart_init();

    /* Call components board initial (use INIT_BOARD_EXPORT()) */
#ifdef RT_USING_COMPONENTS_INIT
    rt_components_board_init();
#endif /* RT_USING_COMPONENTS_INIT */

#if defined(RT_USING_USER_MAIN) && defined(RT_USING_HEAP)
    rt_system_heap_init(rt_heap_begin_get(), rt_heap_end_get());
#endif /* RT_USING_USER_MAIN && RT_USING_HEAP */
}

void __attribute__((interrupt)) CLINT_Handler(void)
{
	csi_coret_clr(md_cmu_get_sys_clock() / RT_TICK_PER_SECOND, CLINT_IRQn);
	
	md_inc_tick();
	
	/* enter interrupt */
	rt_interrupt_enter();
	
	rt_tick_increase();
	
	/* leave interrupt */
	rt_interrupt_leave();
}