/*
 * Copyright (c) 2006-2021, RT-Thread Development Team
 *
 * SPDX-License-Identifier: Apache-2.0
 *
 * Change Logs:
 * Date           Author       Notes
 * 2018-08-24     yangjie      the first version
 * 2024-01-17     shiwa        update comments
 */

/*
 * 程序清单：消息队列例程
 *
 * 这个程序会创建2个动态线程，一个线程会从消息队列中收取消息；一个线程会定时给消
 * 息队列发送 普通消息和紧急消息。
 */
/* Includes ------------------------------------------------------------------ */
#include <rtthread.h>

/* Private Macros ------------------------------------------------------------ */
#define THREAD_PRIORITY      5
#define THREAD_TIMESLICE     40

/* Private Constants --------------------------------------------------------- */

/* Private function prototypes ----------------------------------------------- */

/* Private Variables --------------------------------------------------------- */
/* 消息队列控制块 */
static struct rt_messagequeue s_msgq;
/* 消息队列中用到的放置消息的内存池 */
static rt_uint8_t s_msg_pool[256];

ALIGN(RT_ALIGN_SIZE)
static char s_thread1_stack[256];
static struct rt_thread s_thread1;

/* Public Variables ---------------------------------------------------------- */

/* Private Function ---------------------------------------------------------- */
/* 线程1入口函数 */
static void thread1_entry(void *parameter)
{
    char buf = 0;
    rt_uint8_t cnt = 0;

    while (1)
    {
        /* 从消息队列中接收消息 */
        if (rt_mq_recv(&s_msgq, &buf, sizeof(buf), RT_WAITING_FOREVER) == RT_EOK)
        {
            rt_kprintf("s_thread1:recv:%c\n", buf);

            if (cnt == 19)
            {
                break;
            }
        }

        /* 延时50ms */
        cnt++;
        rt_thread_mdelay(50);
    }

    rt_kprintf("s_thread1: detach s_msgq \n");
    rt_mq_detach(&s_msgq);
}

ALIGN(RT_ALIGN_SIZE)
static char s_thread2_stack[256];
static struct rt_thread s_thread2;

/* 线程2入口 */
static void thread2_entry(void *parameter)
{
    int result;
    char buf = 'A';
    rt_uint8_t cnt = 0;

    while (1)
    {
        if (cnt == 8)
        {
            /* 发送紧急消息到消息队列中 */
            result = rt_mq_urgent(&s_msgq, &buf, 1);

            if (result != RT_EOK)
            {
                rt_kprintf("rt_mq_urgent ERR\n");
            }
            else
            {
                rt_kprintf("s_thread2:send-%c\n", buf);
            }
        }
        else if (cnt >= 20)/* 发送20次消息之后退出 */
        {
            rt_kprintf("message queue stop send, s_thread2 quit\n");
            break;
        }
        else
        {
            /* 发送消息到消息队列中 */
            result = rt_mq_send(&s_msgq, &buf, 1);

            if (result != RT_EOK)
            {
                rt_kprintf("rt_mq_send ERR\n");
            }

            rt_kprintf("s_thread2:send-%c\n", buf);
        }

        buf++;
        cnt++;
        /* 延时5ms */
        rt_thread_mdelay(5);
    }
}

/* 消息队列示例的初始化 */
int msgq_sample(void)
{
    rt_err_t result;

    /* 初始化消息队列 */
    result = rt_mq_init(&s_msgq,
                        "mqt",
                        &s_msg_pool[0],               /* 内存池指向msg_pool */
                        1,                          /* 每个消息的大小是 1 字节 */
                        sizeof(s_msg_pool),           /* 内存池的大小是msg_pool的大小 */
                        RT_IPC_FLAG_PRIO);          /* 如果有多个线程等待，按照先来先得到的方法分配消息 */

    if (result != RT_EOK)
    {
        rt_kprintf("init message queue failed.\n");
        return -1;
    }

    rt_thread_init(&s_thread1,
                   "s_thread1",
                   thread1_entry,
                   RT_NULL,
                   &s_thread1_stack[0],
                   sizeof(s_thread1_stack),
                   THREAD_PRIORITY, THREAD_TIMESLICE);
    rt_thread_startup(&s_thread1);

    rt_thread_init(&s_thread2,
                   "s_thread2",
                   thread2_entry,
                   RT_NULL,
                   &s_thread2_stack[0],
                   sizeof(s_thread2_stack),
                   THREAD_PRIORITY, THREAD_TIMESLICE);
    rt_thread_startup(&s_thread2);

    return 0;
}

