/**********************************************************************************
 *
 * @file    std_printf.c
 * @brief   printf to uart
 *
 * @date    17 Jan. 2023
 * @author  AE Team
 * @note
 *          Change Logs:
 *          Date            Author          Notes
 *          17Jan. 2023     Shiwa           the first version
 *
 * Copyright (C) Shanghai Eastsoft Microelectronics Co. Ltd. All rights reserved.
 *
 * SPDX-License-Identifier: Apache-2.0
 *
 * Licensed under the Apache License, Version 2.0 (the License); you may
 * not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an AS IS BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 *
 **********************************************************************************
 */

/* Includes ------------------------------------------------------------------ */
#include <stdarg.h>
#include <stdio.h>
#include <string.h>
#include "ald_cmu.h"
#include "ald_gpio.h"
#include "ald_uart.h"
#include "std_printf.h"


/* Private Macros ------------------------------------------------------------ */

/* Private Constants --------------------------------------------------------- */

/* Private function prototypes ----------------------------------------------- */

/* Private Variables --------------------------------------------------------- */
static uart_handle_t s_h_uart;
static char s_uart_buf[64];

/* Public Variables ---------------------------------------------------------- */

/* Private Function ---------------------------------------------------------- */

/**
  * @brief  Output debug information via UART.
  * @param  fmt: Varibale parameter
  * @retval None
  */
void printf_e(const char *fmt, ...)
{
	va_list args;

	va_start(args, fmt);
	vsnprintf(s_uart_buf, 64, fmt, args);
	va_end(args);

	ald_uart_send(&s_h_uart, (uint8_t *)s_uart_buf, strlen(s_uart_buf), 1000);
	return;
}

/**
  * @brief  Initialize the UART1
  * @retval None
  */
void uart1_init(void)
{
	gpio_init_t x;

	ald_cmu_perh_clock_config(CMU_PERH_GPIO, ENABLE);
	ald_cmu_perh_clock_config(CMU_PERH_UART1, ENABLE);
	
	/* Initialize tx pin */
	x.mode  = GPIO_MODE_OUTPUT;
	x.odos  = GPIO_PUSH_PULL;
	x.pupd  = GPIO_PUSH_UP;
	x.podrv = GPIO_OUT_DRIVE_6;
	x.nodrv = GPIO_OUT_DRIVE_6;
	x.flt   = GPIO_FILTER_DISABLE;
	x.type  = GPIO_TYPE_TTL;
	x.func  = GPIO_FUNC_3;
	ald_gpio_init(GPIOC, GPIO_PIN_0, &x);

	/* Initialize rx pin */
	x.mode  = GPIO_MODE_INPUT;
	x.odos  = GPIO_PUSH_PULL;
	x.pupd  = GPIO_PUSH_UP;
	x.podrv = GPIO_OUT_DRIVE_6;
	x.nodrv = GPIO_OUT_DRIVE_6;
	x.flt   = GPIO_FILTER_DISABLE;
	x.type  = GPIO_TYPE_TTL;
	x.func  = GPIO_FUNC_3;
	ald_gpio_init(GPIOC, GPIO_PIN_1, &x);

	/* Initialize uart */
	s_h_uart.perh             = UART1;
	s_h_uart.init.baud        = 115200;
	s_h_uart.init.word_length = UART_WORD_LENGTH_8B;
	s_h_uart.init.stop_bits   = UART_STOP_BITS_1;
	s_h_uart.init.parity      = UART_PARITY_NONE;
	s_h_uart.init.mode        = UART_MODE_UART;
	s_h_uart.init.fctl        = UART_HW_FLOW_CTL_DISABLE;
	s_h_uart.tx_cplt_cbk      = NULL;
	s_h_uart.rx_cplt_cbk      = NULL;
	s_h_uart.error_cbk        = NULL;
	ald_uart_init(&s_h_uart);

	return;
}
