/**********************************************************************************
 *
 * @file    main.c
 * @brief   Main file for DEMO
 *
 * @date    9 Feb. 2023
 * @author  AE Team
 * @note
 *          Change Logs:
 *          Date            Author          Notes
 *          9 Feb. 2023     Lisq            the first version
 *
 * Copyright (C) Shanghai Eastsoft Microelectronics Co. Ltd. All rights reserved.
 *
 * SPDX-License-Identifier: Apache-2.0
 *
 * Licensed under the Apache License, Version 2.0 (the License); you may
 * not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an AS IS BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 *
 **********************************************************************************
 */


/* Includes ----------------------------------------------------------------- */

#include <string.h>
#include "main.h"
#include "iap_rom.h"

/* Public Variables ---------------------------------------------------------- */

/* Public Function ---------------------------------------------------------- */

/* Private Macros ------------------------------------------------------------ */

/* Private Variables--------------------------------------------------------- */

/* Private Constants --------------------------------------------------------- */

/* Private function prototypes ----------------------------------------------- */

/* Private Function ---------------------------------------------------------- */
typedef  void (*FunVoidType)(void);

/**
  * @brief:  Init gpio for led.
  * @param:  None
  * @retval: None
  */
void gpio_init(void)
{
    md_gpio_init_t gpio_config;

    /* Initialize led pin */
    gpio_config.mode  = MD_GPIO_MODE_OUTPUT;
    gpio_config.odos  = MD_GPIO_PUSH_PULL;
    gpio_config.pupd  = MD_GPIO_PUSH_UP;
    gpio_config.odrv  = MD_GPIO_OUT_DRIVE_NORMAL;
    gpio_config.flt   = MD_GPIO_FILTER_DISABLE;
    gpio_config.type  = MD_GPIO_TYPE_CMOS;
    gpio_config.func  = MD_GPIO_FUNC_1;
    md_gpio_init(LED_PORT, LED_PIN, &gpio_config);

    md_gpio_write_pin(LED_PORT, LED_PIN, 1);

    return;
}

/**
  * @brief  init pins used for i2c.
  * @param  None
  * @retval None
  */
void i2c_pins_init(void)
{
    md_gpio_init_t gpio_init;
    md_gpio_init_struct(&gpio_init);

    /* Initialize scl pin */
    gpio_init.mode  = MD_GPIO_MODE_INPUT;
    gpio_init.odos  = MD_GPIO_OPEN_DRAIN;
    gpio_init.pupd  = MD_GPIO_PUSH_UP;
    gpio_init.odrv  = MD_GPIO_OUT_DRIVE_NORMAL;
    gpio_init.flt   = MD_GPIO_FILTER_DISABLE;
    gpio_init.type  = MD_GPIO_TYPE_CMOS;
    gpio_init.func  = MD_GPIO_FUNC_5;
    md_gpio_init(I2C_SCL_PORT, I2C_SCL_PIN, &gpio_init);

    /* Initialize sda pin */
    gpio_init.mode  = MD_GPIO_MODE_OUTPUT;
    gpio_init.odos  = MD_GPIO_OPEN_DRAIN;
    gpio_init.pupd  = MD_GPIO_PUSH_UP;
    gpio_init.odrv  = MD_GPIO_OUT_DRIVE_NORMAL;
    gpio_init.flt   = MD_GPIO_FILTER_DISABLE;
    gpio_init.type  = MD_GPIO_TYPE_CMOS;
    gpio_init.func  = MD_GPIO_FUNC_5;
    md_gpio_init(I2C_SDA_PORT, I2C_SDA_PIN, &gpio_init);

    return;
}

/**
  * @brief  define the function used to jump to app or boot.
  * @param  None
  * @retval None
  */
static void fsm_go(uint32_t para)
{
    uint32_t go_addr = 0x0U;

    __disable_irq();

    if (para == GO_APP)
    {
        go_addr = APP_ADDR;
    }
    else if (para == GO_BOOT)
    {
        go_addr = BOOT_ADDR;

        FLASH_REG_UNLOCK();
        IAPROM_PAGE_ERASE(CRC_CAL_ADDR, ~CRC_CAL_ADDR, 0);
        FLASH_REG_LOCK();
    }
    else
    {

    }

    MD_SYSCFG_UNLOCK();
    md_syscfg_set_cpu_boot_addr(go_addr);
    MD_SYSCFG_LOCK();

    __enable_irq();

    md_rmu_reset_system();

    return;
}

/**
  * @brief  Test main function
  * @retval Status.
  */
int main(void)
{
    static md_i2c_init_t s_i2c_init;

    /* Configure system clock */
    md_cmu_pll_config(MD_CMU_PLL_INPUT_HOSC8M, MD_CMU_PLL_OUTPUT_48M);
    md_cmu_clock_config(MD_CMU_CLOCK_PLL, 48000000);
    md_init_1ms_tick();

    /* Enable ALL peripheral */
    MD_SYSCFG_UNLOCK();
    md_cmu_enable_perh_all();
    MD_SYSCFG_LOCK();

    /* init gpio for led */
    gpio_init();

    /* Config i2c interrupt */
    md_mcu_irq_config(I2C0_IRQn, 4, ENABLE);
    __enable_irq();

    /* Init communication parameters */
    i2c_pins_init();
    md_i2c_struct_init(&s_i2c_init);
    /* Initialize I2C */
    s_i2c_init.module       = MD_I2C_MODULE_SLAVE;
    s_i2c_init.addr_mode    = MD_I2C_ADDR_7BIT;
    s_i2c_init.clk_speed    = MD_I2C_STANDARD_MODE_MAX_CLK;
    s_i2c_init.dual_addr    = MD_I2C_DUALADDR_ENABLE;
    s_i2c_init.general_call = MD_I2C_GENERALCALL_DISABLE;
    s_i2c_init.no_stretch   = MD_I2C_NOSTRETCH_DISABLE;
    s_i2c_init.own_addr1    = SLAVE_ADDR << 1;
    md_i2c_init(COMMUNICATION_TYPE, &s_i2c_init);

    md_i2c_enable_it_addr(COMMUNICATION_TYPE);
    md_i2c_enable_it_stop(COMMUNICATION_TYPE);
    md_i2c_enable_it_rxne(COMMUNICATION_TYPE);

    while (1)
    {
        if (g_flag)
        {
            g_flag = 0;

            fsm_go(GO_BOOT);
        }

        md_gpio_toggle_pin_output(LED_PORT, LED_PIN);
        md_delay_1ms(250);
    }
}

/************* (C) COPYRIGHT Eastsoft Microelectronics *****END OF FILE****/