/**********************************************************************************
 *
 * @file    irq.c
 * @brief   Interrupt handler
 *
 * @date    9 Feb. 2023
 * @author  AE Team
 * @note
 *          Change Logs:
 *          Date            Author          Notes
 *          9 Feb. 2023     Lisq            the first version
 *
 * Copyright (C) Shanghai Eastsoft Microelectronics Co. Ltd. All rights reserved.
 *
 * SPDX-License-Identifier: Apache-2.0
 *
 * Licensed under the Apache License, Version 2.0 (the License); you may
 * not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an AS IS BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 *
 **********************************************************************************
 */
/* Includes ------------------------------------------------------------------ */
#include "main.h"

/* Private Macros ------------------------------------------------------------ */

/* Private Variables --------------------------------------------------------- */

/* Public Variables ---------------------------------------------------------- */
uint8_t g_rx_buf[64];
uint8_t g_tx_buf = NEG_ACK;
uint16_t g_rx_len = sizeof(g_rx_buf);
uint16_t g_rx_i = 0U;
uint8_t g_flag = 0U;

uint16_t g_frame_timer;

/* Private Constants --------------------------------------------------------- */

/* Private function prototypes ----------------------------------------------- */

/* Private Function ---------------------------------------------------------- */

/** @addtogroup Projects_Examples_MD
  * @{
  */

/** @addtogroup Examples
  * @{
  */

/**
  * @brief  CLINT(Systick) IRQ handler
  * @retval None
  */
void __attribute__((interrupt)) CLINT_Handler(void)
{
    /* Nested IRQ start : Save CSR and enable global interrupt. */
    MD_NEST_INT_START();

    csi_coret_clr(md_cmu_get_clock() / 1000, CLINT_IRQn);
    md_inc_tick();

    /* the time interval of received adjacent bytes in the same frame is more than expected, reset rx FIFO */
    if (g_frame_timer != 0)
    {
        g_frame_timer--;

        if (g_frame_timer == 0)
            g_rx_i = 0;
    }

    /* Nested IRQ end : Restore CSR and disable global interrupt. */
    MD_NEST_INT_END();

    return;
}

/**
  * @brief  I2C IRQ handler
  * @retval None
  */
void __attribute__((interrupt)) I2C0_Handler(void)
{
    /* Nested IRQ start : Save CSR and enable global interrupt. */
    MD_NEST_INT_START();

    /* if tx data is empty, send data */
    if (md_i2c_mask_it_txe(COMMUNICATION_TYPE))
    {
        md_i2c_set_tx_reg_data(COMMUNICATION_TYPE, g_tx_buf);
        md_i2c_clear_flag_txe(COMMUNICATION_TYPE);

        if (g_tx_buf == POS_ACK)
            g_flag = 1;
        else
            g_flag = 0;
    }

    /* byte receive */
    if (md_i2c_mask_it_rxne(COMMUNICATION_TYPE))
    {
        g_rx_buf[g_rx_i++] = (uint8_t)md_i2c_get_rx_reg_data(COMMUNICATION_TYPE);

        /* assign timer to default when MCU receive each byte */
        g_frame_timer = FRAME_INTERVAL;

        md_i2c_clear_flag_rxne(COMMUNICATION_TYPE);

        if (g_rx_buf[0] != 0x3F)
            g_rx_i = 0;

        if (g_rx_i >= 8)
        {
            g_rx_i = 0;

            if (g_rx_buf[1] == 0xA1 && g_rx_buf[2] == 0x5E && g_rx_buf[3] == 0xFF
                    && g_rx_buf[4] == 0xFF && g_rx_buf[5] == 0xFF && g_rx_buf[6] == 0xFD
                    && g_rx_buf[7] == 0xFA)
                g_tx_buf = POS_ACK;
            else
                g_tx_buf = NEG_ACK;

            md_i2c_enable_it_nack(COMMUNICATION_TYPE);
            md_i2c_enable_it_txe(COMMUNICATION_TYPE);
        }

        if (g_rx_i >= g_rx_len)
            g_rx_i = 0;
    }

    /* if a matching address is received */
    if (md_i2c_mask_it_addr(COMMUNICATION_TYPE))
        md_i2c_clear_flag_addr(COMMUNICATION_TYPE);

    if (md_i2c_mask_it_stop(COMMUNICATION_TYPE))
    {
        md_i2c_clear_flag_stop(COMMUNICATION_TYPE);

        g_rx_i = 0;
    }

    if (md_i2c_mask_it_nack(COMMUNICATION_TYPE))
    {
        md_i2c_clear_flag_nack(COMMUNICATION_TYPE);

        md_i2c_disable_it_txe(COMMUNICATION_TYPE);

        g_rx_i = 0;
    }

    /* Nested IRQ end : Restore CSR and disable global interrupt. */
    MD_NEST_INT_END();

    return;
}

/**
  * @}
  */
/**
  * @}
  */

/************* (C) COPYRIGHT Eastsoft Microelectronics *****END OF FILE****/