/**********************************************************************************
 *
 * @file    main.c
 * @brief   main C file
 *
 * @date    17 Jan. 2023
 * @author  AE Team
 * @note
 *          Change Logs:
 *          Date            Author          Notes
 *          17 Jan. 2023    Shiwa           the first version
 *
 * Copyright (C) Shanghai Eastsoft Microelectronics Co. Ltd. All rights reserved.
 *
 * SPDX-License-Identifier: Apache-2.0
 *
 * Licensed under the Apache License, Version 2.0 (the License); you may
 * not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an AS IS BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 *
 **********************************************************************************
 */
 /* Includes ------------------------------------------------------------------ */
#include "main.h"
#include "printf.h"
/* Private Macros ------------------------------------------------------------ */
#define USE_BINARY_SEMAPHORE 0
/* Private Variables --------------------------------------------------------- */
TaskHandle_t g_task_app_handle = NULL;
TaskHandle_t g_task_1_handle = NULL;
TaskHandle_t g_task_2_handle = NULL;
TaskHandle_t g_task_3_handle = NULL;
uint32_t g_high_pri_cycle = 0U;
uint32_t g_medium_pri_cycle = 0U;
uint32_t g_low_pri_cycle = 0U;
SemaphoreHandle_t g_mutex_test;
/* Public Variables ---------------------------------------------------------- */
/* Private Constants --------------------------------------------------------- */
/* Private function prototypes ----------------------------------------------- */
/* Private Function ---------------------------------------------------------- */
void uart_pin_init(void);
void uart_init(void);
static void task_app(void *);
/*
 * Task1 -> Task3 = High priroty -> low priroty
 */
static void task_1(void *pvParameters);
static void task_2(void *pvParameters);
static void task_3(void *pvParameters);

/** @addtogroup Projects_Examples_MD
  * @{
  */

/** @addtogroup Examples
  * @{
  */

/** @addtogroup Projects_Examples_MD
  * @{
  */

/** @addtogroup Examples
  * @{
  */

/**
  * @brief  Test main function
  * @retval Status.
  */
int main(void)
{
    /* Configure system clock */
    md_cmu_pll_config(MD_CMU_PLL_INPUT_HOSC8M, MD_CMU_PLL_OUTPUT_72M);
    md_cmu_clock_config(MD_CMU_CLOCK_PLL, 72000000);
	
    /* Enable ALL peripheral */
    MD_SYSCFG_UNLOCK();
    md_cmu_enable_perh_all();
    MD_SYSCFG_LOCK();
	
	md_mcu_irq_config(CUART1_IRQn, 0, ENABLE);
	md_mcu_irq_config(MACHINE_MODE_SOFT_IRQn, 5, ENABLE);
	CLIC->CLICINT[MACHINE_MODE_SOFT_IRQn].ATTR |= (3); 
	__enable_irq();
	
    uart_pin_init();
    uart_init();
	
	/* Create AppTaskCreate
    * Arguments: TaskProc TaskName StackDepth Param Priority PtrTaskHandle */
    xTaskCreate(task_app, "AppTaskCreate", 128, NULL, 1, &g_task_app_handle);

    /* Start task schedule*/
	vTaskStartScheduler();
	
	/* Should nerver run to here */
    while (1);
}

/**
  * @brief  App task creator
  * @param  parameter: Parameter passed when created
  * @retval None
  */
static void task_app(void *param)
{
    BaseType_t retval = pdPASS;

    taskENTER_CRITICAL();
	
    /* Create mutx or binary semaphore */
#if !(USE_BINARY_SEMAPHORE)
    g_mutex_test = xSemaphoreCreateMutex();
#else
    g_mutex_test = xSemaphoreCreateBinary();
    xSemaphoreGive(g_mutex_test);
#endif /* Set to 1 to use mutex, 0 to use binary semaphore */

    if (g_mutex_test != NULL)
        printf("Create mutex OK\r\n");
    /* Create Task1 */
    retval = xTaskCreate(task_1, "Task_1", 128, NULL, 5, &g_task_1_handle);

    if (pdPASS == retval)
        printf("Task 1 Created\r\n");

    /* Create Task2 */
    retval = xTaskCreate(task_2, "Task_2", 128, NULL, 4, &g_task_2_handle);

    if (pdPASS == retval)
        printf("Task 2 Created\r\n");

    /* Create Task3 */
    retval = xTaskCreate(task_3, "Task_3", 128, NULL, 3, &g_task_3_handle);

    if (pdPASS == retval)
        printf("Task 3 Created\r\n");
    /* Delete self */
    vTaskDelete(g_task_app_handle);

    taskEXIT_CRITICAL();
}

/**
  * @brief  Task1
  * @param  param Parameter passed when created
  * @retval None
  */
static void task_1(void *param)
{
	while (1)
	{
		/* Get mutex */
        if (xSemaphoreTake(g_mutex_test, portMAX_DELAY) != pdTRUE)
        {
            printf("ERR:task_high_pri take failed\r\n");
        }

        printf("Mutex Take:  1\r\n");
        vTaskDelay(1000);

        if (xSemaphoreGive(g_mutex_test) != pdTRUE)
        {
            printf("ERR:task_high_pri give failed\r\n");
        }

        /* Keep count of the number of cycles this thread has performed */
        g_high_pri_cycle++;

        vTaskSuspend(NULL);
	}
}

/**
  * @brief  Task2
  * @param  param Parameter passed when created
  * @retval None
  */
static void task_2(void *param)
{
    while (1)
    {
		if (xSemaphoreTake(g_mutex_test, portMAX_DELAY) == pdTRUE)
        {

            printf("Mutex Take:    2\r\n");
            vTaskDelay(1000);

            if (eTaskGetState(g_task_1_handle) != eSuspended)
            {
                /* Did not expect to execute until the high priority thread was
                suspended.*/
                printf("ERR:task_high_pri should be suspended!\r\n");
            }
            else
            {
                /* Give the mutex back before suspending ourselves to allow
                the low priority thread to obtain the mutex */
                if (xSemaphoreGive(g_mutex_test) != pdTRUE)
                {
                    printf("ERR:task_medium_pri give failed");
                }

                vTaskSuspend(NULL);
            }
        }
        else
        {
            /* We should not leave the osMutexWait() function
            until the mutex was obtained.
            Toggle LED3 to indicate error */
            printf("ERR:task_medium_pri cant take failed\r\n");
        }

        /* The High and Medium priority threads should be in lock step */
        if (g_high_pri_cycle != (g_medium_pri_cycle + 1))
        {
            printf("Lock step failed\r\n");
        }

        /* Print a message to indicate medium task is running */
        printf("Task2 Run :    2\r\n");

        /* Keep count of the number of cycles this task has performed so a
        stall can be detected */
        g_medium_pri_cycle++;
        vTaskDelay(500);
    }
}

/**
  * @brief  Task3
  * @param  param Parameter passed when created
  * @retval None
  */
static void task_3(void *param)
{
    while (1)
    {
        /* Keep attempting to obtain the mutex.  We should only obtain it when
        the medium-priority thread has suspended itself, which in turn should only
        happen when the high-priority thread is also suspended */
        if (xSemaphoreTake(g_mutex_test, portMAX_DELAY) == pdTRUE)
        {
            printf("Mutex Take:      3\r\n");
            vTaskDelay(1000);

            /* Is the haigh and medium-priority threads suspended? */
            if ((eTaskGetState(g_task_1_handle) != eSuspended) || (eTaskGetState(g_task_2_handle) != eSuspended))
            {
                printf("ERR:task_high_pri and task_medium_pri should be suspended!\r\n");
            }
            else
            {
                /* Keep count of the number of cycles this task has performed
                so a stall can be detected */
                g_low_pri_cycle++;

                /* We can resume the other tasks here even though they have a
                higher priority than the this thread. When they execute they
                will attempt to obtain the mutex but fail because the low-priority
                thread is still the mutex holder.  this thread will then inherit
                the higher priority.  The medium-priority thread will block indefinitely
                when it attempts to obtain the mutex, the high-priority thread will only
                block for a fixed period and an error will be latched if the
                high-priority thread has not returned the mutex by the time this
                fixed period has expired */
                /* If binary semaphore is used, you will see "Task 2 run" before we give
                mutex to high priority task. Otherwise, Task 2 would not run before high
                priority task run*/
                vTaskResume(g_task_1_handle);
                vTaskResume(g_task_2_handle);

                /* The other two tasks should now have executed and no longer
                be suspended */
                if ((eTaskGetState(g_task_1_handle) == eSuspended) || (eTaskGetState(g_task_2_handle) == eSuspended))
                {
                    printf("ERR:task_high_pri and task_medium_pri should NOT be suspended!\r\n");
                }

                /* Give the mutex back before suspending ourselves to allow
                the low priority thread to obtain the mutex */
                if (xSemaphoreGive(g_mutex_test) != pdTRUE)
                {
                    printf("ERR:task_low_pri give failed");
                }
            }
        }
    }
}

/**
  * @brief  Init UART pin
  * @retval None
  */
void uart_pin_init(void)
{
    md_gpio_init_t gpio_init;
    md_gpio_init_struct(&gpio_init);

    /* Initialize tx pin */
    gpio_init.mode  = MD_GPIO_MODE_OUTPUT;
    gpio_init.odos  = MD_GPIO_PUSH_PULL;
    gpio_init.pupd  = MD_GPIO_PUSH_UP;
    gpio_init.odrv  = MD_GPIO_OUT_DRIVE_NORMAL;
    gpio_init.flt   = MD_GPIO_FILTER_DISABLE;
    gpio_init.type  = MD_GPIO_TYPE_CMOS;
    gpio_init.func  = MD_GPIO_FUNC_2;
    md_gpio_init(UART_TX_PORT, UART_TX_PIN, &gpio_init);

    /* Initialize rx pin */
    gpio_init.mode  = MD_GPIO_MODE_INPUT;
    gpio_init.odos  = MD_GPIO_PUSH_PULL;
    gpio_init.pupd  = MD_GPIO_PUSH_UP;
    gpio_init.odrv  = MD_GPIO_OUT_DRIVE_NORMAL;
    gpio_init.flt   = MD_GPIO_FILTER_DISABLE;
    gpio_init.type  = MD_GPIO_TYPE_CMOS;
    gpio_init.func  = MD_GPIO_FUNC_2;
    md_gpio_init(UART_RX_PORT, UART_RX_PIN, &gpio_init);
}
/**
  * @brief:  Initialize the uart.
  * @param:  None
  * @retval: None
  */
void uart_init(void)
{
    md_uart_init_t uart_init;
    md_uart_init_struct(&uart_init);

    /* Initialize UART */
    uart_init.baud        = 115200;  /* 冗余度: 110155 ~ 121452 */
    uart_init.word_length = MD_UART_WORD_LENGTH_8B;
    uart_init.stop_bits   = MD_UART_STOP_BITS_1;
    uart_init.parity      = MD_UART_PARITY_NONE;
    uart_init.fctl        = MD_UART_FLOW_CTL_DISABLE;
    uart_init.mode        = MD_UART_MODE;
    md_uart_init(UARTX, &uart_init);
}

int _write(int file, const void *ptr, int len)
{
    uint16_t cnt; 
	(void)file;
	uint8_t *ptr_char = (uint8_t *)ptr;
	
	for(int i = 0 ; i < len ; i++)
	{
		if (*ptr_char == '\n')
		{
            cnt = 4000;
			while ((READ_BIT(UARTX->STAT, UART_STAT_TFEMPTY_MSK)) != UART_STAT_TFEMPTY_MSK
			        && (--cnt))
			{
			}
			md_uart_set_send_data8(UARTX, '\r');
		}

        cnt = 4000;
        while ((READ_BIT(UARTX->STAT, UART_STAT_TFEMPTY_MSK)) != UART_STAT_TFEMPTY_MSK
		        && (--cnt))
		{
		}
		md_uart_set_send_data8(UARTX, *ptr_char++);
	}

	return len;
}
void _putchar(char chr)
{
	while (!md_uart_is_active_flag_tfempty(UARTX));
	md_uart_set_send_data8(UARTX, (uint8_t)chr);
}
/**
  * @}
  */
/**
  * @}
  */

/************* (C) COPYRIGHT Eastsoft Microelectronics *****END OF FILE****/
