/**********************************************************************************
 *
 * @file    .c
 * @brief
 *
 * @date
 * @author  AE Team
 * @note
 *          Change Logs:
 *          Date            Author          Notes
 *          2022.11.30      liuhy           the first version
 *
 * Copyright (C) Shanghai Eastsoft Microelectronics Co. Ltd. All rights reserved.
 *
 * SPDX-License-Identifier: Apache-2.0
 *
 * Licensed under the Apache License, Version 2.0 (the License); you may
 * not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an AS IS BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 *
 **********************************************************************************
 */

/* Includes-------------------------------------------------------------------*/
#include <sfud.h>
#include <stdarg.h>
#include <stdio.h>
#include <string.h>
#include <spi_flash.h>

/* Private Macros ------------------------------------------------------------ */
/* Private Variables --------------------------------------------------------- */
static char s_log_buf[256];

/* Public Variables ---------------------------------------------------------- */
uint32_t g_spi_userdata;

/* Private Constants --------------------------------------------------------- */
/* Private function prototypes ----------------------------------------------- */
void sfud_log_debug(const char *file, const long line, const char *format, ...);

/* Private Function ---------------------------------------------------------- */
static void spi_lock(const sfud_spi *spi)
{
    __disable_irq();
}

static void spi_unlock(const sfud_spi *spi)
{
    __enable_irq();
}

/**
 * SPI write data then read data
 */
static sfud_err spi_write_read(const sfud_spi *spi, const uint8_t *write_buf, size_t write_size, uint8_t *read_buf,
                               size_t read_size)
{
    sfud_err result = SFUD_SUCCESS;

    FLASH_CS_CLR();

    /* write data function */
    spi_flash_write(write_buf, write_size);

    /* read data function */
    spi_flash_read(read_buf, read_size);

    FLASH_CS_SET();

    return result;
}

/* about 100 microsecond delay */
static void retry_delay_100us(void)
{
    uint32_t delay = 120U;

    while (delay--);
}

sfud_err sfud_spi_port_init(sfud_flash *flash)
{
    sfud_err result = SFUD_SUCCESS;

    switch (flash->index)
    {
        case SFUD_MX25L64_DEVICE_INDEX:
        {

            init_mcu_spi();     /*Initialize the SPI module*/

            /* Synchronize interfaces and data required for Flash migration */
            flash->spi.wr = spi_write_read;
            flash->spi.lock = spi_lock;
            flash->spi.unlock = spi_unlock;
            flash->spi.user_data = &g_spi_userdata;
            /* about 100 microsecond delay */
            flash->retry.delay = retry_delay_100us;
            /* adout 60 seconds timeout */
            flash->retry.times = 60 * 10000;

            break;

        }

        default:
            break;
    }

    return result;
}

/**
 * This function is print debug info.
 *
 * @param file the file which has call this function
 * @param line the line number which has call this function
 * @param format output format
 * @param ... args
 */
void sfud_log_debug(const char *file, const long line, const char *format, ...)
{
    va_list args;

    /* args point to the first variable parameter */
    va_start(args, format);
    printf("[SFUD](%s:%ld) ", file, line);
    /* must use vprintf to print */
    vsnprintf(s_log_buf, sizeof(s_log_buf), format, args);
    printf("%s\r\n", s_log_buf);
    va_end(args);
}

/**
 * This function is print routine info.
 *
 * @param format output format
 * @param ... args
 */
void sfud_log_info(const char *format, ...)
{
    va_list args;

    /* args point to the first variable parameter */
    va_start(args, format);
    printf("[SFUD]");
    /* must use vprintf to print */
    vsnprintf(s_log_buf, sizeof(s_log_buf), format, args);
    printf("%s\r\n", s_log_buf);
    va_end(args);
}
