/**********************************************************************************
 *
 * @file    fft_test.c
 * @brief   Test FFT
 *
 * @date    17 Jan. 2023
 * @author  AE Team
 * @note
 *          Change Logs:
 *          Date            Author          Notes
 *          17 Jan. 2023    Shiwa           the first version
 *
 * Copyright (C) Shanghai Eastsoft Microelectronics Co. Ltd. All rights reserved.
 *
 * SPDX-License-Identifier: Apache-2.0
 *
 * Licensed under the Apache License, Version 2.0 (the License); you may
 * not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an AS IS BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 *
 **********************************************************************************
 */
/* Includes ------------------------------------------------------------------ */
#include "main.h"
#include "printf.h"
#include "test_timer.h"
#include "kiss_fft.h"
#include "es32_dsp.h"
/* Private Macros ------------------------------------------------------------ */
#define _STROF(x)   #x
#define _NAME(x) _STROF(x)
#define _CAT2(a,b) a##b
#define _CAT3(a,b,c) a##b##c
#define FFT_X(len) _CAT2(cfft_r4_,len)
#define TEST_COUNTS 1000
#define TEST_CYCLES(f) start_timer(&t);for (int i=0;i<TEST_COUNTS;i++){f;}end_timer(&t);
#define TEST_SHOW_DATA 0

/* Private Constants --------------------------------------------------------- */
const int16_t TEST_FFT_INPUT[]={
            0,   6825,  -3829,  13346,   2093,   7760,  12940,    254,
        15144,   1491,   4796,   7800,  -6737,   6737,  -7800,  -4796,
        -1491, -15144,   -254, -12940,  -7760,  -2093, -13346,   3829,
        -6825,      0,   6825,  -3829,  13346,   2093,   7760,  12940,
          254,  15144,   1491,   4796,   7800,  -6737,   6737,  -7800,
        -4796,  -1491, -15144,   -254, -12940,  -7760,  -2093, -13346,
         3829,  -6825,      0,   6825,  -3829,  13346,   2093,   7760,
        12940,    254,  15144,   1491,   4796,   7800,  -6737,   6737,
        -7800,  -4796,  -1491, -15144,   -254, -12940,  -7760,  -2093,
       -13346,   3829,  -6825,      0,   6825,  -3829,  13346,   2093,
         7760,  12940,    254,  15144,   1491,   4796,   7800,  -6737,
         6737,  -7800,  -4796,  -1491, -15144,   -254, -12940,  -7760,
        -2093, -13346,   3829,  -6825,      0,   6825,  -3829,  13346,
         2093,   7760,  12940,    254,  15144,   1491,   4796,   7800,
        -6737,   6737,  -7800,  -4796,  -1491, -15144,   -254, -12940,
        -7760,  -2093, -13346,   3829,  -6825,      0,   6825,  -3829,
    
            0,   6825,  -3829,  13346,   2093,   7760,  12940,    254,
        15144,   1491,   4796,   7800,  -6737,   6737,  -7800,  -4796,
        -1491, -15144,   -254, -12940,  -7760,  -2093, -13346,   3829,
        -6825,      0,   6825,  -3829,  13346,   2093,   7760,  12940,
          254,  15144,   1491,   4796,   7800,  -6737,   6737,  -7800,
        -4796,  -1491, -15144,   -254, -12940,  -7760,  -2093, -13346,
         3829,  -6825,      0,   6825,  -3829,  13346,   2093,   7760,
        12940,    254,  15144,   1491,   4796,   7800,  -6737,   6737,
        -7800,  -4796,  -1491, -15144,   -254, -12940,  -7760,  -2093,
       -13346,   3829,  -6825,      0,   6825,  -3829,  13346,   2093,
         7760,  12940,    254,  15144,   1491,   4796,   7800,  -6737,
         6737,  -7800,  -4796,  -1491, -15144,   -254, -12940,  -7760,
        -2093, -13346,   3829,  -6825,      0,   6825,  -3829,  13346,
         2093,   7760,  12940,    254,  15144,   1491,   4796,   7800,
        -6737,   6737,  -7800,  -4796,  -1491, -15144,   -254, -12940,
        -7760,  -2093, -13346,   3829,  -6825,      0,   6825,  -3829
}; 

/* Private function prototypes ----------------------------------------------- */

/* Private Variables --------------------------------------------------------- */
#define TEST_LEN 1024
#define FFT_BUF_LEN (TEST_LEN*4+512)
static int16_t s_fft_input[TEST_LEN*2];
static int16_t s_fft_output[TEST_LEN*2+2];
static int8_t s_fft_buf[FFT_BUF_LEN];
static size_t s_fft_buf_len=FFT_BUF_LEN;
/* Public Variables ---------------------------------------------------------- */

/* Private Function ---------------------------------------------------------- */
void fft_test(void)
{
    test_timer_t t;
    
    /*
    FFT in ASM
    */
    for (int i=0;i<TEST_LEN;i++)
    {
        s_fft_input[i*2]=TEST_FFT_INPUT[i%256];
        s_fft_input[i*2+1]=0;
    }
    TEST_CYCLES(FFT_X(TEST_LEN)(s_fft_output,s_fft_input,TEST_LEN));
    printf("FFT" _NAME(TEST_LEN) "(ASM):%uus(%u/s)\r\n",get_timer_us(&t)/TEST_COUNTS,1000000*TEST_COUNTS/get_timer_us(&t));
    #if TEST_SHOW_DATA
    for (int i=0;i<TEST_LEN*2+2;i++)
    {
        printf("[%d]=%d\r\n",i,s_fft_output[i]);
    }
    #endif  /* TEST_SHOW_DATA */
		
    /*
    FFT in C (kiss_fft)
    */
    for (int i=0;i<TEST_LEN;i++)
    {
        s_fft_input[i*2]=TEST_FFT_INPUT[i%256];
        s_fft_input[i*2+1]=0;
    }
    kiss_fft_cfg fcfg=kiss_fft_alloc(TEST_LEN,0,s_fft_buf,&s_fft_buf_len);
    TEST_CYCLES(kiss_fft(fcfg,(kiss_fft_cpx*)s_fft_input,(kiss_fft_cpx*)s_fft_output));
    printf("FFT" _NAME(TEST_LEN) "(KISS):%uus(%u/s)\r\n",get_timer_us(&t)/TEST_COUNTS,1000000*TEST_COUNTS/get_timer_us(&t));
    #if TEST_SHOW_DATA
    for (int i=0;i<TEST_LEN*2+2;i++)
    {
        printf("[%d]=%d\r\n",i,s_fft_output[i]);
    }
    #endif  /* TEST_SHOW_DATA */
}

#undef TEST_LEN

#define TEST_LEN 256
void fft_test2(void)
{
    test_timer_t t;
    
    /*
    FFT in ASM
    */
    for (int i=0;i<TEST_LEN;i++)
    {
        s_fft_input[i*2]=TEST_FFT_INPUT[i%256];
        s_fft_input[i*2+1]=0;
    }
    TEST_CYCLES(FFT_X(TEST_LEN)(s_fft_output,s_fft_input,TEST_LEN));
    printf("FFT" _NAME(TEST_LEN) "(ASM):%uus(%u/s)\r\n",get_timer_us(&t)/TEST_COUNTS,1000000*TEST_COUNTS/get_timer_us(&t));
    #if TEST_SHOW_DATA
    for (int i=0;i<TEST_LEN*2+2;i++)
    {
        printf("[%d]=%d\r\n",i,s_fft_output[i]);
    }
    #endif  /* TEST_SHOW_DATA */
		
    /*
    FFT in C (kiss_fft)
    */
    for (int i=0;i<TEST_LEN;i++)
    {
        s_fft_input[i*2]=TEST_FFT_INPUT[i%256];
        s_fft_input[i*2+1]=0;
    }
    s_fft_buf_len=FFT_BUF_LEN;
    kiss_fft_cfg fcfg=kiss_fft_alloc(TEST_LEN,0,s_fft_buf,&s_fft_buf_len);
    TEST_CYCLES(kiss_fft(fcfg,(kiss_fft_cpx*)s_fft_input,(kiss_fft_cpx*)s_fft_output));
    printf("FFT" _NAME(TEST_LEN) "(KISS):%uus(%u/s)\r\n",get_timer_us(&t)/TEST_COUNTS,1000000*TEST_COUNTS/get_timer_us(&t));
    #if TEST_SHOW_DATA
    for (int i=0;i<TEST_LEN*2+2;i++)
    {
        printf("[%d]=%d\r\n",i,s_fft_output[i]);
    }
    #endif  /* TEST_SHOW_DATA */
}

#undef TEST_LEN
#define TEST_LEN 64
void fft_test3(void)
{
    test_timer_t t;
    
    /*
    FFT in ASM
    */
    for (int i=0;i<TEST_LEN;i++)
    {
        s_fft_input[i*2]=TEST_FFT_INPUT[i%256];
        s_fft_input[i*2+1]=0;
    }
    TEST_CYCLES(FFT_X(TEST_LEN)(s_fft_output,s_fft_input,TEST_LEN));
    printf("FFT" _NAME(TEST_LEN) "(ASM):%uus(%u/s)\r\n",get_timer_us(&t)/TEST_COUNTS,1000000*TEST_COUNTS/get_timer_us(&t));
    #if TEST_SHOW_DATA
    for (int i=0;i<TEST_LEN*2+2;i++)
    {
        printf("[%d]=%d\r\n",i,s_fft_output[i]);
    }
    #endif /* TEST_SHOW_DATA */
	
    /*
    FFT in C (kiss_fft)
    */
    for (int i=0;i<TEST_LEN;i++)
    {
        s_fft_input[i*2]=TEST_FFT_INPUT[i%256];
        s_fft_input[i*2+1]=0;
    }
    s_fft_buf_len=FFT_BUF_LEN;
    kiss_fft_cfg fcfg=kiss_fft_alloc(TEST_LEN,0,s_fft_buf,&s_fft_buf_len);
    TEST_CYCLES(kiss_fft(fcfg,(kiss_fft_cpx*)s_fft_input,(kiss_fft_cpx*)s_fft_output));
    printf("FFT" _NAME(TEST_LEN) "(KISS):%uus(%u/s)\r\n",get_timer_us(&t)/TEST_COUNTS,1000000*TEST_COUNTS/get_timer_us(&t));
    #if TEST_SHOW_DATA
    for (int i=0;i<TEST_LEN*2+2;i++)
    {
        printf("[%d]=%d\r\n",i,s_fft_output[i]);
    }
    #endif /* TEST_SHOW_DATA */
}
