/**********************************************************************************
 *
 * @file    .c
 * @brief   Source file
 *
 * @author  AE Team
 * @note
 *          Change Logs:
 *          Date               Author          Notes
 *          2023-01-17         liuhy           the first version
 *
 * Copyright (C) Shanghai Eastsoft Microelectronics Co. Ltd. All rights reserved.
 *
 * SPDX-License-Identifier: Apache-2.0
 *
 * Licensed under the Apache License, Version 2.0 (the License); you may
 * not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an AS IS BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 *
 **********************************************************************************
 */

/* Includes ------------------------------------------------------------------ */
#include "main.h"
#include "usbd_core.h"
#include "usbd_hid.h"

/* Private Macros ------------------------------------------------------------ */
/*!< custom hid report descriptor size */
#define HID_CUSTOM_REPORT_DESC_SIZE (74)
#define SUPPORT_USB_REMOTE_WAKEUP   (1)

/* Private Variables --------------------------------------------------------- */
/*!< custom hid report descriptor */
static const uint8_t s_hid_custom_report_desc[HID_CUSTOM_REPORT_DESC_SIZE] =
{
    0x05, 0x01, /* USAGE_PAGE (Generic Desktop) */
    0x09, 0x02, /* USAGE (Mouse) */
    0xA1, 0x01, /* COLLECTION (Application) */
    0x09, 0x01, /*   USAGE (Pointer) */

    0xA1, 0x00, /*   COLLECTION (Physical) */
    0x05, 0x09, /*     USAGE_PAGE (Button) */
    0x19, 0x01, /*     USAGE_MINIMUM (Button 1) */
    0x29, 0x03, /*     USAGE_MAXIMUM (Button 3) */

    0x15, 0x00, /*     LOGICAL_MINIMUM (0) */
    0x25, 0x01, /*     LOGICAL_MAXIMUM (1) */
    0x95, 0x03, /*     REPORT_COUNT (3) */
    0x75, 0x01, /*     REPORT_SIZE (1) */

    0x81, 0x02, /*     INPUT (Data,Var,Abs) */
    0x95, 0x01, /*     REPORT_COUNT (1) */
    0x75, 0x05, /*     REPORT_SIZE (5) */
    0x81, 0x01, /*     INPUT (Cnst,Var,Abs) */

    0x05, 0x01, /*     USAGE_PAGE (Generic Desktop) */
    0x09, 0x30, /*     USAGE (X) */
    0x09, 0x31, /*     USAGE (Y) */
    0x09, 0x38,

    0x15, 0x81, /*     LOGICAL_MINIMUM (-127) */
    0x25, 0x7F, /*     LOGICAL_MAXIMUM (127) */
    0x75, 0x08, /*     REPORT_SIZE (8) */
    0x95, 0x03, /*     REPORT_COUNT (2) */

    0x81, 0x06, /*     INPUT (Data,Var,Rel) */
    0xC0, 0x09,
    0x3c, 0x05,
    0xff, 0x09,

    0x01, 0x15,
    0x00, 0x25,
    0x01, 0x75,
    0x01, 0x95,

    0x02, 0xb1,
    0x22, 0x75,
    0x06, 0x95,
    0x01, 0xb1,

    0x01, 0xc0 /*   END_COLLECTION */
};

static struct usbd_interface s_hid_custom_intf[1];
static volatile uint8_t s_configed_flag = 0U;

/* Public Variables ---------------------------------------------------------- */
uint8_t g_usb_hid_idle = 0;
uint8_t g_usb_hid_protocol = 0;
#if SUPPORT_USB_REMOTE_WAKEUP
uint8_t g_usb_remote_wakeup_flag = 0;
#endif /*SUPPORT_USB_REMOTE_WAKEUP*/
/* Private Constants --------------------------------------------------------- */
/* Private function prototypes ----------------------------------------------- */
/* Private Function ---------------------------------------------------------- */

/*******************************************************START OF ChryUSB_Configurator Generation*************/
#ifndef WBVAL
    #define WBVAL(x) (unsigned char)((x) & 0xFF), (unsigned char)(((x) >> 8) & 0xFF)
#endif/*WBVAL*/

/*!< USBD CONFIG */
#define USBD_VERSION 0x0200
#define USBD_PRODUCT_VERSION 0x0001
#define USBD_VID 0x30cc
#define USBD_PID 0xae02
#define USBD_MAX_POWER 0xfa
#define USBD_LANGID_STRING 1033
#define USBD_CONFIG_DESCRIPTOR_SIZE 34

/*!< USBD ENDPOINT CONFIG */

#define USBD_IF0_AL0_EP1_ADDR 0x81
#define USBD_IF0_AL0_EP1_SIZE 0x4
#define USBD_IF0_AL0_EP1_INTERVAL 0x01

/*!< USBD HID CONFIG */
#define USBD_HID_VERSION 0x0111
#define USBD_HID_COUNTRY_CODE 0
#define USBD_IF0_AL0_HID_REPORT_DESC_SIZE 2

/*!< USBD Descriptor */
const unsigned char usbd_descriptor[] =
{
    /********************************************** Device Descriptor */
    0x12,                                       /*!< bLength */
    0x01,                                       /*!< bDescriptorType */
    WBVAL(USBD_VERSION),                        /*!< bcdUSB */
    0x00,                                       /*!< bDeviceClass */
    0x00,                                       /*!< bDeviceSubClass */
    0x00,                                       /*!< bDeviceProtocol */
    0x40,                                       /*!< bMaxPacketSize */
    WBVAL(USBD_VID),                            /*!< idVendor */
    WBVAL(USBD_PID),                            /*!< idProduct */
    WBVAL(USBD_PRODUCT_VERSION),                /*!< bcdDevice */
    0x01,                                       /*!< iManufacturer */
    0x02,                                       /*!< iProduct */
    0x03,                                       /*!< iSerial */
    0x01,                                       /*!< bNumConfigurations */
    /********************************************** Config Descriptor */
    0x09,                                       /*!< bLength */
    0x02,                                       /*!< bDescriptorType */
    WBVAL(USBD_CONFIG_DESCRIPTOR_SIZE),         /*!< wTotalLength */
    0x01,                                       /*!< bNumInterfaces */
    0x01,                                       /*!< bConfigurationValue */
    0x00,                                       /*!< iConfiguration */
#if SUPPORT_USB_REMOTE_WAKEUP
    0xA0,                                       /*!< bmAttributes */
#else
    0x80,                                       /*!< bmAttributes */
#endif /*SUPPORT_USB_REMOTE_WAKEUP*/
    USBD_MAX_POWER,                             /*!< bMaxPower */
    /********************************************** Interface 0 Alternate 0 Descriptor: Descriptor of Joystick Mouse interface */
    0x09,                                       /*!< bLength */
    0x04,                                       /*!< bDescriptorType */
    0x00,                                       /* bInterfaceNumber: Number of Interface */
    0x00,                                       /* bAlternateSetting: Alternate setting */
    0x01,                                       /* bNumEndpoints */
    0x03,                                       /* bInterfaceClass: HID */
    0x01,                                       /* bInterfaceSubClass : 1=BOOT, 0=no boot */
    0x02,                          /* nInterfaceProtocol : 0=none, 1=keyboard, 2=mouse */
    0x00,                                       /* iInterface: Index of string descriptor */
    /********************************************** Class Specific Descriptor of HID */
    0x09,                                       /*!< bLength */
    0x21,                                       /*!< bDescriptorType */
    WBVAL(USBD_HID_VERSION),                    /*!< bcdHID */
    USBD_HID_COUNTRY_CODE,                      /*!< bCountryCode */
    0x01,                                       /*!< bNumDescriptors */
    0x22,                                       /*!< bDescriptorType */
    WBVAL(HID_CUSTOM_REPORT_DESC_SIZE),   /*!< wItemLength */
    /********************************************** Endpoint 1 Descriptor */
    0x07,                                       /*!< bLength */
    0x05,                                       /*!< bDescriptorType */
    USBD_IF0_AL0_EP1_ADDR,                      /*!< bEndpointAddress */
    0x03,                                       /*!< bmAttributes */
    WBVAL(USBD_IF0_AL0_EP1_SIZE),               /*!< wMaxPacketSize */
    USBD_IF0_AL0_EP1_INTERVAL,                  /*!< bInterval */
    /********************************************** Language ID String Descriptor */
    0x04,                                       /*!< bLength */
    0x03,                                       /*!< bDescriptorType */
    WBVAL(USBD_LANGID_STRING),                  /*!< wLangID0 */
    /********************************************** String 1 Descriptor */
    /* Eastsoft */
    0x12,                                       /*!< bLength */
    0x03,                                       /*!< bDescriptorType */
    0x45, 0x00,                                 /*!< 'E' wcChar0 */
    0x61, 0x00,                                 /*!< 'a' wcChar1 */
    0x73, 0x00,                                 /*!< 's' wcChar2 */
    0x74, 0x00,                                 /*!< 't' wcChar3 */
    0x73, 0x00,                                 /*!< 's' wcChar4 */
    0x6f, 0x00,                                 /*!< 'o' wcChar5 */
    0x66, 0x00,                                 /*!< 'f' wcChar6 */
    0x74, 0x00,                                 /*!< 't' wcChar7 */
    /********************************************** String 2 Descriptor */
    /* Eastsoft HID Keyboard */
    0x2C,                                       /*!< bLength */
    0x03,                                       /*!< bDescriptorType */
    0x45, 0x00,                                 /*!< 'E' wcChar0 */
    0x61, 0x00,                                 /*!< 'a' wcChar1 */
    0x73, 0x00,                                 /*!< 's' wcChar2 */
    0x74, 0x00,                                 /*!< 't' wcChar3 */
    0x73, 0x00,                                 /*!< 's' wcChar4 */
    0x6f, 0x00,                                 /*!< 'o' wcChar5 */
    0x66, 0x00,                                 /*!< 'f' wcChar6 */
    0x74, 0x00,                                 /*!< 't' wcChar7 */
    0x20, 0x00,                                 /*!< ' ' wcChar8 */
    0x48, 0x00,                                 /*!< 'H' wcChar9 */
    0x49, 0x00,                                 /*!< 'I' wcChar10 */
    0x44, 0x00,                                 /*!< 'D' wcChar11 */
    0x20, 0x00,                                 /*!< ' ' wcChar12 */
    0x4B, 0x00,                                 /*!< 'K' wcChar13 */
    0x65, 0x00,                                 /*!< 'e' wcChar14 */
    0x79, 0x00,                                 /*!< 'y' wcChar15 */
    0x62, 0x00,                                 /*!< 'b' wcChar16 */
    0x6f, 0x00,                                 /*!< 'o' wcChar17 */
    0x61, 0x00,                                 /*!< 'a' wcChar18 */
    0x72, 0x00,                                 /*!< 'r' wcChar19 */
    0x64, 0x00,                                 /*!< 'd' wcChar20 */
    /********************************************** String 3 Descriptor */
    /* 202210201641 */
    0x1a,                                       /*!< bLength */
    0x03,                                       /*!< bDescriptorType */
    0x32, 0x00,                                 /*!< '2' wcChar0 */
    0x30, 0x00,                                 /*!< '0' wcChar1 */
    0x32, 0x00,                                 /*!< '2' wcChar2 */
    0x32, 0x00,                                 /*!< '2' wcChar3 */
    0x31, 0x00,                                 /*!< '1' wcChar4 */
    0x30, 0x00,                                 /*!< '0' wcChar5 */
    0x32, 0x00,                                 /*!< '2' wcChar6 */
    0x30, 0x00,                                 /*!< '0' wcChar7 */
    0x31, 0x00,                                 /*!< '1' wcChar8 */
    0x36, 0x00,                                 /*!< '6' wcChar9 */
    0x34, 0x00,                                 /*!< '4' wcChar10 */
    0x31, 0x00,                                 /*!< '1' wcChar11 */
    /********************************************** Device Qualifier Descriptor */
    0x0a,                                       /*!< bLength */
    0x06,                                       /*!< bDescriptorType */
    WBVAL(USBD_VERSION),                        /*!< bcdUSB */
    0x00,                                       /*!< bDeviceClass */
    0x00,                                       /*!< bDeviceSubClass */
    0x00,                                       /*!< bDeviceProtocol */
    0x40,                                       /*!< bMaxPacketSize0 */
    0x01,                                       /*!< bNumConfigurations */
    0x00,                                       /*!< bReserved */
    0x00
};

/*!< USBD HID REPORT 0 Descriptor */
const unsigned char usbd_hid_0_report_descriptor[USBD_IF0_AL0_HID_REPORT_DESC_SIZE] =
{
    0x00,
    0xC0    /* END_COLLECTION */
};
/*******************************************************END OF ChryUSB_Configurator Generation*************/

/*!< endpoint call back */

static void usbd_hid_custom_in_callback(uint8_t busid, uint8_t ep, uint32_t nbytes)
{

}

static struct usbd_endpoint s_custom_in_ep =
{
    .ep_cb = usbd_hid_custom_in_callback,
    .ep_addr = USBD_IF0_AL0_EP1_ADDR
};

void usbd_event_handler(uint8_t busid, uint8_t event)
{
    switch (event)
    {
        case USBD_EVENT_RESET:
            break;

        case USBD_EVENT_CONNECTED:
            break;

        case USBD_EVENT_DISCONNECTED:
            break;

        case USBD_EVENT_RESUME:
            break;

        case USBD_EVENT_SUSPEND:
            break;

        case USBD_EVENT_CONFIGURED:
        {
            s_configed_flag = 1;
            break;
        }

        case USBD_EVENT_SET_REMOTE_WAKEUP:
#if SUPPORT_USB_REMOTE_WAKEUP
            g_usb_remote_wakeup_flag = 1;
#endif /*SUPPORT_USB_REMOTE_WAKEUP*/
            break;

        case USBD_EVENT_CLR_REMOTE_WAKEUP:
#if SUPPORT_USB_REMOTE_WAKEUP
            g_usb_remote_wakeup_flag = 0;
#endif /*SUPPORT_USB_REMOTE_WAKEUP*/
            break;

        default:
            break;
    }
}

/**
  * @brief            cdc hid winusb init
  * @pre              none
  * @param[in]        none
  * @retval           none
  */
void cdc_acm_hid_winusb_descriptor_init(void)
{
    usbd_desc_register(0, usbd_descriptor);

    usbd_add_interface(0, usbd_hid_init_intf(0, s_hid_custom_intf, s_hid_custom_report_desc, HID_CUSTOM_REPORT_DESC_SIZE));
    usbd_add_endpoint(0, &s_custom_in_ep);

    s_configed_flag = 0;
    usbd_initialize(0, 0, usbd_event_handler);

    while (!s_configed_flag);
}

/*One frame of mouse data is 4 bytes:
 * byte0 = Whether the modifier key is pressed
 *      byte0[bit0]: Left click press
 *      byte0[bit1]: Right click press
 *      byte0[bit2]: mid click press
 * byte1 = X coordinate change
 * byte2 = Y coordinate change
 * byte3 = Roller variation . 0x80 is stillness*/
static const uint8_t s_test_send_buf_none[] =
{
    0x00, 0x00, 0x00, 0x00,                 /*null*/
};

static const uint8_t s_test_send_buf_y_increase[] =
{
    0x00, 0x00, 0x02, 0x00,                 /*Y coordinate change: +2*/
};

static const uint8_t s_test_send_buf_y_reduce[] =
{
    0x00, 0x00, ~0x02, 0x00,                /*Y coordinate change: -2*/
};

static const uint8_t s_test_send_buf_x_increase[] =
{
    0x00, 0x02, 0x00, 0x00,                 /*X coordinate change: +2*/
};

static const uint8_t s_test_send_buf_x_reduce[] =
{
    0x00, ~0x02, 0x00, 0x00,                /*X coordinate change: -2*/
};

static const uint8_t s_test_send_buf_roll_increase[] =
{
    0x00, 0x00, 0x00, 0x01,                 /*Roller variation +1*/
};

static const uint8_t s_test_send_buf_roll_reduce[] =
{
    0x00, 0x00, 0x00, ~0x01,                /*Roller variation -1*/
};

static const uint8_t s_test_send_buf_left_click[] =
{
    (1U << 0), 0x00, 0x00, 0x00,            /*Left click press*/
};

static const uint8_t s_test_send_buf_right_click[] =
{
    (1U << 1), 0x00, 0x00, 0x00,            /*Right click press*/
};

void usb_dev_loop(void)
{
    static uint8_t test_mode = 0U;
    static uint8_t key_state_last = 0U;
    uint8_t key_state = 0U;

#if SUPPORT_USB_REMOTE_WAKEUP
	USB->POWER |= 0x1;
	while(((USB->POWER) & 0x3) == 0x3)
	{
		if (!(md_gpio_get_input_data(KEY1_PORT, KEY1_PIN)))
		{
			if(g_usb_remote_wakeup_flag)
			{                                                   
				USB->POWER |= ALD_USB_POWER_RESUME;
				   
				md_delay_1ms(11);
						
				USB->POWER &= (~(ALD_USB_POWER_RESUME));
				
				printf("wake up host\r\n");
			}
		}
		
		md_delay_1ms(50);
	}
#endif /*SUPPORT_USB_REMOTE_WAKEUP*/

    /*board mid key: change mode*/
    if (!(md_gpio_get_input_data(KEY1_PORT, KEY1_PIN)))
    {
        key_state |= KEY1_FLAG;
    }
    else
    {
        if (key_state_last & KEY1_FLAG)
        {
            test_mode = !test_mode;
        }
    }

    /*board up key: mouse move up(mode 0),wheel up(mode1)*/
    if (!(md_gpio_get_input_data(KEY2_PORT, KEY2_PIN)))
    {
        key_state |= KEY2_FLAG;

        if (test_mode)
            usbd_ep_start_write(0, s_custom_in_ep.ep_addr, s_test_send_buf_roll_increase, sizeof(s_test_send_buf_roll_increase));
        else
            usbd_ep_start_write(0, s_custom_in_ep.ep_addr, s_test_send_buf_y_reduce, sizeof(s_test_send_buf_y_reduce));
    }
    else
    {
        if (key_state_last & KEY2_FLAG)
        {
            usbd_ep_start_write(0, s_custom_in_ep.ep_addr, s_test_send_buf_none, sizeof(s_test_send_buf_none));
        }
    }

    /*board down key: mouse move down(mode 0),wheel down(mode1)*/
    if (!(md_gpio_get_input_data(KEY3_PORT, KEY3_PIN)))
    {
        key_state |= KEY3_FLAG;

        if (test_mode)
            usbd_ep_start_write(0, s_custom_in_ep.ep_addr, s_test_send_buf_roll_reduce, sizeof(s_test_send_buf_roll_reduce));
        else
            usbd_ep_start_write(0, s_custom_in_ep.ep_addr, s_test_send_buf_y_increase, sizeof(s_test_send_buf_y_increase));
    }
    else
    {
        if (key_state_last & KEY3_FLAG)
        {
            usbd_ep_start_write(0, s_custom_in_ep.ep_addr, s_test_send_buf_none, sizeof(s_test_send_buf_none));
        }
    }

    /*board left key: mouse move left (mode 0),left click press(mode1)*/
    if (!(md_gpio_get_input_data(KEY4_PORT, KEY4_PIN)))
    {
        key_state |= KEY4_FLAG;

        if (test_mode)
        {
            if (!(key_state_last & KEY4_FLAG))
            {
                usbd_ep_start_write(0, s_custom_in_ep.ep_addr, s_test_send_buf_left_click, sizeof(s_test_send_buf_left_click));
            }
        }
        else
            usbd_ep_start_write(0, s_custom_in_ep.ep_addr, s_test_send_buf_x_reduce, sizeof(s_test_send_buf_x_reduce));
    }
    else
    {
        if (key_state_last & KEY4_FLAG)
        {
            usbd_ep_start_write(0, s_custom_in_ep.ep_addr, s_test_send_buf_none, sizeof(s_test_send_buf_none));
        }
    }

    /*board right key: mouse move right (mode 0),right click press(mode1)*/
    if (!(md_gpio_get_input_data(KEY5_PORT, KEY5_PIN)))
    {
        key_state |= KEY5_FLAG;

        if (test_mode)
        {
            if (!(key_state_last & KEY5_FLAG))
            {
                usbd_ep_start_write(0, s_custom_in_ep.ep_addr, s_test_send_buf_right_click, sizeof(s_test_send_buf_right_click));
            }
        }
        else
            usbd_ep_start_write(0, s_custom_in_ep.ep_addr, s_test_send_buf_x_increase, sizeof(s_test_send_buf_x_increase));
    }
    else
    {
        if (key_state_last & KEY5_FLAG)
        {
            usbd_ep_start_write(0, s_custom_in_ep.ep_addr, s_test_send_buf_none, sizeof(s_test_send_buf_none));
        }
    }

    key_state_last = key_state;

    md_delay_1ms(10);
}

void usbd_hid_set_idle(uint8_t busid, uint8_t intf, uint8_t report_id, uint8_t duration)
{
    g_usb_hid_idle = duration;
}

uint8_t usbd_hid_get_idle(uint8_t busid, uint8_t intf, uint8_t report_id)
{
    return g_usb_hid_idle;
}

void usbd_hid_set_protocol(uint8_t busid, uint8_t intf, uint8_t protocol)
{
    g_usb_hid_protocol = protocol;
}

uint8_t usbd_hid_get_protocol(uint8_t busid, uint8_t intf)
{
	return g_usb_hid_protocol;
}

/*******************************************************END OF FILE*************/
