/**********************************************************************************
 *
 * @file    .c
 * @brief   Source file
 *
 * @author  AE Team
 * @note
 *          Change Logs:
 *          Date               Author          Notes
 *          2023-01-17         liuhy           the first version
 *
 * Copyright (C) Shanghai Eastsoft Microelectronics Co. Ltd. All rights reserved.
 *
 * SPDX-License-Identifier: Apache-2.0
 *
 * Licensed under the Apache License, Version 2.0 (the License); you may
 * not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an AS IS BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 *
 **********************************************************************************
 */

/* Includes ------------------------------------------------------------------ */
#include "usbd_core.h"

/* Private Macros ------------------------------------------------------------ */

#define PRINTER_SUBCLASS 0x01U

#define PRINTER_REQUEST_GET_DEVICE_ID   0x00U
#define PRINTER_REQUEST_GET_PORT_SATTUS 0x01U
#define PRINTER_REQUEST_SOFT_RESET      0x02U

#define PRINTER_STATUS_NO_ERROR    0x00U
#define PRINTER_STATUS_SELECTED    0x08U
#define PRINTER_STATUS_PAPER_EMPTY 0x10U

/*!< endpoint address */
#define PRINTER_IN_EP          0x81
#define PRINTER_IN_EP_SIZE     0x40
#define PRINTER_OUT_EP         0x02
#define PRINTER_OUT_EP_SIZE    0x40

#define USBD_VID            0x30cc
#define USBD_PID            0xae07
#define USBD_MAX_POWER      100
#define USBD_LANGID_STRING  0x0409

/*!< config descriptor size */
#define USB_CONFIG_SIZE (32)

/* Private Variables --------------------------------------------------------- */
/*!< global descriptor */
static const uint8_t printer_descriptor[] =
{
    USB_DEVICE_DESCRIPTOR_INIT(USB_2_0, 0x00, 0x00, 0x00, USBD_VID, USBD_PID, 0x0000, 0x01),
    USB_CONFIG_DESCRIPTOR_INIT(USB_CONFIG_SIZE, 0x01, 0x01, USB_CONFIG_SELF_POWERED, USBD_MAX_POWER),
    USB_INTERFACE_DESCRIPTOR_INIT(0x00, 0x00, 0x02, 0x07, 0x01, 0x02, 0x00),
    USB_ENDPOINT_DESCRIPTOR_INIT(PRINTER_IN_EP, 0x02, PRINTER_IN_EP_SIZE, 0x00),
    USB_ENDPOINT_DESCRIPTOR_INIT(PRINTER_OUT_EP, 0x02, PRINTER_OUT_EP_SIZE, 0x00),
    /**************************************/
    /* string0 descriptor*/
    /**************************************/
    USB_LANGID_INIT(USBD_LANGID_STRING),
    /**************************************/
    /* string1 descriptor*/
    /**************************************/
    0x14,                       /* bLength */
    USB_DESCRIPTOR_TYPE_STRING, /* bDescriptorType */
    'C', 0x00,                  /* wcChar0 */
    'h', 0x00,                  /* wcChar1 */
    'e', 0x00,                  /* wcChar2 */
    'r', 0x00,                  /* wcChar3 */
    'r', 0x00,                  /* wcChar4 */
    'y', 0x00,                  /* wcChar5 */
    'U', 0x00,                  /* wcChar6 */
    'S', 0x00,                  /* wcChar7 */
    'B', 0x00,                  /* wcChar8 */
    /**************************************/
    /* string2 descriptor*/
    /**************************************/
    0x2A,                       /* bLength */
    USB_DESCRIPTOR_TYPE_STRING, /* bDescriptorType */
    'C', 0x00,                  /* wcChar0 */
    'h', 0x00,                  /* wcChar1 */
    'e', 0x00,                  /* wcChar2 */
    'r', 0x00,                  /* wcChar3 */
    'r', 0x00,                  /* wcChar4 */
    'y', 0x00,                  /* wcChar5 */
    'U', 0x00,                  /* wcChar6 */
    'S', 0x00,                  /* wcChar7 */
    'B', 0x00,                  /* wcChar8 */
    ' ', 0x00,                  /* wcChar9 */
    'P', 0x00,                  /* wcChar10 */
    'R', 0x00,                  /* wcChar11 */
    'I', 0x00,                  /* wcChar12 */
    'N', 0x00,                  /* wcChar13 */
    'T', 0x00,                  /* wcChar14 */
    ' ', 0x00,                  /* wcChar15 */
    'D', 0x00,                  /* wcChar16 */
    'E', 0x00,                  /* wcChar17 */
    'M', 0x00,                  /* wcChar18 */
    'O', 0x00,                  /* wcChar19 */
    /**************************************/
    /* string3 descriptor*/
    /**************************************/
    0x16,                       /* bLength */
    USB_DESCRIPTOR_TYPE_STRING, /* bDescriptorType */
    '2', 0x00,                  /* wcChar0 */
    '0', 0x00,                  /* wcChar1 */
    '2', 0x00,                  /* wcChar2 */
    '2', 0x00,                  /* wcChar3 */
    '1', 0x00,                  /* wcChar4 */
    '2', 0x00,                  /* wcChar5 */
    '3', 0x00,                  /* wcChar6 */
    '4', 0x00,                  /* wcChar7 */
    '5', 0x00,                  /* wcChar8 */
    '6', 0x00,                  /* wcChar9 */
#ifdef CONFIG_USB_HS
    /**************************************/
    /* device qualifier descriptor*/
    /**************************************/
    0x0a,
    USB_DESCRIPTOR_TYPE_DEVICE_QUALIFIER,
    0x00,
    0x02,
    0x02,
    0x02,
    0x01,
    0x40,
    0x01,
    0x00,
#endif/*CONFIG_USB_HS*/
    0x00
};

USB_NOCACHE_RAM_SECTION USB_MEM_ALIGNX uint8_t g_read_buffer[PRINTER_OUT_EP_SIZE];
USB_NOCACHE_RAM_SECTION USB_MEM_ALIGNX uint8_t g_write_buffer[PRINTER_IN_EP_SIZE];

/* Public Variables ---------------------------------------------------------- */
/* Private Constants --------------------------------------------------------- */
/* Private function prototypes ----------------------------------------------- */
/* Private Function ---------------------------------------------------------- */

void usbd_event_handler(uint8_t busid, uint8_t event)
{
    switch (event)
    {
        case USBD_EVENT_RESET:
            break;

        case USBD_EVENT_CONNECTED:
            break;

        case USBD_EVENT_DISCONNECTED:
            break;

        case USBD_EVENT_RESUME:
            break;

        case USBD_EVENT_SUSPEND:
            break;

        case USBD_EVENT_CONFIGURED:
            /* setup first out ep read transfer */
            usbd_ep_start_read(0, PRINTER_OUT_EP, g_read_buffer, PRINTER_OUT_EP_SIZE);
            break;

        case USBD_EVENT_SET_REMOTE_WAKEUP:
            break;

        case USBD_EVENT_CLR_REMOTE_WAKEUP:
            break;

        default:
            break;
    }
}

void usbd_printer_bulk_out(uint8_t busid, uint8_t ep, uint32_t nbytes)
{
#if 1
    printf("actual out len:%d\r\n", (int)nbytes);

    for (int i = 0; i < nbytes; i++)
    {
        printf("%02x ", g_read_buffer[i]);
    }

    printf("\r\n");
#else

    for (int i = 0; i < nbytes; i++)
    {
        printf("%c", g_read_buffer[i]);
    }

    printf("\r\n");
#endif/*1*/

    /* setup next out ep read transfer */
    usbd_ep_start_read(0, PRINTER_OUT_EP, g_read_buffer, PRINTER_OUT_EP_SIZE);
}

void usbd_printer_bulk_in(uint8_t busid, uint8_t ep, uint32_t nbytes)
{
    printf("actual in len:%d\r\n", (int)nbytes);

    if ((nbytes % PRINTER_IN_EP_SIZE) == 0 && nbytes)
    {
        /* send zlp */
        usbd_ep_start_write(0, PRINTER_IN_EP, NULL, 0);
    }
    else
    {
    }
}

/*!< endpoint call back */
struct usbd_endpoint printer_out_ep =
{
    .ep_addr = PRINTER_OUT_EP,
    .ep_cb = usbd_printer_bulk_out
};

struct usbd_endpoint printer_in_ep =
{
    .ep_addr = PRINTER_IN_EP,
    .ep_cb = usbd_printer_bulk_in
};

struct usbd_interface g_intf0;

static const uint8_t printer_device_id[] =
{
    0x00, 49,
    'M', 'F', 'G', ':', 'E', 'S', '_', ';',
    'C', 'M', 'D', ':', 'G', 'D', 'I', ';',
    'M', 'D', 'L', ':', 'T', 'E', 'S', 'T', '1', ';',
    'C', 'L', 'S', ':', 'P', 'R', 'I', 'N', 'T', 'E', 'R', ';',
    'M', 'O', 'D', 'E', ':', 'G', 'D', 'I', ';'
};

struct usbd_printer_priv {
    const uint8_t *device_id;
    uint8_t device_id_len;
    uint8_t port_status;
} g_usbd_printer;

static int printer_class_interface_request_handler(uint8_t busid, struct usb_setup_packet *setup, uint8_t **data, uint32_t *len)
{
    USB_LOG_DBG("Printer Class request: "
                "bRequest 0x%02x\r\n",
                setup->bRequest);

    switch (setup->bRequest) {
        case PRINTER_REQUEST_GET_DEVICE_ID:
            memcpy(*data, g_usbd_printer.device_id, g_usbd_printer.device_id_len);
            *len = g_usbd_printer.device_id_len;
            break;
        case PRINTER_REQUEST_GET_PORT_SATTUS:

            break;
        case PRINTER_REQUEST_SOFT_RESET:

            break;
        default:
            USB_LOG_WRN("Unhandled Printer Class bRequest 0x%02x\r\n", setup->bRequest);
            return -1;
    }

    return 0;
}

static void printer_notify_handler(uint8_t busid, uint8_t event, void *arg)
{
    switch (event) {
        case USBD_EVENT_RESET:
            break;

        default:
            break;
    }
}

struct usbd_interface *usbd_printer_init_intf(struct usbd_interface *intf, const uint8_t *device_id, uint8_t device_id_len)
{
    intf->class_interface_handler = printer_class_interface_request_handler;
    intf->class_endpoint_handler = NULL;
    intf->vendor_handler = NULL;
    intf->notify_handler = printer_notify_handler;

    g_usbd_printer.device_id = device_id;
    g_usbd_printer.device_id_len = device_id_len;
    return intf;
}

void printer_init(void)
{
    usbd_desc_register(0, printer_descriptor);
    usbd_add_interface(0, usbd_printer_init_intf(&g_intf0, printer_device_id, sizeof(printer_device_id)));
    usbd_add_endpoint(0, &printer_out_ep);
    usbd_add_endpoint(0, &printer_in_ep);

    usbd_initialize(0, 0, usbd_event_handler);
}

/*******************************************************END OF FILE*************/
