/**********************************************************************************
 *
 * @file    spi_flash.c
 * @brief   SPI Flash file for DEMO
 *
 * @date    9 Jan. 2023
 * @author  AE Team
 * @note
 *          Change Logs:
 *          Date            Author          Notes
 *          9 Jan. 2023     Lisq            the first version
 *
 * Copyright (C) Shanghai Eastsoft Microelectronics Co. Ltd. All rights reserved.
 *
 * SPDX-License-Identifier: Apache-2.0
 *
 * Licensed under the Apache License, Version 2.0 (the License); you may
 * not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an AS IS BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 *
 **********************************************************************************
 */

/* Includes ------------------------------------------------------------------ */

#include "spi_flash.h"

/* Private Macros ------------------------------------------------------------ */
#define FLASH_CS_SET()  md_gpio_write_pin(GPIOA, MD_GPIO_PIN_4, 1);
#define FLASH_CS_CLR()  md_gpio_write_pin(GPIOA, MD_GPIO_PIN_4, 0);

/* Private Variables --------------------------------------------------------- */

static md_spi_init_t s_spi;

/* Public Variables ---------------------------------------------------------- */

/* Private Constants --------------------------------------------------------- */

/* Private function prototypes ----------------------------------------------- */

/* Private Function ---------------------------------------------------------- */

/**
  * @brief  wait until flash unbusy.
  * @retval Status, see @ref md_status_t.
  */
md_status_t flash_wait_unbusy(void)
{
    uint8_t status;
    int r_flag = 0;

    FLASH_CS_CLR(); /* Choose lower, the selected Flash */

    if (md_spi_send_byte_fast(&s_spi, (uint8_t)FLASH_STATUS) != MD_OK)   /*Send to read status command*/
    {
        FLASH_CS_SET();     /* Pick up and release of Flash */
        return MD_ERROR;
    }

    do
    {
        status = md_spi_recv_byte_fast(&s_spi, &r_flag);

        if (r_flag != MD_OK)
        {
            FLASH_CS_SET();
            return MD_ERROR;
        }
    }
    while (status & 0x01);

    FLASH_CS_SET();

    return MD_OK;
}

/**
  * @brief  Read flash id in blocking mode.
  * @retval flash id.
  */
uint32_t flash_read_id(void)
{
    uint8_t i;
    int r_flag = 0;
    uint8_t flash_id[4] = {0U};

    flash_id[0] = FLASH_ID;

    FLASH_CS_CLR(); /* Choose lower, the selected Flash */

    for (i = 0; i < sizeof(flash_id); i++)
    {
        if (md_spi_send_byte_fast(&s_spi, flash_id[i]) != MD_OK)
        {
            FLASH_CS_SET();     /* Pick up and release of Flash */
            return MD_ERROR;
        }
    }

    for (i = 0; i < 3; i++)
    {
        flash_id[i] = md_spi_recv_byte_fast(&s_spi, &r_flag);

        if (r_flag != MD_OK)
        {
            FLASH_CS_SET();
            return MD_ERROR;
        }
    }

    FLASH_CS_SET();

    return ((flash_id[0] << 16) | (flash_id[1] << 8) | (flash_id[2]));  /*Manufacturer ID flash_id [0] and device ID flash_id [1]*/
}

/**
  * @brief  Initializate spi flash pin
  * @retval None.
  */
static void spi_pin_init(void)
{
    md_gpio_init_t l_gpio;

    l_gpio.type  = MD_GPIO_TYPE_CMOS;
    l_gpio.odos  = MD_GPIO_PUSH_PULL;
    l_gpio.pupd  = MD_GPIO_PUSH_UP;
    l_gpio.odrv  = MD_GPIO_OUT_DRIVE_NORMAL;
    l_gpio.flt   = MD_GPIO_FILTER_DISABLE;

    l_gpio.func  = MD_GPIO_FUNC_1;
    l_gpio.mode  = MD_GPIO_MODE_OUTPUT;
    md_gpio_init(GPIOA, MD_GPIO_PIN_4, &l_gpio);    /* Initialize PA4 for selected pin */
    FLASH_CS_SET();   /* Choose the output high, release of Flash */

    l_gpio.func  = MD_GPIO_FUNC_2;
    l_gpio.mode  = MD_GPIO_MODE_OUTPUT;
    md_gpio_init(GPIOA, MD_GPIO_PIN_5, &l_gpio);    /* Initialize PA5 for clock output pin */

    l_gpio.func  = MD_GPIO_FUNC_2;
    l_gpio.mode  = MD_GPIO_MODE_OUTPUT;
    md_gpio_init(GPIOA, MD_GPIO_PIN_7, &l_gpio);    /* Initialize PA7 for MOSI pin */

    l_gpio.func  = MD_GPIO_FUNC_2;
    l_gpio.mode  = MD_GPIO_MODE_INPUT;
    md_gpio_init(GPIOA, MD_GPIO_PIN_6, &l_gpio);    /* Initialization for PA6 MISO pins */
}

/**
  * @brief spi function
  * @retval None.
  */
static void mcu_spi_init(void)
{
    spi_pin_init();

    md_spi_struct_init(&s_spi);
    s_spi.SPIx      = SPI0;                 /* Using SPI0 */
    s_spi.mode      = MD_SPI_MODE_MASTER;   /* SPI host mode */
    s_spi.baud      = MD_SPI_BAUD_2;      /* clock */
    s_spi.data_size = MD_SPI_DATA_SIZE_8;   /* 8 bit pattern */
    s_spi.polarity  = MD_SPI_CPOL_HIGH;     /* Free high level */
    s_spi.phase     = MD_SPI_CPHA_SECOND;   /* The second edge receiving data */
    s_spi.first_bit = MD_SPI_FIRSTBIT_MSB;  /* Send the MSB first */
    s_spi.dir       = MD_SPI_DIRECTION_2LINES;
    s_spi.ss_en     = DISABLE;
    s_spi.crc_calc  = DISABLE;
    s_spi.crc_poly  = 0;
    md_spi_init(&s_spi);                    /* According to the parameter initialization SPI peripherals */
}

/**
  * @brief  flash write enable function.
  * @retval None.
  */
static uint8_t flash_write_enable(void)
{
    /* wait until flash unbusy */
    flash_wait_unbusy();

    /* enable the selected Flash */
    FLASH_CS_CLR();

    /* send write enable command */
    if (md_spi_send_byte_fast(&s_spi, FLASH_WRITE_ENABLE) != MD_OK)
    {
        FLASH_CS_SET();
        return 1;
    }

    /* release Flash */
    FLASH_CS_SET();

    /* wait until flash unbusy */
    flash_wait_unbusy();

    return 0;
}

/**
  * @brief  low-level flash sector erase function.
  * @param  start_sec: Specified start sector to be erase.
  * @param  end_sec: Specified end sector to be erase.
  * @retval Status.
  */
uint8_t ll_flash_sector_erase(uint32_t start_sec, uint32_t end_sec)
{
    uint8_t  cmd_buf[4];
    uint8_t  i = 0U;
    uint32_t sec_addr;

    /*Flash sector erase command*/
    cmd_buf[0] = FLASH_SEC_ERASE;

    /*Erase Flash sectors */
    for (; start_sec <= end_sec; start_sec++)
    {
        /* group command and address */
        sec_addr = start_sec * SEC_SIZE;
        cmd_buf[1] = (sec_addr >> 16) & 0xFF;   /*24 bit Flash address*/
        cmd_buf[2] = (sec_addr >> 8) & 0xFF;
        cmd_buf[3] = (sec_addr) & 0xFF;

        /* enable flash write */
        flash_write_enable();

        /* erase current sector */
        /* enable the selected Flash */
        FLASH_CS_CLR();

        /*Send the sector erase instructions and Flash address three bytes*/
        for (i = 0; i < sizeof(cmd_buf); i++)
        {
            if (md_spi_send_byte_fast(&s_spi, cmd_buf[i]) != MD_OK)
            {
                FLASH_CS_SET();
                return 1;
            }
        }

        /* release Flash */
        FLASH_CS_SET();

        /* wait until flash not in write progress */
        flash_wait_unbusy();
    }

    return 0;
}

/**
  * @brief  low-level flash sector write operation function.
  * @param  src_buf: pointer to the expected written data.
  * @param  sec: start sector to write.
  * @param  sec_num: amount of sector.
  * @retval Status, see @ref ald_status_t.
  */
uint8_t ll_flash_sector_write(const uint8_t *src_buf, uint32_t sec, uint32_t sec_num)
{
    uint8_t  cmd_buf[4];
    uint32_t  i, j, k;
    uint32_t page_addr, sec_addr;

    if (src_buf == NULL)
        return 1;

    /*Flash page program command*/
    cmd_buf[0] = FLASH_PAGE_PROGRAM;

    /*Erase Flash sectors */
    for (i = 0; i < sec_num; i++)
    {
        /* calculate current sector start address */
        sec_addr = sec * SEC_SIZE;

        for (j = 0; j < PAGE_PER_SEC; j++)
        {
            /* group command and address */
            page_addr = sec_addr + PAGE_SIZE * j;
            cmd_buf[1] = (page_addr >> 16) & 0xff;
            cmd_buf[2] = (page_addr >> 8) & 0xff;
            cmd_buf[3] = (page_addr) & 0xff;

            /* enable flash write */
            flash_write_enable();

            /* write current page */
            /* enable the selected Flash */
            FLASH_CS_CLR();

            /*Send the sector erase instructions and Flash address three bytes*/
            for (k = 0; k < sizeof(cmd_buf); k++)
            {
                if (md_spi_send_byte_fast(&s_spi, cmd_buf[k]) != MD_OK)
                {
                    FLASH_CS_SET();
                    return 1;
                }
            }

            /* write the data to the Flash*/
            for (k = 0; k < PAGE_SIZE; k++)
            {
                if (md_spi_send_byte_fast(&s_spi, src_buf[i * SEC_SIZE + j * PAGE_SIZE + k]) != MD_OK)
                {
                    FLASH_CS_SET();
                    return 1;
                }
            }

            /* release Flash */
            FLASH_CS_SET();

            /* wait until flash not in write progress */
            flash_wait_unbusy();
        }
    }

    return 0;
}

/**
  * @brief  low-level flash sector read operation function.
  * @param  str_buf: pointer to the first item of the byte array to store read data.
  * @param  sec: start sector.
  * @param  sec_num: amount of sector to be read.
  * @retval Status, see @ref ald_status_t.
  */
uint8_t ll_flash_sector_read(uint8_t *str_buf, uint32_t sec, uint32_t sec_num)
{

    uint8_t  cmd_buf[4];
    uint32_t  i, j;
    uint32_t sec_addr;
    int      r_flag;

    if (str_buf == NULL)
        return 1;

    /*Flash read command*/
    cmd_buf[0] = FLASH_READ;
    /* group command and address */
    sec_addr = sec * SEC_SIZE;
    cmd_buf[1] = (sec_addr >> 16) & 0xFF;   /*24 bit Flash address*/
    cmd_buf[2] = (sec_addr >> 8) & 0xFF;
    cmd_buf[3] = (sec_addr) & 0xFF;

    /* enable the selected Flash */
    FLASH_CS_CLR();

    /* Send the read instructions and Flash address */
    for (i = 0; i < sizeof(cmd_buf); i++)
    {
        if (md_spi_send_byte_fast(&s_spi, cmd_buf[i]) != MD_OK)
        {
            FLASH_CS_SET();
            return 1;
        }
    }

    /* read sectors data */
    for (i = 0; i < sec_num; i++)
    {
        /* read data cycle */
        for (j = 0; j < SEC_SIZE; j++)
        {
            str_buf[i * SEC_SIZE + j] = md_spi_recv_byte_fast(&s_spi, &r_flag);

            if (r_flag != MD_OK)
            {
                FLASH_CS_SET();
                return 1;
            }
        }
    }

    /* release Flash */
    FLASH_CS_SET();

    return 0;
}

/**
  * @brief spi flash init function
  * @retval None.
  */
uint8_t ll_spi_flash_init(void)
{
    mcu_spi_init();

    return 0;
}
/**
  * @}
  */
/**
  * @}
  */

/************* (C) COPYRIGHT Eastsoft Microelectronics *****END OF FILE****/
