/**********************************************************************************
 *
 * @file    .c
 * @brief   Source file
 *
 * @author  AE Team
 * @note
 *          Change Logs:
 *          Date               Author          Notes
 *          2023-01-17         liuhy           the first version
 *
 * Copyright (C) Shanghai Eastsoft Microelectronics Co. Ltd. All rights reserved.
 *
 * SPDX-License-Identifier: Apache-2.0
 *
 * Licensed under the Apache License, Version 2.0 (the License); you may
 * not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an AS IS BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 *
 **********************************************************************************
 */

/* Includes ------------------------------------------------------------------ */
#include "main.h"
#include "usbd_core.h"
#include "usbd_cdc.h"
#include "usbd_hid.h"

/* Private Macros ------------------------------------------------------------ */
/*!< custom hid report descriptor size */
#define HID_CUSTOM_REPORT_DESC_SIZE (34)
#define ES_USB_DEV_EP_READ_BUFFER_SIZE (64)
#define ES_USB_DEV_EP_WIRTE_BUFFER_NUM (4)
#define ES_USB_SEND_0_PACKEGE_AS_END   1
#define ES_USB_DEV_EP_TX_FLAG_IDLE    (0U)
#define ES_USB_DEV_EP_TX_FLAG_BUSY    (1U)
#define ES_USB_DEV_EP_TX_FLAG_END     (2U)
#define WCID_VENDOR_CODE              0x01

/* Private Variables --------------------------------------------------------- */
/*!< custom hid report descriptor */
static const uint8_t s_hid_custom_report_desc[HID_CUSTOM_REPORT_DESC_SIZE] =
{
    /* USER CODE BEGIN 0 */
    0x06, 0x00, 0xff, /*USAGE_PAGE (Vendor Defined Page 1)*/
    0x09, 0x01,       /*USAGE (Vendor Usage 1)*/
    0xa1, 0x01,       /*COLLECTION (Application)*/
    0x09, 0x01,       /*USAGE (Vendor Usage 1)*/
    0x15, 0x00,       /*LOGICAL_MINIMUM (0)*/
    0x26, 0xff, 0x00, /*LOGICAL_MAXIMUM (255)*/
    0x95, 0x40,       /*REPORT_COUNT (64)*/
    0x75, 0x08,       /*REPORT_SIZE (8)*/
    0x81, 0x02,       /*INPUT (Data,Var,Abs)*/
    /* <___________________________________________________> */
    0x09, 0x01,       /*USAGE (Vendor Usage 1)*/
    0x15, 0x00,       /*LOGICAL_MINIMUM (0)*/
    0x26, 0xff, 0x00, /*LOGICAL_MAXIMUM (255)*/
    0x95, 0x40,       /*REPORT_COUNT (64)*/
    0x75, 0x08,       /*REPORT_SIZE (8)*/
    0x91, 0x02,       /*OUTPUT (Data,Var,Abs)*/
    /* USER CODE END 0 */
    0xC0 /*     END_COLLECTION               */
};

static __ALIGN_BEGIN const uint8_t s_WCID_StringDescriptor_MSOS[18] __ALIGN_END =
{
    /* MS OS string descriptor */
    0x12,                       /* bLength */
    USB_DESCRIPTOR_TYPE_STRING, /* bDescriptorType */
    /* MSFT100 */
    'M', 0x00, 'S', 0x00, 'F', 0x00, 'T', 0x00, /* wcChar_7 */
    '1', 0x00, '0', 0x00, '0', 0x00,            /* wcChar_7 */
    WCID_VENDOR_CODE,                           /* bVendorCode */
    0x00,                                       /* bReserved */
};

static __ALIGN_BEGIN const uint8_t s_WINUSB_WCIDDescriptor[40] __ALIGN_END =
{
    /* WCID descriptor */
    0x28, 0x00, 0x00, 0x00,                   /* dwLength */
    0x00, 0x01,                               /* bcdVersion */
    0x04, 0x00,                               /* wIndex */
    0x01,                                     /* bCount */
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* bReserved_7 */

    /* WCID function descriptor */
    0x03, /* bFirstInterfaceNumber */
    0x01, /* bReserved */
    /* WINUSB */
    'W', 'I', 'N', 'U', 'S', 'B', 0x00, 0x00, /* cCID_8 */
    /*  */
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* cSubCID_8 */
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00,             /* bReserved_6 */
};

static __ALIGN_BEGIN const uint8_t s_WINUSB_IF0_WCIDProperties[142] __ALIGN_END =
{
    /* WCID property descriptor */
    0x8e, 0x00, 0x00, 0x00, /* dwLength */
    0x00, 0x01,             /* bcdVersion */
    0x05, 0x00,             /* wIndex */
    0x01, 0x00,             /* wCount */

    /* registry propter descriptor */
    0x84, 0x00, 0x00, 0x00, /* dwSize */
    0x01, 0x00, 0x00, 0x00, /* dwPropertyDataType */
    0x28, 0x00,             /* wPropertyNameLength */
    /* DeviceInterfaceGUID */
    'D', 0x00, 'e', 0x00, 'v', 0x00, 'i', 0x00,  /* wcName_20 */
    'c', 0x00, 'e', 0x00, 'I', 0x00, 'n', 0x00,  /* wcName_20 */
    't', 0x00, 'e', 0x00, 'r', 0x00, 'f', 0x00,  /* wcName_20 */
    'a', 0x00, 'c', 0x00, 'e', 0x00, 'G', 0x00,  /* wcName_20 */
    'U', 0x00, 'I', 0x00, 'D', 0x00, 0x00, 0x00, /* wcName_20 */
    0x4e, 0x00, 0x00, 0x00,                      /* dwPropertyDataLength */
    /* {CDB3B5AD-293B-4663-AA36-1AAE46463776} */
    '{', 0x00, 'C', 0x00, 'D', 0x00, 'B', 0x00, /* wcData_39 */
    '3', 0x00, 'B', 0x00, '5', 0x00, 'A', 0x00, /* wcData_39 */
    'D', 0x00, '-', 0x00, '2', 0x00, '9', 0x00, /* wcData_39 */
    '3', 0x00, 'B', 0x00, '-', 0x00, '4', 0x00, /* wcData_39 */
    '6', 0x00, '6', 0x00, '3', 0x00, '-', 0x00, /* wcData_39 */
    'A', 0x00, 'A', 0x00, '3', 0x00, '6', 0x00, /* wcData_39 */
    '-', 0x00, '1', 0x00, 'A', 0x00, 'A', 0x00, /* wcData_39 */
    'E', 0x00, '4', 0x00, '6', 0x00, '4', 0x00, /* wcData_39 */
    '6', 0x00, '3', 0x00, '7', 0x00, '7', 0x00, /* wcData_39 */
    '6', 0x00, '}', 0x00, 0x00, 0x00,           /* wcData_39 */
};

const uint8_t *s_WINUSB_IFx_WCIDProperties[] =
{
    0,
    0,
    0,
    s_WINUSB_IF0_WCIDProperties,     /*s_WINUSB_WCIDDescriptor: bFirstInterfaceNumber[1] = 0x3 */
};

static struct usb_msosv1_descriptor s_msosv1_desc =
{
    .string = s_WCID_StringDescriptor_MSOS,
    .vendor_code = WCID_VENDOR_CODE,
    .compat_id = s_WINUSB_WCIDDescriptor,
    .comp_id_property = s_WINUSB_IFx_WCIDProperties,
};

static struct usbd_interface s_cdc_ctl_intf[1];
static struct usbd_interface s_cdc_data_intf[1];
static struct usbd_interface s_winusb_intf[1];
static struct usbd_interface s_hid_custom_intf[1];
static volatile uint8_t s_configed_flag = 0U;
static volatile uint32_t s_read_num_cdc[ES_USB_DEV_EP_WIRTE_BUFFER_NUM];
static volatile uint32_t s_read_num_hid[ES_USB_DEV_EP_WIRTE_BUFFER_NUM];
static volatile uint32_t s_read_num_winusb[ES_USB_DEV_EP_WIRTE_BUFFER_NUM];
static volatile uint8_t s_cdc_ep_tx_flag = ES_USB_DEV_EP_TX_FLAG_IDLE;
static volatile uint8_t s_hid_ep_tx_flag = ES_USB_DEV_EP_TX_FLAG_IDLE;
static volatile uint8_t s_winusb_ep_tx_flag = ES_USB_DEV_EP_TX_FLAG_IDLE;
static volatile uint8_t s_cdc_ep_buf_fill_index = 0U;
static volatile uint8_t s_hid_ep_buf_fill_index = 0U;
static volatile uint8_t s_winusb_ep_buf_fill_index = 0U;
static volatile uint8_t s_cdc_ep_buf_tx_index = 0U;
static volatile uint8_t s_hid_ep_buf_tx_index = 0U;
static volatile uint8_t s_winusb_ep_buf_tx_index = 0U;
static volatile uint8_t s_cdc_ep_buf_fill_num = 0U;
static volatile uint8_t s_hid_ep_buf_fill_num = 0U;
static volatile uint8_t s_winusb_ep_rx_buf_fill_num = 0U;
static volatile struct cdc_line_coding s_cdc_line_code = {0};
static USB_NOCACHE_RAM_SECTION USB_MEM_ALIGNX uint8_t s_read_buffer_cdc[ES_USB_DEV_EP_READ_BUFFER_SIZE];
static USB_NOCACHE_RAM_SECTION USB_MEM_ALIGNX uint8_t s_read_buffer_hid[ES_USB_DEV_EP_READ_BUFFER_SIZE];
static USB_NOCACHE_RAM_SECTION USB_MEM_ALIGNX uint8_t s_read_buffer_winusb[ES_USB_DEV_EP_READ_BUFFER_SIZE];
static USB_NOCACHE_RAM_SECTION USB_MEM_ALIGNX uint8_t s_write_buffer_cdc[ES_USB_DEV_EP_WIRTE_BUFFER_NUM][ES_USB_DEV_EP_READ_BUFFER_SIZE];
static USB_NOCACHE_RAM_SECTION USB_MEM_ALIGNX uint8_t s_write_buffer_hid[ES_USB_DEV_EP_WIRTE_BUFFER_NUM][ES_USB_DEV_EP_READ_BUFFER_SIZE];
static USB_NOCACHE_RAM_SECTION USB_MEM_ALIGNX uint8_t s_write_buffer_winusb[ES_USB_DEV_EP_WIRTE_BUFFER_NUM][ES_USB_DEV_EP_READ_BUFFER_SIZE];

/* Public Variables ---------------------------------------------------------- */
uint8_t g_usb_hid_idle = 0;
uint8_t g_usb_hid_protocol = 0;
/* Private Constants --------------------------------------------------------- */

/*******************************************************START OF ChryUSB_Configurator Generation*************/

#ifndef WBVAL
    #define WBVAL(x) (unsigned char)((x) & 0xFF), (unsigned char)(((x) >> 8) & 0xFF)
#endif/*WBVAL*/

/*!< USBD CONFIG */
#define USBD_VERSION 0x0200
#define USBD_PRODUCT_VERSION 0x0001
#define USBD_VID 0x30cc
#define USBD_PID 0xae04
#define USBD_MAX_POWER 0xfa
#define USBD_LANGID_STRING 1033
#define USBD_CONFIG_DESCRIPTOR_SIZE 130

/*!< USBD ENDPOINT CONFIG */

#define USBD_IF0_AL0_EP0_ADDR 0x81
#define USBD_IF0_AL0_EP0_SIZE 0x08
#define USBD_IF0_AL0_EP0_INTERVAL 0x0a

#define USBD_IF1_AL0_EP0_ADDR 0x02
#define USBD_IF1_AL0_EP0_SIZE 0x40
#define USBD_IF1_AL0_EP0_INTERVAL 0x00

#define USBD_IF1_AL0_EP1_ADDR 0x82
#define USBD_IF1_AL0_EP1_SIZE 0x40
#define USBD_IF1_AL0_EP1_INTERVAL 0x00

#define USBD_IF2_AL0_EP0_ADDR 0x03
#define USBD_IF2_AL0_EP0_SIZE 0x40
#define USBD_IF2_AL0_EP0_INTERVAL 0x01

#define USBD_IF2_AL0_EP1_ADDR 0x83
#define USBD_IF2_AL0_EP1_SIZE 0x40
#define USBD_IF2_AL0_EP1_INTERVAL 0x01

#define USBD_IF3_AL0_EP0_ADDR 0x04
#define USBD_IF3_AL0_EP0_SIZE 0x40
#define USBD_IF3_AL0_EP0_INTERVAL 0x00

#define USBD_IF3_AL0_EP1_ADDR 0x84
#define USBD_IF3_AL0_EP1_SIZE 0x40
#define USBD_IF3_AL0_EP1_INTERVAL 0x00

/*!< USBD HID CONFIG */
#define USBD_HID_VERSION 0x0111
#define USBD_HID_COUNTRY_CODE 0
#define USBD_IF2_AL0_HID_REPORT_DESC_SIZE 2

/*!< USBD Descriptor */
const unsigned char usbd_descriptor[] =
{
    /********************************************** Device Descriptor */
    0x12,                                       /*!< bLength */
    0x01,                                       /*!< bDescriptorType */
    WBVAL(USBD_VERSION),                        /*!< bcdUSB */
    0xef,                                       /*!< bDeviceClass */
    0x02,                                       /*!< bDeviceSubClass */
    0x01,                                       /*!< bDeviceProtocol */
    0x40,                                       /*!< bMaxPacketSize */
    WBVAL(USBD_VID),                            /*!< idVendor */
    WBVAL(USBD_PID),                            /*!< idProduct */
    WBVAL(USBD_PRODUCT_VERSION),                /*!< bcdDevice */
    0x01,                                       /*!< iManufacturer */
    0x02,                                       /*!< iProduct */
    0x03,                                       /*!< iSerial */
    0x01,                                       /*!< bNumConfigurations */
    /********************************************** Config Descriptor */
    0x09,                                       /*!< bLength */
    0x02,                                       /*!< bDescriptorType */
    WBVAL(USBD_CONFIG_DESCRIPTOR_SIZE),         /*!< wTotalLength */
    0x04,                                       /*!< bNumInterfaces */
    0x01,                                       /*!< bConfigurationValue */
    0x00,                                       /*!< iConfiguration */
    0x80,                                       /*!< bmAttributes */
    USBD_MAX_POWER,                             /*!< bMaxPower */
    /********************************************** Interface Associate Descriptor */
    0x08,                                       /*!< bLength */
    0x0b,                                       /*!< bDescriptorType */
    0x00,                                       /*!< bFirstInterface */
    0x02,                                       /*!< bInterfaceCount */
    0x02,                                       /*!< bFunctionClass */
    0x02,                                       /*!< bFunctionSubClass */
    0x01,                                       /*!< bFunctionProtocol */
    0x00,                                       /*!< iFunction */
    /********************************************** Interface 0 Alternate 0 Descriptor */
    0x09,                                       /*!< bLength */
    0x04,                                       /*!< bDescriptorType */
    0x00,                                       /*!< bInterfaceNumber */
    0x00,                                       /*!< bAlternateSetting */
    0x01,                                       /*!< bNumEndpoints */
    0x02,                                       /*!< bInterfaceClass */
    0x02,                                       /*!< bInterfaceSubClass */
    0x01,                                       /*!< bInterfaceProtocol */
    0x04,                                       /*!< iInterface */
    /********************************************** Class Specific Descriptor of CDC ACM Control */
    0x05,                                       /*!< bLength */
    0x24,                                       /*!< bDescriptorType */
    0x00,                                       /*!< bDescriptorSubtype */
    WBVAL(0x0110),                              /*!< bcdCDC */
    0x05,                                       /*!< bLength */
    0x24,                                       /*!< bDescriptorType */
    0x01,                                       /*!< bDescriptorSubtype */
    0x00,                                       /*!< bmCapabilities */
    0x01,                                       /*!< bDataInterface */
    0x04,                                       /*!< bLength */
    0x24,                                       /*!< bDescriptorType */
    0x02,                                       /*!< bDescriptorSubtype */
    0x02,                                       /*!< bmCapabilities */
    0x05,                                       /*!< bLength */
    0x24,                                       /*!< bDescriptorType */
    0x06,                                       /*!< bDescriptorSubtype */
    0x00,                                       /*!< bMasterInterface */
    0x01,                                       /*!< bSlaveInterface0 */
    /********************************************** Endpoint 0 Descriptor */
    0x07,                                       /*!< bLength */
    0x05,                                       /*!< bDescriptorType */
    USBD_IF0_AL0_EP0_ADDR,                      /*!< bEndpointAddress */
    0x03,                                       /*!< bmAttributes */
    WBVAL(USBD_IF0_AL0_EP0_SIZE),               /*!< wMaxPacketSize */
    USBD_IF0_AL0_EP0_INTERVAL,                  /*!< bInterval */
    /********************************************** Interface 1 Alternate 0 Descriptor */
    0x09,                                       /*!< bLength */
    0x04,                                       /*!< bDescriptorType */
    0x01,                                       /*!< bInterfaceNumber */
    0x00,                                       /*!< bAlternateSetting */
    0x02,                                       /*!< bNumEndpoints */
    0x0a,                                       /*!< bInterfaceClass */
    0x00,                                       /*!< bInterfaceSubClass */
    0x00,                                       /*!< bInterfaceProtocol */
    0x05,                                       /*!< iInterface */
    /********************************************** Class Specific Descriptor of CDC ACM Data */
    /********************************************** Endpoint 0 Descriptor */
    0x07,                                       /*!< bLength */
    0x05,                                       /*!< bDescriptorType */
    USBD_IF1_AL0_EP0_ADDR,                      /*!< bEndpointAddress */
    0x02,                                       /*!< bmAttributes */
    WBVAL(USBD_IF1_AL0_EP0_SIZE),               /*!< wMaxPacketSize */
    USBD_IF1_AL0_EP0_INTERVAL,                  /*!< bInterval */
    /********************************************** Endpoint 1 Descriptor */
    0x07,                                       /*!< bLength */
    0x05,                                       /*!< bDescriptorType */
    USBD_IF1_AL0_EP1_ADDR,                      /*!< bEndpointAddress */
    0x02,                                       /*!< bmAttributes */
    WBVAL(USBD_IF1_AL0_EP1_SIZE),               /*!< wMaxPacketSize */
    USBD_IF1_AL0_EP1_INTERVAL,                  /*!< bInterval */
    /********************************************** Interface 2 Alternate 0 Descriptor */
    0x09,                                       /*!< bLength */
    0x04,                                       /*!< bDescriptorType */
    0x02,                                       /*!< bInterfaceNumber */
    0x00,                                       /*!< bAlternateSetting */
    0x02,                                       /*!< bNumEndpoints */
    0x03,                                       /*!< bInterfaceClass */
    0x00,                                       /*!< bInterfaceSubClass */
    0x00,                                       /*!< bInterfaceProtocol */
    0x06,                                       /*!< iInterface */
    /********************************************** Class Specific Descriptor of HID */
    0x09,                                       /*!< bLength */
    0x21,                                       /*!< bDescriptorType */
    WBVAL(USBD_HID_VERSION),                    /*!< bcdHID */
    USBD_HID_COUNTRY_CODE,                      /*!< bCountryCode */
    0x01,                                       /*!< bNumDescriptors */
    0x22,                                       /*!< bDescriptorType */
    WBVAL(HID_CUSTOM_REPORT_DESC_SIZE),   /*!< wItemLength */
    /********************************************** Endpoint 0 Descriptor */
    0x07,                                       /*!< bLength */
    0x05,                                       /*!< bDescriptorType */
    USBD_IF2_AL0_EP0_ADDR,                      /*!< bEndpointAddress */
    0x03,                                       /*!< bmAttributes */
    WBVAL(USBD_IF2_AL0_EP0_SIZE),               /*!< wMaxPacketSize */
    USBD_IF2_AL0_EP0_INTERVAL,                  /*!< bInterval */
    /********************************************** Endpoint 1 Descriptor */
    0x07,                                       /*!< bLength */
    0x05,                                       /*!< bDescriptorType */
    USBD_IF2_AL0_EP1_ADDR,                      /*!< bEndpointAddress */
    0x03,                                       /*!< bmAttributes */
    WBVAL(USBD_IF2_AL0_EP1_SIZE),               /*!< wMaxPacketSize */
    USBD_IF2_AL0_EP1_INTERVAL,                  /*!< bInterval */
    /********************************************** Interface 3 Alternate 0 Descriptor */
    0x09,                                       /*!< bLength */
    0x04,                                       /*!< bDescriptorType */
    0x03,                                       /*!< bInterfaceNumber */
    0x00,                                       /*!< bAlternateSetting */
    0x02,                                       /*!< bNumEndpoints */
    0xff,                                       /*!< bInterfaceClass */
    0x00,                                       /*!< bInterfaceSubClass */
    0x00,                                       /*!< bInterfaceProtocol */
    0x07,                                       /*!< iInterface */
    /********************************************** Endpoint 0 Descriptor */
    0x07,                                       /*!< bLength */
    0x05,                                       /*!< bDescriptorType */
    USBD_IF3_AL0_EP0_ADDR,                      /*!< bEndpointAddress */
    0x02,                                       /*!< bmAttributes */
    WBVAL(USBD_IF3_AL0_EP0_SIZE),               /*!< wMaxPacketSize */
    USBD_IF3_AL0_EP0_INTERVAL,                  /*!< bInterval */
    /********************************************** Endpoint 1 Descriptor */
    0x07,                                       /*!< bLength */
    0x05,                                       /*!< bDescriptorType */
    USBD_IF3_AL0_EP1_ADDR,                      /*!< bEndpointAddress */
    0x02,                                       /*!< bmAttributes */
    WBVAL(USBD_IF3_AL0_EP1_SIZE),               /*!< wMaxPacketSize */
    USBD_IF3_AL0_EP1_INTERVAL,                  /*!< bInterval */
    /********************************************** Language ID String Descriptor */
    0x04,                                       /*!< bLength */
    0x03,                                       /*!< bDescriptorType */
    WBVAL(USBD_LANGID_STRING),                  /*!< wLangID0 */
    /********************************************** String 1 Descriptor */
    /* Cherry USB */
    0x16,                                       /*!< bLength */
    0x03,                                       /*!< bDescriptorType */
    0x43, 0x00,                                 /*!< 'C' wcChar0 */
    0x68, 0x00,                                 /*!< 'h' wcChar1 */
    0x65, 0x00,                                 /*!< 'e' wcChar2 */
    0x72, 0x00,                                 /*!< 'r' wcChar3 */
    0x72, 0x00,                                 /*!< 'r' wcChar4 */
    0x79, 0x00,                                 /*!< 'y' wcChar5 */
    0x20, 0x00,                                 /*!< ' ' wcChar6 */
    0x55, 0x00,                                 /*!< 'U' wcChar7 */
    0x53, 0x00,                                 /*!< 'S' wcChar8 */
    0x42, 0x00,                                 /*!< 'B' wcChar9 */
    /********************************************** String 2 Descriptor */
    /* Eastsoft CherryUSB HID CDC WINUSB demo */
    0x4e,                                       /*!< bLength */
    0x03,                                       /*!< bDescriptorType */
    0x45, 0x00,                                 /*!< 'E' wcChar0 */
    0x61, 0x00,                                 /*!< 'a' wcChar1 */
    0x73, 0x00,                                 /*!< 's' wcChar2 */
    0x74, 0x00,                                 /*!< 't' wcChar3 */
    0x73, 0x00,                                 /*!< 's' wcChar4 */
    0x6f, 0x00,                                 /*!< 'o' wcChar5 */
    0x66, 0x00,                                 /*!< 'f' wcChar6 */
    0x74, 0x00,                                 /*!< 't' wcChar7 */
    0x20, 0x00,                                 /*!< ' ' wcChar8 */
    0x43, 0x00,                                 /*!< 'C' wcChar9 */
    0x68, 0x00,                                 /*!< 'h' wcChar10 */
    0x65, 0x00,                                 /*!< 'e' wcChar11 */
    0x72, 0x00,                                 /*!< 'r' wcChar12 */
    0x72, 0x00,                                 /*!< 'r' wcChar13 */
    0x79, 0x00,                                 /*!< 'y' wcChar14 */
    0x55, 0x00,                                 /*!< 'U' wcChar15 */
    0x53, 0x00,                                 /*!< 'S' wcChar16 */
    0x42, 0x00,                                 /*!< 'B' wcChar17 */
    0x20, 0x00,                                 /*!< ' ' wcChar18 */
    0x48, 0x00,                                 /*!< 'H' wcChar19 */
    0x49, 0x00,                                 /*!< 'I' wcChar20 */
    0x44, 0x00,                                 /*!< 'D' wcChar21 */
    0x20, 0x00,                                 /*!< ' ' wcChar22 */
    0x43, 0x00,                                 /*!< 'C' wcChar23 */
    0x44, 0x00,                                 /*!< 'D' wcChar24 */
    0x43, 0x00,                                 /*!< 'C' wcChar25 */
    0x20, 0x00,                                 /*!< ' ' wcChar26 */
    0x57, 0x00,                                 /*!< 'W' wcChar27 */
    0x49, 0x00,                                 /*!< 'I' wcChar28 */
    0x4e, 0x00,                                 /*!< 'N' wcChar29 */
    0x55, 0x00,                                 /*!< 'U' wcChar30 */
    0x53, 0x00,                                 /*!< 'S' wcChar31 */
    0x42, 0x00,                                 /*!< 'B' wcChar32 */
    0x20, 0x00,                                 /*!< ' ' wcChar33 */
    0x64, 0x00,                                 /*!< 'd' wcChar34 */
    0x65, 0x00,                                 /*!< 'e' wcChar35 */
    0x6d, 0x00,                                 /*!< 'm' wcChar36 */
    0x6f, 0x00,                                 /*!< 'o' wcChar37 */
    /********************************************** String 3 Descriptor */
    /* 202210201641 */
    0x1a,                                       /*!< bLength */
    0x03,                                       /*!< bDescriptorType */
    0x32, 0x00,                                 /*!< '2' wcChar0 */
    0x30, 0x00,                                 /*!< '0' wcChar1 */
    0x32, 0x00,                                 /*!< '2' wcChar2 */
    0x32, 0x00,                                 /*!< '2' wcChar3 */
    0x31, 0x00,                                 /*!< '1' wcChar4 */
    0x30, 0x00,                                 /*!< '0' wcChar5 */
    0x32, 0x00,                                 /*!< '2' wcChar6 */
    0x30, 0x00,                                 /*!< '0' wcChar7 */
    0x31, 0x00,                                 /*!< '1' wcChar8 */
    0x36, 0x00,                                 /*!< '6' wcChar9 */
    0x34, 0x00,                                 /*!< '4' wcChar10 */
    0x31, 0x00,                                 /*!< '1' wcChar11 */
    /********************************************** String 4 Descriptor */
    /* cdc control interface */
    0x2c,                                       /*!< bLength */
    0x03,                                       /*!< bDescriptorType */
    0x63, 0x00,                                 /*!< 'c' wcChar0 */
    0x64, 0x00,                                 /*!< 'd' wcChar1 */
    0x63, 0x00,                                 /*!< 'c' wcChar2 */
    0x20, 0x00,                                 /*!< ' ' wcChar3 */
    0x63, 0x00,                                 /*!< 'c' wcChar4 */
    0x6f, 0x00,                                 /*!< 'o' wcChar5 */
    0x6e, 0x00,                                 /*!< 'n' wcChar6 */
    0x74, 0x00,                                 /*!< 't' wcChar7 */
    0x72, 0x00,                                 /*!< 'r' wcChar8 */
    0x6f, 0x00,                                 /*!< 'o' wcChar9 */
    0x6c, 0x00,                                 /*!< 'l' wcChar10 */
    0x20, 0x00,                                 /*!< ' ' wcChar11 */
    0x69, 0x00,                                 /*!< 'i' wcChar12 */
    0x6e, 0x00,                                 /*!< 'n' wcChar13 */
    0x74, 0x00,                                 /*!< 't' wcChar14 */
    0x65, 0x00,                                 /*!< 'e' wcChar15 */
    0x72, 0x00,                                 /*!< 'r' wcChar16 */
    0x66, 0x00,                                 /*!< 'f' wcChar17 */
    0x61, 0x00,                                 /*!< 'a' wcChar18 */
    0x63, 0x00,                                 /*!< 'c' wcChar19 */
    0x65, 0x00,                                 /*!< 'e' wcChar20 */
    /********************************************** String 5 Descriptor */
    /* cdc data interface */
    0x26,                                       /*!< bLength */
    0x03,                                       /*!< bDescriptorType */
    0x63, 0x00,                                 /*!< 'c' wcChar0 */
    0x64, 0x00,                                 /*!< 'd' wcChar1 */
    0x63, 0x00,                                 /*!< 'c' wcChar2 */
    0x20, 0x00,                                 /*!< ' ' wcChar3 */
    0x64, 0x00,                                 /*!< 'd' wcChar4 */
    0x61, 0x00,                                 /*!< 'a' wcChar5 */
    0x74, 0x00,                                 /*!< 't' wcChar6 */
    0x61, 0x00,                                 /*!< 'a' wcChar7 */
    0x20, 0x00,                                 /*!< ' ' wcChar8 */
    0x69, 0x00,                                 /*!< 'i' wcChar9 */
    0x6e, 0x00,                                 /*!< 'n' wcChar10 */
    0x74, 0x00,                                 /*!< 't' wcChar11 */
    0x65, 0x00,                                 /*!< 'e' wcChar12 */
    0x72, 0x00,                                 /*!< 'r' wcChar13 */
    0x66, 0x00,                                 /*!< 'f' wcChar14 */
    0x61, 0x00,                                 /*!< 'a' wcChar15 */
    0x63, 0x00,                                 /*!< 'c' wcChar16 */
    0x65, 0x00,                                 /*!< 'e' wcChar17 */
    /********************************************** String 6 Descriptor */
    /* hid interface */
    0x1c,                                       /*!< bLength */
    0x03,                                       /*!< bDescriptorType */
    0x68, 0x00,                                 /*!< 'h' wcChar0 */
    0x69, 0x00,                                 /*!< 'i' wcChar1 */
    0x64, 0x00,                                 /*!< 'd' wcChar2 */
    0x20, 0x00,                                 /*!< ' ' wcChar3 */
    0x69, 0x00,                                 /*!< 'i' wcChar4 */
    0x6e, 0x00,                                 /*!< 'n' wcChar5 */
    0x74, 0x00,                                 /*!< 't' wcChar6 */
    0x65, 0x00,                                 /*!< 'e' wcChar7 */
    0x72, 0x00,                                 /*!< 'r' wcChar8 */
    0x66, 0x00,                                 /*!< 'f' wcChar9 */
    0x61, 0x00,                                 /*!< 'a' wcChar10 */
    0x63, 0x00,                                 /*!< 'c' wcChar11 */
    0x65, 0x00,                                 /*!< 'e' wcChar12 */
    /********************************************** String 7 Descriptor */
    /* winusb interface */
    0x22,                                       /*!< bLength */
    0x03,                                       /*!< bDescriptorType */
    0x77, 0x00,                                 /*!< 'w' wcChar0 */
    0x69, 0x00,                                 /*!< 'i' wcChar1 */
    0x6e, 0x00,                                 /*!< 'n' wcChar2 */
    0x75, 0x00,                                 /*!< 'u' wcChar3 */
    0x73, 0x00,                                 /*!< 's' wcChar4 */
    0x62, 0x00,                                 /*!< 'b' wcChar5 */
    0x20, 0x00,                                 /*!< ' ' wcChar6 */
    0x69, 0x00,                                 /*!< 'i' wcChar7 */
    0x6e, 0x00,                                 /*!< 'n' wcChar8 */
    0x74, 0x00,                                 /*!< 't' wcChar9 */
    0x65, 0x00,                                 /*!< 'e' wcChar10 */
    0x72, 0x00,                                 /*!< 'r' wcChar11 */
    0x66, 0x00,                                 /*!< 'f' wcChar12 */
    0x61, 0x00,                                 /*!< 'a' wcChar13 */
    0x63, 0x00,                                 /*!< 'c' wcChar14 */
    0x65, 0x00,                                 /*!< 'e' wcChar15 */
    /********************************************** Device Qualifier Descriptor */
    0x0a,                                       /*!< bLength */
    0x06,                                       /*!< bDescriptorType */
    WBVAL(USBD_VERSION),                        /*!< bcdUSB */
    0x00,                                       /*!< bDeviceClass */
    0x00,                                       /*!< bDeviceSubClass */
    0x00,                                       /*!< bDeviceProtocol */
    0x40,                                       /*!< bMaxPacketSize0 */
    0x01,                                       /*!< bNumConfigurations */
    0x00,                                       /*!< bReserved */
    0x00
};

/*!< USBD HID REPORT 0 Descriptor */
const unsigned char usbd_hid_0_report_descriptor[USBD_IF2_AL0_HID_REPORT_DESC_SIZE] =
{
    0x00,
    0xC0    /* END_COLLECTION */
};

/*******************************************************END OF ChryUSB_Configurator Generation*************/

/* Private function prototypes ----------------------------------------------- */
static void usbd_hid_custom_in_callback(uint8_t busid, uint8_t ep, uint32_t nbytes);
static void usbd_hid_custom_out_callback(uint8_t busid, uint8_t ep, uint32_t nbytes);
void usbd_winusb_out(uint8_t busid, uint8_t ep, uint32_t nbytes);
void usbd_winusb_in(uint8_t busid, uint8_t ep, uint32_t nbytes);
void usbd_cdc_acm_bulk_out(uint8_t busid, uint8_t ep, uint32_t nbytes);
void usbd_cdc_acm_bulk_in(uint8_t busid, uint8_t ep, uint32_t nbytes);
void usbd_cdc_acm_int_in(uint8_t busid, uint8_t ep, uint32_t nbytes);

/*!< endpoint call back */
static struct usbd_endpoint s_cdc_out_ep =
{
    .ep_addr = USBD_IF1_AL0_EP0_ADDR,
    .ep_cb = usbd_cdc_acm_bulk_out
};

static struct usbd_endpoint s_cdc_in_ep =
{
    .ep_addr = USBD_IF1_AL0_EP1_ADDR,
    .ep_cb = usbd_cdc_acm_bulk_in
};

static struct usbd_endpoint s_cdc_int_ep =
{
    .ep_addr = USBD_IF0_AL0_EP0_ADDR,
    .ep_cb = usbd_cdc_acm_int_in
};

static struct usbd_endpoint s_winusb_out_ep =
{
    .ep_addr = USBD_IF3_AL0_EP0_ADDR,
    .ep_cb = usbd_winusb_out
};

static struct usbd_endpoint s_winusb_in_ep =
{
    .ep_addr = USBD_IF3_AL0_EP1_ADDR,
    .ep_cb = usbd_winusb_in
};

static struct usbd_endpoint s_custom_in_ep =
{
    .ep_cb = usbd_hid_custom_in_callback,
    .ep_addr = USBD_IF2_AL0_EP1_ADDR
};

static struct usbd_endpoint s_custom_out_ep =
{
    .ep_cb = usbd_hid_custom_out_callback,
    .ep_addr = USBD_IF2_AL0_EP0_ADDR
};

/* Private Function ---------------------------------------------------------- */
void usbd_event_handler(uint8_t busid, uint8_t event)
{
    switch (event)
    {
        case USBD_EVENT_RESET:
            break;

        case USBD_EVENT_CONNECTED:
            break;

        case USBD_EVENT_DISCONNECTED:
            break;

        case USBD_EVENT_RESUME:
            break;

        case USBD_EVENT_SUSPEND:
            break;

        case USBD_EVENT_CONFIGURED:
        {
            usbd_ep_start_read(0, s_cdc_out_ep.ep_addr, s_read_buffer_cdc, ES_USB_DEV_EP_READ_BUFFER_SIZE);
            usbd_ep_start_read(0, s_custom_out_ep.ep_addr, s_read_buffer_hid, ES_USB_DEV_EP_READ_BUFFER_SIZE);
            usbd_ep_start_read(0, s_winusb_out_ep.ep_addr, s_read_buffer_winusb, ES_USB_DEV_EP_READ_BUFFER_SIZE);
            s_configed_flag = 1;

            break;
        }

        case USBD_EVENT_SET_REMOTE_WAKEUP:
            break;

        case USBD_EVENT_CLR_REMOTE_WAKEUP:
            break;

        default:
            break;
    }
}

static void usbd_hid_custom_in_callback(uint8_t busid, uint8_t ep, uint32_t nbytes)
{
    int ret;

    if (s_hid_ep_buf_fill_num)
    {
        ret = usbd_ep_start_write(0, s_custom_in_ep.ep_addr, s_write_buffer_hid[s_hid_ep_buf_tx_index], s_read_num_hid[s_hid_ep_buf_tx_index]);

        if (ret == 0)
        {
            s_hid_ep_buf_fill_num--;
            s_hid_ep_buf_tx_index = (s_hid_ep_buf_tx_index + 1) % ES_USB_DEV_EP_WIRTE_BUFFER_NUM;
            s_hid_ep_tx_flag = ES_USB_DEV_EP_TX_FLAG_BUSY;
        }
        else
            USB_LOG_ERR("hid ep write fail");
    }
    else
        s_hid_ep_tx_flag = ES_USB_DEV_EP_TX_FLAG_IDLE;
}

static void usbd_hid_custom_out_callback(uint8_t busid, uint8_t ep, uint32_t nbytes)
{
    int ret;

    if (nbytes == 0)
        USB_LOG_ERR("ep read buf size too small");

    if (s_hid_ep_buf_fill_num < ES_USB_DEV_EP_WIRTE_BUFFER_NUM)
    {
        memcpy(s_write_buffer_hid[s_hid_ep_buf_fill_index], s_read_buffer_hid, nbytes);
        s_read_num_hid[s_hid_ep_buf_fill_index] = nbytes;
        s_hid_ep_buf_fill_num++;
        s_hid_ep_buf_fill_index = (s_hid_ep_buf_fill_index + 1) % ES_USB_DEV_EP_WIRTE_BUFFER_NUM;
    }

    ret = usbd_ep_start_read(0, s_custom_out_ep.ep_addr, s_read_buffer_hid, ES_USB_DEV_EP_READ_BUFFER_SIZE);

    if (ret != 0)
        USB_LOG_ERR("hid ep read fail");

    if ((s_hid_ep_tx_flag == ES_USB_DEV_EP_TX_FLAG_IDLE) && (s_hid_ep_buf_fill_num))
    {
        ret = usbd_ep_start_write(0, s_custom_in_ep.ep_addr, s_write_buffer_hid[s_hid_ep_buf_tx_index], s_read_num_hid[s_hid_ep_buf_tx_index]);

        if (ret == 0)
        {
            s_hid_ep_buf_fill_num--;
            s_hid_ep_buf_tx_index = (s_hid_ep_buf_tx_index + 1) % ES_USB_DEV_EP_WIRTE_BUFFER_NUM;
            s_hid_ep_tx_flag = ES_USB_DEV_EP_TX_FLAG_BUSY;
        }
        else
            USB_LOG_ERR("hid ep write fail");
    }
}

void usbd_winusb_in(uint8_t busid, uint8_t ep, uint32_t nbytes)
{
    int ret;

    if (s_winusb_ep_rx_buf_fill_num)
    {
        ret = usbd_ep_start_write(0, s_winusb_in_ep.ep_addr, s_write_buffer_winusb[s_winusb_ep_buf_tx_index], s_read_num_winusb[s_winusb_ep_buf_tx_index]);

        if (ret == 0)
        {
            s_winusb_ep_rx_buf_fill_num--;
            s_winusb_ep_buf_tx_index = (s_winusb_ep_buf_tx_index + 1) % ES_USB_DEV_EP_WIRTE_BUFFER_NUM;
            s_winusb_ep_tx_flag = ES_USB_DEV_EP_TX_FLAG_BUSY;
        }
        else
            USB_LOG_ERR("winusb ep write fail");
    }
    else
    {
#if ES_USB_SEND_0_PACKEGE_AS_END

        if (nbytes)
        {
            ret = usbd_ep_start_write(0, s_winusb_in_ep.ep_addr, NULL, 0);

            if (ret == 0)
                s_winusb_ep_tx_flag = ES_USB_DEV_EP_TX_FLAG_END;
            else
                USB_LOG_ERR("winusb ep write fail");
        }
        else
        {
            s_winusb_ep_tx_flag = ES_USB_DEV_EP_TX_FLAG_IDLE;
        }

#else
        s_winusb_ep_tx_flag = ES_USB_DEV_EP_TX_FLAG_IDLE;
#endif/*ES_USB_SEND_0_PACKEGE_AS_END*/
    }
}

void usbd_winusb_out(uint8_t busid, uint8_t ep, uint32_t nbytes)
{
    int ret;

    if (nbytes == 0)
        USB_LOG_ERR("ep read buf size too small");

    if (s_winusb_ep_rx_buf_fill_num < ES_USB_DEV_EP_WIRTE_BUFFER_NUM)
    {
        memcpy(s_write_buffer_winusb[s_winusb_ep_buf_fill_index], s_read_buffer_winusb, nbytes);
        s_read_num_winusb[s_winusb_ep_buf_fill_index] = nbytes;
        s_winusb_ep_rx_buf_fill_num++;
        s_winusb_ep_buf_fill_index = (s_winusb_ep_buf_fill_index + 1) % ES_USB_DEV_EP_WIRTE_BUFFER_NUM;
    }

    ret = usbd_ep_start_read(0, s_winusb_out_ep.ep_addr, s_read_buffer_winusb, ES_USB_DEV_EP_READ_BUFFER_SIZE);

    if (ret != 0)
        USB_LOG_ERR("winusb ep read fail");

    if ((s_winusb_ep_tx_flag == ES_USB_DEV_EP_TX_FLAG_IDLE) && (s_winusb_ep_rx_buf_fill_num))
    {
        ret = usbd_ep_start_write(0, s_winusb_in_ep.ep_addr, s_write_buffer_winusb[s_winusb_ep_buf_tx_index], s_read_num_winusb[s_winusb_ep_buf_tx_index]);

        if (ret == 0)
        {
            s_winusb_ep_rx_buf_fill_num--;
            s_winusb_ep_buf_tx_index = (s_winusb_ep_buf_tx_index + 1) % ES_USB_DEV_EP_WIRTE_BUFFER_NUM;
            s_winusb_ep_tx_flag = ES_USB_DEV_EP_TX_FLAG_BUSY;
        }
        else
            USB_LOG_ERR("winusb ep write fail");
    }
}

void usbd_cdc_acm_bulk_in(uint8_t busid, uint8_t ep, uint32_t nbytes)
{
    int ret;

    if (s_cdc_ep_buf_fill_num)
    {
        ret = usbd_ep_start_write(0, s_cdc_in_ep.ep_addr, s_write_buffer_cdc[s_cdc_ep_buf_tx_index], s_read_num_cdc[s_cdc_ep_buf_tx_index]);

        if (ret == 0)
        {
            s_cdc_ep_buf_fill_num--;
            s_cdc_ep_buf_tx_index = (s_cdc_ep_buf_tx_index + 1) % ES_USB_DEV_EP_WIRTE_BUFFER_NUM;
            s_cdc_ep_tx_flag = ES_USB_DEV_EP_TX_FLAG_BUSY;
        }
        else
            USB_LOG_ERR("cdc ep write fail");
    }
    else
    {
#if ES_USB_SEND_0_PACKEGE_AS_END

        if (nbytes)
        {
            ret = usbd_ep_start_write(0, s_cdc_in_ep.ep_addr, NULL, 0);

            if (ret == 0)
                s_cdc_ep_tx_flag = ES_USB_DEV_EP_TX_FLAG_END;
            else
                USB_LOG_ERR("cdc ep write fail");
        }
        else
        {
            s_cdc_ep_tx_flag = ES_USB_DEV_EP_TX_FLAG_IDLE;
        }

#else
        s_cdc_ep_tx_flag = ES_USB_DEV_EP_TX_FLAG_IDLE;
#endif/*ES_USB_SEND_0_PACKEGE_AS_END*/
    }
}

void usbd_cdc_acm_bulk_out(uint8_t busid, uint8_t ep, uint32_t nbytes)
{
    int ret;

    if (nbytes == 0)
        USB_LOG_ERR("ep read buf size too small");

    if (s_cdc_ep_buf_fill_num < ES_USB_DEV_EP_WIRTE_BUFFER_NUM)
    {
        memcpy(s_write_buffer_cdc[s_cdc_ep_buf_fill_index], s_read_buffer_cdc, nbytes);
        s_read_num_cdc[s_cdc_ep_buf_fill_index] = nbytes;
        s_cdc_ep_buf_fill_num++;
        s_cdc_ep_buf_fill_index = (s_cdc_ep_buf_fill_index + 1) % ES_USB_DEV_EP_WIRTE_BUFFER_NUM;
    }

    ret = usbd_ep_start_read(0, s_cdc_out_ep.ep_addr, s_read_buffer_cdc, ES_USB_DEV_EP_READ_BUFFER_SIZE);

    if (ret != 0)
        USB_LOG_ERR("cdc ep read fail");

    if ((s_cdc_ep_tx_flag == ES_USB_DEV_EP_TX_FLAG_IDLE) && (s_cdc_ep_buf_fill_num))
    {
        ret = usbd_ep_start_write(0, s_cdc_in_ep.ep_addr, s_write_buffer_cdc[s_cdc_ep_buf_tx_index], s_read_num_cdc[s_cdc_ep_buf_tx_index]);

        if (ret == 0)
        {
            s_cdc_ep_buf_fill_num--;
            s_cdc_ep_buf_tx_index = (s_cdc_ep_buf_tx_index + 1) % ES_USB_DEV_EP_WIRTE_BUFFER_NUM;
            s_cdc_ep_tx_flag = ES_USB_DEV_EP_TX_FLAG_BUSY;
        }
        else
            USB_LOG_ERR("cdc ep write fail");
    }
}

void usbd_cdc_acm_int_in(uint8_t busid, uint8_t ep, uint32_t nbytes)
{
    USB_LOG_RAW("cdc control");
}

void usbd_cdc_acm_set_line_coding(uint8_t busid,  uint8_t intf, struct cdc_line_coding *line_coding)
{
    if (intf == 0)
    {
        s_cdc_line_code.bCharFormat = line_coding->bCharFormat;
        s_cdc_line_code.bParityType = line_coding->bParityType;
        s_cdc_line_code.bDataBits = line_coding->bDataBits;
        s_cdc_line_code.dwDTERate = line_coding->dwDTERate;
    }
}

void usbd_cdc_acm_get_line_coding(uint8_t busid, uint8_t intf, struct cdc_line_coding *line_coding)
{
    if (intf == 0)
    {
        line_coding->dwDTERate = s_cdc_line_code.dwDTERate;
        line_coding->bDataBits = s_cdc_line_code.bDataBits;
        line_coding->bParityType = s_cdc_line_code.bParityType;
        line_coding->bCharFormat = s_cdc_line_code.bCharFormat;
    }
}

void usbd_cdc_acm_send_break(uint8_t busid, uint8_t intf)
{

}

/**
  * @brief            cdc hid winusb init
  * @pre              none
  * @param[in]        none
  * @retval           none
  */
void cdc_acm_hid_winusb_descriptor_init(void)
{
    usbd_desc_register(0, usbd_descriptor);

    usbd_add_interface(0, usbd_cdc_acm_init_intf(0, s_cdc_ctl_intf));
    usbd_add_interface(0, usbd_cdc_acm_init_intf(0, s_cdc_data_intf));
    usbd_add_endpoint(0, &s_cdc_out_ep);
    usbd_add_endpoint(0, &s_cdc_in_ep);
    usbd_add_endpoint(0, &s_cdc_int_ep);

    usbd_add_interface(0, usbd_hid_init_intf(0, s_hid_custom_intf, s_hid_custom_report_desc, HID_CUSTOM_REPORT_DESC_SIZE));
    usbd_add_endpoint(0, &s_custom_in_ep);
    usbd_add_endpoint(0, &s_custom_out_ep);

    usbd_msosv1_desc_register(0, &s_msosv1_desc);
    usbd_add_interface(0, s_winusb_intf);
    usbd_add_endpoint(0, &s_winusb_out_ep);
    usbd_add_endpoint(0, &s_winusb_in_ep);

    s_configed_flag = 0;
    usbd_initialize(0, 0, usbd_event_handler);

    while (!s_configed_flag);
}


void usbd_hid_set_idle(uint8_t busid, uint8_t intf, uint8_t report_id, uint8_t duration)
{
    g_usb_hid_idle = duration;
}

uint8_t usbd_hid_get_idle(uint8_t busid, uint8_t intf, uint8_t report_id)
{
    return g_usb_hid_idle;
}

void usbd_hid_set_protocol(uint8_t busid, uint8_t intf, uint8_t protocol)
{
    g_usb_hid_protocol = protocol;
}

uint8_t usbd_hid_get_protocol(uint8_t busid, uint8_t intf)
{
	return g_usb_hid_protocol;
}

/*******************************************************END OF FILE*************/
