/**********************************************************************************
 *
 * @file    .c
 * @brief   Source file
 *
 * @author  AE Team
 * @note
 *          Change Logs:
 *          Date               Author          Notes
 *          2023-01-17         liuhy           the first version
 *
 * Copyright (C) Shanghai Eastsoft Microelectronics Co. Ltd. All rights reserved.
 *
 * SPDX-License-Identifier: Apache-2.0
 *
 * Licensed under the Apache License, Version 2.0 (the License); you may
 * not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an AS IS BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 *
 **********************************************************************************
 */

/* Includes ------------------------------------------------------------------ */
#include "main.h"
#include "usbd_core.h"

/* Private Macros ------------------------------------------------------------ */
/*!< custom hid report descriptor size */
#define HID_CUSTOM_REPORT_DESC_SIZE (34)
#define ES_USB_DEV_EP_READ_BUFFER_SIZE (64)
#define ES_USB_DEV_EP_WIRTE_BUFFER_NUM (4)
#define ES_USB_SEND_0_PACKEGE_AS_END   1
#define ES_USB_DEV_EP_TX_FLAG_IDLE    (0U)
#define ES_USB_DEV_EP_TX_FLAG_BUSY    (1U)
#define ES_USB_DEV_EP_TX_FLAG_END     (2U)
#define WCID_VENDOR_CODE              0x17

/* Private Variables --------------------------------------------------------- */
/*!< custom hid report descriptor */
static __ALIGN_BEGIN const uint8_t s_WCID_StringDescriptor_MSOS[18] __ALIGN_END =
{
    /* MS OS string descriptor */
    0x12,                       /* bLength */
    USB_DESCRIPTOR_TYPE_STRING, /* bDescriptorType */
    /* MSFT100 */
    'M', 0x00, 'S', 0x00, 'F', 0x00, 'T', 0x00, /* wcChar_7 */
    '1', 0x00, '0', 0x00, '0', 0x00,            /* wcChar_7 */
    WCID_VENDOR_CODE,                           /* bVendorCode */
    0x00,                                       /* bReserved */
};

static __ALIGN_BEGIN const uint8_t s_WINUSB_WCIDDescriptor[40] __ALIGN_END =
{
    /* WCID descriptor */
    0x28, 0x00, 0x00, 0x00,                   /* dwLength */
    0x00, 0x01,                               /* bcdVersion */
    0x04, 0x00,                               /* wIndex */
    0x01,                                     /* bCount */
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* bReserved_7 */

    /* WCID function descriptor */
    0x00, /* bFirstInterfaceNumber */
    0x01, /* bReserved */
    /* WINUSB */
    'W', 'I', 'N', 'U', 'S', 'B', 0x00, 0x00, /* cCID_8 */
    /*  */
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* cSubCID_8 */
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00,             /* bReserved_6 */
};

static __ALIGN_BEGIN const uint8_t s_WINUSB_IF0_WCIDProperties[142] __ALIGN_END =
{
    /* WCID property descriptor */
    0x8e, 0x00, 0x00, 0x00, /* dwLength */
    0x00, 0x01,             /* bcdVersion */
    0x05, 0x00,             /* wIndex */
    0x01, 0x00,             /* wCount */

    /* registry propter descriptor */
    0x84, 0x00, 0x00, 0x00, /* dwSize */
    0x01, 0x00, 0x00, 0x00, /* dwPropertyDataType */
    0x28, 0x00,             /* wPropertyNameLength */
    /* DeviceInterfaceGUID */
    'D', 0x00, 'e', 0x00, 'v', 0x00, 'i', 0x00,  /* wcName_20 */
    'c', 0x00, 'e', 0x00, 'I', 0x00, 'n', 0x00,  /* wcName_20 */
    't', 0x00, 'e', 0x00, 'r', 0x00, 'f', 0x00,  /* wcName_20 */
    'a', 0x00, 'c', 0x00, 'e', 0x00, 'G', 0x00,  /* wcName_20 */
    'U', 0x00, 'I', 0x00, 'D', 0x00, 0x00, 0x00, /* wcName_20 */
    0x4e, 0x00, 0x00, 0x00,                      /* dwPropertyDataLength */
    /* {CDB3B5AD-293B-4663-AA36-1AAE46463776} */
    '{', 0x00, 'C', 0x00, 'D', 0x00, 'B', 0x00, /* wcData_39 */
    '3', 0x00, 'B', 0x00, '5', 0x00, 'A', 0x00, /* wcData_39 */
    'D', 0x00, '-', 0x00, '2', 0x00, '9', 0x00, /* wcData_39 */
    '3', 0x00, 'B', 0x00, '-', 0x00, '4', 0x00, /* wcData_39 */
    '6', 0x00, '6', 0x00, '3', 0x00, '-', 0x00, /* wcData_39 */
    'A', 0x00, 'A', 0x00, '3', 0x00, '6', 0x00, /* wcData_39 */
    '-', 0x00, '1', 0x00, 'A', 0x00, 'A', 0x00, /* wcData_39 */
    'E', 0x00, '4', 0x00, '6', 0x00, '4', 0x00, /* wcData_39 */
    '6', 0x00, '3', 0x00, '7', 0x00, '7', 0x00, /* wcData_39 */
    '6', 0x00, '}', 0x00, 0x00, 0x00,           /* wcData_39 */
};

const uint8_t *s_WINUSB_IFx_WCIDProperties[] =
{
    s_WINUSB_IF0_WCIDProperties,
};

static struct usb_msosv1_descriptor s_msosv1_desc =
{
    .string = s_WCID_StringDescriptor_MSOS,
    .vendor_code = WCID_VENDOR_CODE,
    .compat_id = s_WINUSB_WCIDDescriptor,
    .comp_id_property = s_WINUSB_IFx_WCIDProperties,
};

static struct usbd_interface s_winusb_intf[1];
static volatile uint8_t s_configed_flag = 0U;
static volatile uint32_t s_read_num_winusb[ES_USB_DEV_EP_WIRTE_BUFFER_NUM];
static volatile uint8_t s_winusb_ep_tx_flag = ES_USB_DEV_EP_TX_FLAG_IDLE;
static volatile uint8_t s_winusb_ep_buf_fill_index = 0U;
static volatile uint8_t s_winusb_ep_buf_tx_index = 0U;
static volatile uint8_t s_winusb_ep_rx_buf_fill_num = 0U;
static USB_NOCACHE_RAM_SECTION USB_MEM_ALIGNX uint8_t s_read_buffer_winusb[ES_USB_DEV_EP_READ_BUFFER_SIZE];
static USB_NOCACHE_RAM_SECTION USB_MEM_ALIGNX uint8_t s_write_buffer_winusb[ES_USB_DEV_EP_WIRTE_BUFFER_NUM][ES_USB_DEV_EP_READ_BUFFER_SIZE];

/* Public Variables ---------------------------------------------------------- */
/* Private Constants --------------------------------------------------------- */

/*******************************************************START OF ChryUSB_Configurator Generation*************/
#ifndef WBVAL
    #define WBVAL(x) (unsigned char)((x) & 0xFF), (unsigned char)(((x) >> 8) & 0xFF)
#endif/*WBVAL*/

/*!< USBD CONFIG */
#define USBD_VERSION 0x0200
#define USBD_PRODUCT_VERSION 0x0001
#define USBD_VID 0x30cc
#define USBD_PID 0xae03
#define USBD_MAX_POWER 0x64
#define USBD_LANGID_STRING 1033
#define USBD_CONFIG_DESCRIPTOR_SIZE 32

/*!< USBD ENDPOINT CONFIG */

#define USBD_IF0_AL0_EP0_ADDR 0x01
#define USBD_IF0_AL0_EP0_SIZE 0x40
#define USBD_IF0_AL0_EP0_INTERVAL 0x00

#define USBD_IF0_AL0_EP1_ADDR 0x81
#define USBD_IF0_AL0_EP1_SIZE 0x40
#define USBD_IF0_AL0_EP1_INTERVAL 0x00

/*!< USBD HID CONFIG */

/*!< USBD Descriptor */
const unsigned char usbd_descriptor[] =
{
    /********************************************** Device Descriptor */
    0x12,                                       /*!< bLength */
    0x01,                                       /*!< bDescriptorType */
    WBVAL(USBD_VERSION),                        /*!< bcdUSB */
    0xff,                                       /*!< bDeviceClass */
    0xff,                                       /*!< bDeviceSubClass */
    0xff,                                       /*!< bDeviceProtocol */
    0x40,                                       /*!< bMaxPacketSize */
    WBVAL(USBD_VID),                            /*!< idVendor */
    WBVAL(USBD_PID),                            /*!< idProduct */
    WBVAL(USBD_PRODUCT_VERSION),                /*!< bcdDevice */
    0x01,                                       /*!< iManufacturer */
    0x02,                                       /*!< iProduct */
    0x03,                                       /*!< iSerial */
    0x01,                                       /*!< bNumConfigurations */
    /********************************************** Config Descriptor */
    0x09,                                       /*!< bLength */
    0x02,                                       /*!< bDescriptorType */
    WBVAL(USBD_CONFIG_DESCRIPTOR_SIZE),         /*!< wTotalLength */
    0x01,                                       /*!< bNumInterfaces */
    0x01,                                       /*!< bConfigurationValue */
    0x00,                                       /*!< iConfiguration */
    0x80,                                       /*!< bmAttributes */
    USBD_MAX_POWER,                             /*!< bMaxPower */
    /********************************************** Interface 0 Alternate 0 Descriptor */
    0x09,                                       /*!< bLength */
    0x04,                                       /*!< bDescriptorType */
    0x00,                                       /*!< bInterfaceNumber */
    0x00,                                       /*!< bAlternateSetting */
    0x02,                                       /*!< bNumEndpoints */
    0xff,                                       /*!< bInterfaceClass */
    0x00,                                       /*!< bInterfaceSubClass */
    0x00,                                       /*!< bInterfaceProtocol */
    0x00,                                       /*!< iInterface */
    /********************************************** Endpoint 0 Descriptor */
    0x07,                                       /*!< bLength */
    0x05,                                       /*!< bDescriptorType */
    USBD_IF0_AL0_EP0_ADDR,                      /*!< bEndpointAddress */
    0x02,                                       /*!< bmAttributes */
    WBVAL(USBD_IF0_AL0_EP0_SIZE),               /*!< wMaxPacketSize */
    USBD_IF0_AL0_EP0_INTERVAL,                  /*!< bInterval */
    /********************************************** Endpoint 1 Descriptor */
    0x07,                                       /*!< bLength */
    0x05,                                       /*!< bDescriptorType */
    USBD_IF0_AL0_EP1_ADDR,                      /*!< bEndpointAddress */
    0x02,                                       /*!< bmAttributes */
    WBVAL(USBD_IF0_AL0_EP1_SIZE),               /*!< wMaxPacketSize */
    USBD_IF0_AL0_EP1_INTERVAL,                  /*!< bInterval */
    /********************************************** Language ID String Descriptor */
    0x04,                                       /*!< bLength */
    0x03,                                       /*!< bDescriptorType */
    WBVAL(USBD_LANGID_STRING),                  /*!< wLangID0 */
    /********************************************** String 1 Descriptor */
    /* Eastsoft */
    0x12,                                       /*!< bLength */
    0x03,                                       /*!< bDescriptorType */
    0x45, 0x00,                                 /*!< 'E' wcChar0 */
    0x61, 0x00,                                 /*!< 'a' wcChar1 */
    0x73, 0x00,                                 /*!< 's' wcChar2 */
    0x74, 0x00,                                 /*!< 't' wcChar3 */
    0x73, 0x00,                                 /*!< 's' wcChar4 */
    0x6f, 0x00,                                 /*!< 'o' wcChar5 */
    0x66, 0x00,                                 /*!< 'f' wcChar6 */
    0x74, 0x00,                                 /*!< 't' wcChar7 */
    /********************************************** String 2 Descriptor */
    /* Eastsoft CherryUSB winusb demo */
    0x3e,                                       /*!< bLength */
    0x03,                                       /*!< bDescriptorType */
    0x45, 0x00,                                 /*!< 'E' wcChar0 */
    0x61, 0x00,                                 /*!< 'a' wcChar1 */
    0x73, 0x00,                                 /*!< 's' wcChar2 */
    0x74, 0x00,                                 /*!< 't' wcChar3 */
    0x73, 0x00,                                 /*!< 's' wcChar4 */
    0x6f, 0x00,                                 /*!< 'o' wcChar5 */
    0x66, 0x00,                                 /*!< 'f' wcChar6 */
    0x74, 0x00,                                 /*!< 't' wcChar7 */
    0x20, 0x00,                                 /*!< ' ' wcChar8 */
    0x43, 0x00,                                 /*!< 'C' wcChar9 */
    0x68, 0x00,                                 /*!< 'h' wcChar10 */
    0x65, 0x00,                                 /*!< 'e' wcChar11 */
    0x72, 0x00,                                 /*!< 'r' wcChar12 */
    0x72, 0x00,                                 /*!< 'r' wcChar13 */
    0x79, 0x00,                                 /*!< 'y' wcChar14 */
    0x55, 0x00,                                 /*!< 'U' wcChar15 */
    0x53, 0x00,                                 /*!< 'S' wcChar16 */
    0x42, 0x00,                                 /*!< 'B' wcChar17 */
    0x20, 0x00,                                 /*!< ' ' wcChar18 */
    0x77, 0x00,                                 /*!< 'w' wcChar19 */
    0x69, 0x00,                                 /*!< 'i' wcChar20 */
    0x6e, 0x00,                                 /*!< 'n' wcChar21 */
    0x75, 0x00,                                 /*!< 'u' wcChar22 */
    0x73, 0x00,                                 /*!< 's' wcChar23 */
    0x62, 0x00,                                 /*!< 'b' wcChar24 */
    0x20, 0x00,                                 /*!< ' ' wcChar25 */
    0x64, 0x00,                                 /*!< 'd' wcChar26 */
    0x65, 0x00,                                 /*!< 'e' wcChar27 */
    0x6d, 0x00,                                 /*!< 'm' wcChar28 */
    0x6f, 0x00,                                 /*!< 'o' wcChar29 */
    /********************************************** String 3 Descriptor */
    /* 202210201641 */
    0x1a,                                       /*!< bLength */
    0x03,                                       /*!< bDescriptorType */
    0x32, 0x00,                                 /*!< '2' wcChar0 */
    0x30, 0x00,                                 /*!< '0' wcChar1 */
    0x32, 0x00,                                 /*!< '2' wcChar2 */
    0x32, 0x00,                                 /*!< '2' wcChar3 */
    0x31, 0x00,                                 /*!< '1' wcChar4 */
    0x30, 0x00,                                 /*!< '0' wcChar5 */
    0x32, 0x00,                                 /*!< '2' wcChar6 */
    0x30, 0x00,                                 /*!< '0' wcChar7 */
    0x31, 0x00,                                 /*!< '1' wcChar8 */
    0x36, 0x00,                                 /*!< '6' wcChar9 */
    0x34, 0x00,                                 /*!< '4' wcChar10 */
    0x31, 0x00,                                 /*!< '1' wcChar11 */
    /********************************************** Device Qualifier Descriptor */
    0x0a,                                       /*!< bLength */
    0x06,                                       /*!< bDescriptorType */
    WBVAL(USBD_VERSION),                        /*!< bcdUSB */
    0x00,                                       /*!< bDeviceClass */
    0x00,                                       /*!< bDeviceSubClass */
    0x00,                                       /*!< bDeviceProtocol */
    0x40,                                       /*!< bMaxPacketSize0 */
    0x01,                                       /*!< bNumConfigurations */
    0x00,                                       /*!< bReserved */
    0x00
};
/*******************************************************END OF ChryUSB_Configurator Generation*************/

/* Private function prototypes ----------------------------------------------- */
void usbd_winusb_out(uint8_t busid, uint8_t ep, uint32_t nbytes);
void usbd_winusb_in(uint8_t busid, uint8_t ep, uint32_t nbytes);

static struct usbd_endpoint s_winusb_out_ep =
{
    .ep_addr = USBD_IF0_AL0_EP0_ADDR,
    .ep_cb = usbd_winusb_out
};

static struct usbd_endpoint s_winusb_in_ep =
{
    .ep_addr = USBD_IF0_AL0_EP1_ADDR,
    .ep_cb = usbd_winusb_in
};

/* Private Function ---------------------------------------------------------- */
void usbd_event_handler(uint8_t busid, uint8_t event)
{
    switch (event)
    {
        case USBD_EVENT_RESET:
            break;

        case USBD_EVENT_CONNECTED:
            break;

        case USBD_EVENT_DISCONNECTED:
            break;

        case USBD_EVENT_RESUME:
            break;

        case USBD_EVENT_SUSPEND:
            break;

        case USBD_EVENT_CONFIGURED:
        {
            usbd_ep_start_read(0, s_winusb_out_ep.ep_addr, s_read_buffer_winusb, ES_USB_DEV_EP_READ_BUFFER_SIZE);
            s_configed_flag = 1;

            break;
        }

        case USBD_EVENT_SET_REMOTE_WAKEUP:
            break;

        case USBD_EVENT_CLR_REMOTE_WAKEUP:
            break;

        default:
            break;
    }
}

void usbd_winusb_in(uint8_t busid, uint8_t ep, uint32_t nbytes)
{
    int ret;

    if (s_winusb_ep_rx_buf_fill_num)
    {
        ret = usbd_ep_start_write(0, s_winusb_in_ep.ep_addr, s_write_buffer_winusb[s_winusb_ep_buf_tx_index], s_read_num_winusb[s_winusb_ep_buf_tx_index]);

        if (ret == 0)
        {
            s_winusb_ep_rx_buf_fill_num--;
            s_winusb_ep_buf_tx_index = (s_winusb_ep_buf_tx_index + 1) % ES_USB_DEV_EP_WIRTE_BUFFER_NUM;
            s_winusb_ep_tx_flag = ES_USB_DEV_EP_TX_FLAG_BUSY;
        }
        else
            USB_LOG_ERR("winusb ep write fail");
    }
    else
    {
#if ES_USB_SEND_0_PACKEGE_AS_END

        if (nbytes)
        {
            ret = usbd_ep_start_write(0, s_winusb_in_ep.ep_addr, NULL, 0);

            if (ret == 0)
                s_winusb_ep_tx_flag = ES_USB_DEV_EP_TX_FLAG_END;
            else
                USB_LOG_ERR("winusb ep write fail");
        }
        else
        {
            s_winusb_ep_tx_flag = ES_USB_DEV_EP_TX_FLAG_IDLE;
        }

#else
        s_winusb_ep_tx_flag = ES_USB_DEV_EP_TX_FLAG_IDLE;
#endif/*ES_USB_SEND_0_PACKEGE_AS_END*/
    }
}

void usbd_winusb_out(uint8_t busid, uint8_t ep, uint32_t nbytes)
{
    int ret;

    if (nbytes == 0)
        USB_LOG_ERR("ep read buf size too small");

    if (s_winusb_ep_rx_buf_fill_num < ES_USB_DEV_EP_WIRTE_BUFFER_NUM)
    {
        memcpy(s_write_buffer_winusb[s_winusb_ep_buf_fill_index], s_read_buffer_winusb, nbytes);
        s_read_num_winusb[s_winusb_ep_buf_fill_index] = nbytes;
        s_winusb_ep_rx_buf_fill_num++;
        s_winusb_ep_buf_fill_index = (s_winusb_ep_buf_fill_index + 1) % ES_USB_DEV_EP_WIRTE_BUFFER_NUM;
    }

    ret = usbd_ep_start_read(0, s_winusb_out_ep.ep_addr, s_read_buffer_winusb, ES_USB_DEV_EP_READ_BUFFER_SIZE);

    if (ret != 0)
        USB_LOG_ERR("winusb ep read fail");

    if ((s_winusb_ep_tx_flag == ES_USB_DEV_EP_TX_FLAG_IDLE) && (s_winusb_ep_rx_buf_fill_num))
    {
        ret = usbd_ep_start_write(0, s_winusb_in_ep.ep_addr, s_write_buffer_winusb[s_winusb_ep_buf_tx_index], s_read_num_winusb[s_winusb_ep_buf_tx_index]);

        if (ret == 0)
        {
            s_winusb_ep_rx_buf_fill_num--;
            s_winusb_ep_buf_tx_index = (s_winusb_ep_buf_tx_index + 1) % ES_USB_DEV_EP_WIRTE_BUFFER_NUM;
            s_winusb_ep_tx_flag = ES_USB_DEV_EP_TX_FLAG_BUSY;
        }
        else
            USB_LOG_ERR("winusb ep write fail");
    }
}

/**
  * @brief            cdc hid winusb init
  * @pre              none
  * @param[in]        none
  * @retval           none
  */
void cdc_acm_hid_winusb_descriptor_init(void)
{
    usbd_desc_register(0, usbd_descriptor);

    usbd_msosv1_desc_register(0, &s_msosv1_desc);
    usbd_add_interface(0, s_winusb_intf);
    usbd_add_endpoint(0, &s_winusb_out_ep);
    usbd_add_endpoint(0, &s_winusb_in_ep);

    s_configed_flag = 0;
    usbd_initialize(0, 0, usbd_event_handler);

    while (!s_configed_flag);
}

/*******************************************************END OF FILE*************/
