/**********************************************************************************
 *
 * @file    .c
 * @brief   Source file
 *
 * @author  AE Team
 * @note
 *          Change Logs:
 *          Date               Author          Notes
 *          2023-01-17         liuhy           the first version
 *
 * Copyright (C) Shanghai Eastsoft Microelectronics Co. Ltd. All rights reserved.
 *
 * SPDX-License-Identifier: Apache-2.0
 *
 * Licensed under the Apache License, Version 2.0 (the License); you may
 * not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an AS IS BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 *
 **********************************************************************************
 */

/* Includes ------------------------------------------------------------------ */
#include "main.h"
#include "usbd_core.h"
#include "usbd_hid.h"

/* Private Macros ------------------------------------------------------------ */
/*!< custom hid report descriptor size */
#define HID_CUSTOM_REPORT_DESC_SIZE (34)
#define ES_USB_DEV_EP_READ_BUFFER_SIZE (64)
#define ES_USB_DEV_WAIT_TO_TX_MS       (2)
#define ES_USB_SEND_0_PACKEGE_AS_END   1
#define ES_USB_DEV_EP_TX_FLAG_IDLE    (0U)
#define ES_USB_DEV_EP_TX_FLAG_BUSY    (1U)
#define ES_USB_DEV_EP_TX_FLAG_END     (2U)
#define WCID_VENDOR_CODE              0x01

/* Private Variables --------------------------------------------------------- */
/*!< custom hid report descriptor */
static const uint8_t s_hid_custom_report_desc[HID_CUSTOM_REPORT_DESC_SIZE] =
{
    /* USER CODE BEGIN 0 */
    0x06, 0x00, 0xff, /*USAGE_PAGE (Vendor Defined Page 1)*/
    0x09, 0x01,       /*USAGE (Vendor Usage 1)*/
    0xa1, 0x01,       /*COLLECTION (Application)*/
    0x09, 0x01,       /*USAGE (Vendor Usage 1)*/
    0x15, 0x00,       /*LOGICAL_MINIMUM (0)*/
    0x26, 0xff, 0x00, /*LOGICAL_MAXIMUM (255)*/
    0x95, 0x40,       /*REPORT_COUNT (64)*/
    0x75, 0x08,       /*REPORT_SIZE (8)*/
    0x81, 0x02,       /*INPUT (Data,Var,Abs)*/
    /* <___________________________________________________> */
    0x09, 0x01,       /*USAGE (Vendor Usage 1)*/
    0x15, 0x00,       /*LOGICAL_MINIMUM (0)*/
    0x26, 0xff, 0x00, /*LOGICAL_MAXIMUM (255)*/
    0x95, 0x40,       /*REPORT_COUNT (64)*/
    0x75, 0x08,       /*REPORT_SIZE (8)*/
    0x91, 0x02,       /*OUTPUT (Data,Var,Abs)*/
    /* USER CODE END 0 */
    0xC0 /*     END_COLLECTION               */
};

static struct usbd_interface s_hid_custom_intf[1];
static volatile uint8_t s_configed_flag = 0U;
static volatile uint32_t s_last_hid_tx_succes_time_ms = 0U;
static volatile uint32_t s_read_num_hid = 0U;
static volatile uint8_t s_hid_ep_tx_flag = ES_USB_DEV_EP_TX_FLAG_IDLE;
static USB_NOCACHE_RAM_SECTION USB_MEM_ALIGNX uint8_t s_read_buffer_hid[ES_USB_DEV_EP_READ_BUFFER_SIZE];

/* Public Variables ---------------------------------------------------------- */
uint8_t g_usb_hid_idle = 0;
uint8_t g_usb_hid_protocol = 0;
/* Private Constants --------------------------------------------------------- */
/* Private function prototypes ----------------------------------------------- */
/* Private Function ---------------------------------------------------------- */

/*******************************************************START OF ChryUSB_Configurator Generation*************/
#ifndef WBVAL
    #define WBVAL(x) (unsigned char)((x) & 0xFF), (unsigned char)(((x) >> 8) & 0xFF)
#endif/*WBVAL*/

/*!< USBD CONFIG */
#define USBD_VERSION 0x0200
#define USBD_PRODUCT_VERSION 0x0001
#define USBD_VID 0x30cc
#define USBD_PID 0xae02
#define USBD_MAX_POWER 0xfa
#define USBD_LANGID_STRING 1033
#define USBD_CONFIG_DESCRIPTOR_SIZE 41

/*!< USBD ENDPOINT CONFIG */

#define USBD_IF0_AL0_EP0_ADDR 0x01
#define USBD_IF0_AL0_EP0_SIZE 0x40
#define USBD_IF0_AL0_EP0_INTERVAL 0x01

#define USBD_IF0_AL0_EP1_ADDR 0x81
#define USBD_IF0_AL0_EP1_SIZE 0x40
#define USBD_IF0_AL0_EP1_INTERVAL 0x01

/*!< USBD HID CONFIG */
#define USBD_HID_VERSION 0x0111
#define USBD_HID_COUNTRY_CODE 0
#define USBD_IF0_AL0_HID_REPORT_DESC_SIZE 2

/*!< USBD Descriptor */
const unsigned char usbd_descriptor[] =
{
    /********************************************** Device Descriptor */
    0x12,                                       /*!< bLength */
    0x01,                                       /*!< bDescriptorType */
    WBVAL(USBD_VERSION),                        /*!< bcdUSB */
    0x00,                                       /*!< bDeviceClass */
    0x00,                                       /*!< bDeviceSubClass */
    0x00,                                       /*!< bDeviceProtocol */
    0x40,                                       /*!< bMaxPacketSize */
    WBVAL(USBD_VID),                            /*!< idVendor */
    WBVAL(USBD_PID),                            /*!< idProduct */
    WBVAL(USBD_PRODUCT_VERSION),                /*!< bcdDevice */
    0x01,                                       /*!< iManufacturer */
    0x02,                                       /*!< iProduct */
    0x03,                                       /*!< iSerial */
    0x01,                                       /*!< bNumConfigurations */
    /********************************************** Config Descriptor */
    0x09,                                       /*!< bLength */
    0x02,                                       /*!< bDescriptorType */
    WBVAL(USBD_CONFIG_DESCRIPTOR_SIZE),         /*!< wTotalLength */
    0x01,                                       /*!< bNumInterfaces */
    0x01,                                       /*!< bConfigurationValue */
    0x00,                                       /*!< iConfiguration */
    0x80,                                       /*!< bmAttributes */
    USBD_MAX_POWER,                             /*!< bMaxPower */
    /********************************************** Interface 0 Alternate 0 Descriptor */
    0x09,                                       /*!< bLength */
    0x04,                                       /*!< bDescriptorType */
    0x00,                                       /*!< bInterfaceNumber */
    0x00,                                       /*!< bAlternateSetting */
    0x02,                                       /*!< bNumEndpoints */
    0x03,                                       /*!< bInterfaceClass */
    0x00,                                       /*!< bInterfaceSubClass */
    0x00,                                       /*!< bInterfaceProtocol */
    0x00,                                       /*!< iInterface */
    /********************************************** Class Specific Descriptor of HID */
    0x09,                                       /*!< bLength */
    0x21,                                       /*!< bDescriptorType */
    WBVAL(USBD_HID_VERSION),                    /*!< bcdHID */
    USBD_HID_COUNTRY_CODE,                      /*!< bCountryCode */
    0x01,                                       /*!< bNumDescriptors */
    0x22,                                       /*!< bDescriptorType */
    WBVAL(HID_CUSTOM_REPORT_DESC_SIZE),   /*!< wItemLength */
    /********************************************** Endpoint 0 Descriptor */
    0x07,                                       /*!< bLength */
    0x05,                                       /*!< bDescriptorType */
    USBD_IF0_AL0_EP0_ADDR,                      /*!< bEndpointAddress */
    0x03,                                       /*!< bmAttributes */
    WBVAL(USBD_IF0_AL0_EP0_SIZE),               /*!< wMaxPacketSize */
    USBD_IF0_AL0_EP0_INTERVAL,                  /*!< bInterval */
    /********************************************** Endpoint 1 Descriptor */
    0x07,                                       /*!< bLength */
    0x05,                                       /*!< bDescriptorType */
    USBD_IF0_AL0_EP1_ADDR,                      /*!< bEndpointAddress */
    0x03,                                       /*!< bmAttributes */
    WBVAL(USBD_IF0_AL0_EP1_SIZE),               /*!< wMaxPacketSize */
    USBD_IF0_AL0_EP1_INTERVAL,                  /*!< bInterval */
    /********************************************** Language ID String Descriptor */
    0x04,                                       /*!< bLength */
    0x03,                                       /*!< bDescriptorType */
    WBVAL(USBD_LANGID_STRING),                  /*!< wLangID0 */
    /********************************************** String 1 Descriptor */
    /* Eastsoft */
    0x12,                                       /*!< bLength */
    0x03,                                       /*!< bDescriptorType */
    0x45, 0x00,                                 /*!< 'E' wcChar0 */
    0x61, 0x00,                                 /*!< 'a' wcChar1 */
    0x73, 0x00,                                 /*!< 's' wcChar2 */
    0x74, 0x00,                                 /*!< 't' wcChar3 */
    0x73, 0x00,                                 /*!< 's' wcChar4 */
    0x6f, 0x00,                                 /*!< 'o' wcChar5 */
    0x66, 0x00,                                 /*!< 'f' wcChar6 */
    0x74, 0x00,                                 /*!< 't' wcChar7 */
    /********************************************** String 2 Descriptor */
    /* Eastsoft CherryUSB HID Custom demo */
    0x46,                                       /*!< bLength */
    0x03,                                       /*!< bDescriptorType */
    0x45, 0x00,                                 /*!< 'E' wcChar0 */
    0x61, 0x00,                                 /*!< 'a' wcChar1 */
    0x73, 0x00,                                 /*!< 's' wcChar2 */
    0x74, 0x00,                                 /*!< 't' wcChar3 */
    0x73, 0x00,                                 /*!< 's' wcChar4 */
    0x6f, 0x00,                                 /*!< 'o' wcChar5 */
    0x66, 0x00,                                 /*!< 'f' wcChar6 */
    0x74, 0x00,                                 /*!< 't' wcChar7 */
    0x20, 0x00,                                 /*!< ' ' wcChar8 */
    0x43, 0x00,                                 /*!< 'C' wcChar9 */
    0x68, 0x00,                                 /*!< 'h' wcChar10 */
    0x65, 0x00,                                 /*!< 'e' wcChar11 */
    0x72, 0x00,                                 /*!< 'r' wcChar12 */
    0x72, 0x00,                                 /*!< 'r' wcChar13 */
    0x79, 0x00,                                 /*!< 'y' wcChar14 */
    0x55, 0x00,                                 /*!< 'U' wcChar15 */
    0x53, 0x00,                                 /*!< 'S' wcChar16 */
    0x42, 0x00,                                 /*!< 'B' wcChar17 */
    0x20, 0x00,                                 /*!< ' ' wcChar18 */
    0x48, 0x00,                                 /*!< 'H' wcChar19 */
    0x49, 0x00,                                 /*!< 'I' wcChar20 */
    0x44, 0x00,                                 /*!< 'D' wcChar21 */
    0x20, 0x00,                                 /*!< ' ' wcChar22 */
    0x43, 0x00,                                 /*!< 'C' wcChar23 */
    0x75, 0x00,                                 /*!< 'u' wcChar24 */
    0x73, 0x00,                                 /*!< 's' wcChar25 */
    0x74, 0x00,                                 /*!< 't' wcChar26 */
    0x6f, 0x00,                                 /*!< 'o' wcChar27 */
    0x6d, 0x00,                                 /*!< 'm' wcChar28 */
    0x20, 0x00,                                 /*!< ' ' wcChar29 */
    0x64, 0x00,                                 /*!< 'd' wcChar30 */
    0x65, 0x00,                                 /*!< 'e' wcChar31 */
    0x6d, 0x00,                                 /*!< 'm' wcChar32 */
    0x6f, 0x00,                                 /*!< 'o' wcChar33 */
    /********************************************** String 3 Descriptor */
    /* 202210201641 */
    0x1a,                                       /*!< bLength */
    0x03,                                       /*!< bDescriptorType */
    0x32, 0x00,                                 /*!< '2' wcChar0 */
    0x30, 0x00,                                 /*!< '0' wcChar1 */
    0x32, 0x00,                                 /*!< '2' wcChar2 */
    0x32, 0x00,                                 /*!< '2' wcChar3 */
    0x31, 0x00,                                 /*!< '1' wcChar4 */
    0x30, 0x00,                                 /*!< '0' wcChar5 */
    0x32, 0x00,                                 /*!< '2' wcChar6 */
    0x30, 0x00,                                 /*!< '0' wcChar7 */
    0x31, 0x00,                                 /*!< '1' wcChar8 */
    0x36, 0x00,                                 /*!< '6' wcChar9 */
    0x34, 0x00,                                 /*!< '4' wcChar10 */
    0x31, 0x00,                                 /*!< '1' wcChar11 */
    /********************************************** Device Qualifier Descriptor */
    0x0a,                                       /*!< bLength */
    0x06,                                       /*!< bDescriptorType */
    WBVAL(USBD_VERSION),                        /*!< bcdUSB */
    0x00,                                       /*!< bDeviceClass */
    0x00,                                       /*!< bDeviceSubClass */
    0x00,                                       /*!< bDeviceProtocol */
    0x40,                                       /*!< bMaxPacketSize0 */
    0x01,                                       /*!< bNumConfigurations */
    0x00,                                       /*!< bReserved */
    0x00
};

/*!< USBD HID REPORT 0 Descriptor */
const unsigned char usbd_hid_0_report_descriptor[USBD_IF0_AL0_HID_REPORT_DESC_SIZE] =
{
    0x00,
    0xC0    /* END_COLLECTION */
};
/*******************************************************END OF ChryUSB_Configurator Generation*************/

/*!< endpoint call back */

static struct usbd_endpoint s_custom_in_ep =
{
    .ep_addr = USBD_IF0_AL0_EP1_ADDR,
    .ep_cb = NULL
};

static struct usbd_endpoint s_custom_out_ep =
{
    .ep_addr = USBD_IF0_AL0_EP0_ADDR,
    .ep_cb = NULL
};

void usbd_event_handler(uint8_t busid, uint8_t event)
{
    switch (event)
    {
        case USBD_EVENT_RESET:
            break;

        case USBD_EVENT_CONNECTED:
            break;

        case USBD_EVENT_DISCONNECTED:
            break;

        case USBD_EVENT_RESUME:
            break;

        case USBD_EVENT_SUSPEND:
            break;

        case USBD_EVENT_CONFIGURED:
        {
            s_configed_flag = 1;

            break;
        }

        case USBD_EVENT_SET_REMOTE_WAKEUP:
            break;

        case USBD_EVENT_CLR_REMOTE_WAKEUP:
            break;

        default:
            break;
    }
}

/**
  * @brief            cdc hid winusb init
  * @pre              none
  * @param[in]        none
  * @retval           none
  */
void cdc_acm_hid_winusb_descriptor_init(void)
{
    usbd_desc_register(0, usbd_descriptor);

    usbd_add_interface(0, usbd_hid_init_intf(0, s_hid_custom_intf, s_hid_custom_report_desc, HID_CUSTOM_REPORT_DESC_SIZE));
    usbd_add_endpoint(0, &s_custom_in_ep);
    usbd_add_endpoint(0, &s_custom_out_ep);

    s_configed_flag = 0;
    usbd_initialize(0, 0, usbd_event_handler);

    while (!s_configed_flag);
}

void usb_dev_loop(void)
{
    int ret;

    /* hid ep handle*/
    if (s_hid_ep_tx_flag == ES_USB_DEV_EP_TX_FLAG_BUSY)
    {
        ret = usbd_write_packet(s_custom_in_ep.ep_addr, s_read_buffer_hid, s_read_num_hid);

        if (ret > 0)
        {
            s_hid_ep_tx_flag = ES_USB_DEV_EP_TX_FLAG_IDLE;
            s_last_hid_tx_succes_time_ms = md_get_tick();
        }
    }

    if ((((md_get_tick()) - s_last_hid_tx_succes_time_ms) > ES_USB_DEV_WAIT_TO_TX_MS) || (s_hid_ep_tx_flag != ES_USB_DEV_EP_TX_FLAG_BUSY))
    {
        ret = usbd_read_packet(s_custom_out_ep.ep_addr, s_read_buffer_hid, ES_USB_DEV_EP_READ_BUFFER_SIZE);

        if (ret > 0)
        {
            s_read_num_hid = ret;
            s_hid_ep_tx_flag = ES_USB_DEV_EP_TX_FLAG_BUSY;
        }
    }
}

void usbd_hid_set_idle(uint8_t busid, uint8_t intf, uint8_t report_id, uint8_t duration)
{
    g_usb_hid_idle = duration;
}

uint8_t usbd_hid_get_idle(uint8_t busid, uint8_t intf, uint8_t report_id)
{
    return g_usb_hid_idle;
}

void usbd_hid_set_protocol(uint8_t busid, uint8_t intf, uint8_t protocol)
{
    g_usb_hid_protocol = protocol;
}

uint8_t usbd_hid_get_protocol(uint8_t busid, uint8_t intf)
{
	return g_usb_hid_protocol;
}

/*******************************************************END OF FILE*************/
