/**
  *********************************************************************************
  *
  * @file    .c
  * @brief  Source file
  *
  * @version V1.0
  * @date    26 Jun 2019
  * @author  AE Team
  * @note
  *          Change Logs:
  *          Date            Author          Notes
  *          26 Jun 2019     AE Team         The first version
  *
  * Copyright (C) Shanghai Eastsoft Microelectronics Co. Ltd. All rights reserved.
  *
  * SPDX-License-Identifier: Apache-2.0
  *
  * Licensed under the Apache License, Version 2.0 (the License); you may
  * not use this file except in compliance with the License.
  * You may obtain a copy of the License at
  *
  * www.apache.org/licenses/LICENSE-2.0
  *
  * Unless required by applicable law or agreed to in writing, software
  * distributed under the License is distributed on an AS IS BASIS, WITHOUT
  * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
  * See the License for the specific language governing permissions and
  * limitations under the License.
  **********************************************************************************
  */

/* Includes ------------------------------------------------------------------*/
#include "usbh_msc_diskio.h"
#include "string.h"
#include "stdio.h"
#include "usb_config.h"
#include "usb_hc.h"
#include "usbh_msc.h"
#include "usb_scsi.h"
#include "boot_conf.h"
#include "md_msc.h"
#include "md_crc.h"

/* Private Macros ------------------------------------------------------------ */

#define ES_FATFS_BASE_PATH_MAX_LEN (32)

/* Private Constants --------------------------------------------------------- */

/* Private function prototypes -----------------------------------------------*/
DSTATUS usbh_msc_fatfs_drv_init(BYTE);
DSTATUS usbh_msc_fatfs_drv_stat(BYTE);
DRESULT usbh_msc_fatfs_drv_read(BYTE, BYTE *, DWORD, UINT);
#if  _USE_WRITE
    DRESULT usbh_msc_fatfs_drv_write(BYTE, const BYTE *, DWORD, UINT);
#endif /*_USE_WRITE*/
#if  _USE_IOCTL
    DRESULT usbh_msc_fatfs_drv_ioctl(BYTE, BYTE, void *);
#endif /*_USE_IOCTL*/
/* Private Variables ---------------------------------------------------------*/

static FATFS s_es_fatfs;        /* File system object for NAND flash volume */
static uint8_t s_es_fatfs_path[ES_FATFS_BASE_PATH_MAX_LEN];
static struct usbh_msc *s_es_usb_msc_class1 = NULL;

static const Diskio_drvTypeDef s_usbh_msc_fatfs_drv =
{
    usbh_msc_fatfs_drv_init,
    usbh_msc_fatfs_drv_stat,
    usbh_msc_fatfs_drv_read,

#if  _USE_WRITE
    usbh_msc_fatfs_drv_write,
#endif /*_USE_WRITE*/

#if  _USE_IOCTL
    usbh_msc_fatfs_drv_ioctl,
#endif /*_USE_IOCTL*/
};

/* Public Variables ---------------------------------------------------------- */
volatile uint8_t g_state_msc_usbdev_connect = 0U;

/* Private Function ---------------------------------------------------------*/
/**
  * @brief  Initializes a Drive
  * @param  NONE
  * @retval DSTATUS: Operation status
  */
DSTATUS usbh_msc_fatfs_drv_init(BYTE lun)
{
    if (s_es_usb_msc_class1 == NULL)
        return RES_ERROR;
    else
        return RES_OK;
}

/**
  * @brief  Gets Disk Status
  * @param  parm_num
  * @param  param
  * @retval DSTATUS: Operation status
  */
DSTATUS usbh_msc_fatfs_drv_stat(BYTE lun)
{
    if (s_es_usb_msc_class1 == NULL)
        return RES_ERROR;
    else
        return RES_OK;
}

/**
  * @brief  Reads Data
  * @param  *buff: Data buffer to store read data
  * @param  sector: Sector address (LBA)
  * @param  count: Number of sectors to read (1..128)
  * @retval DRESULT: Operation result
  */
DRESULT usbh_msc_fatfs_drv_read(BYTE lun, BYTE *buff, DWORD sector, UINT count)
{
    int ret = -1;

    if (s_es_usb_msc_class1)
        ret = usbh_msc_scsi_read10(s_es_usb_msc_class1, sector, buff, count);

    if (ret < 0)
    {
        printf("scsi_read10 error,ret:%d\r\n", ret);
        return RES_ERROR;
    }

    return RES_OK;
}

/**
  * @brief  Writes Sector(s)
  * @param  *buff: Data to be written
  * @param  sector: Sector address (LBA)
  * @param  count: Number of sectors to write (1..128)
  * @retval DRESULT: Operation result
  */
#if _USE_WRITE

DRESULT usbh_msc_fatfs_drv_write(BYTE lun, const BYTE *buff, DWORD sector, UINT count)
{
    int ret = -1;

    if (s_es_usb_msc_class1)
        ret = usbh_msc_scsi_write10(s_es_usb_msc_class1, sector, buff, count);

    if (ret < 0)
    {
        printf("scsi_write10 error,ret:%d\r\n", ret);
        return RES_ERROR;
    }

    return RES_OK;
}
#endif /*_USE_WRITE*/

#if  _USE_IOCTL
/**
 * @brief  I/O control operation
 * @param  cmd: Control code
 * @param  *buff: Buffer to send/receive control data
 * @retval DRESULT: Operation result
 */
DRESULT usbh_msc_fatfs_drv_ioctl(BYTE lun, BYTE cmd, void *buff)
{
    DRESULT res = RES_ERROR;

    if (s_es_usb_msc_class1 == NULL)
        return RES_ERROR;

    switch (cmd)
    {
        /* Make sure that no pending write process */
        case CTRL_SYNC:
            res = RES_OK;
            break;

        /* Get number of sectors on the disk (DWORD) */
        case GET_SECTOR_COUNT :
            *(DWORD *)buff = s_es_usb_msc_class1->blocknum;
            res = RES_OK;
            break;

        /* Get R/W sector size (WORD) */
        case GET_SECTOR_SIZE:
            *(WORD *)buff = s_es_usb_msc_class1->blocksize;
            res = RES_OK;
            break;

        /* Get erase block size in unit of sector (DWORD) */
        case GET_BLOCK_SIZE:
            *(DWORD *)buff = s_es_usb_msc_class1->blocksize;
            res = RES_OK;
            break;

        /* Informs the data on the block of sectors can be erased. */
        case CTRL_TRIM:
            res = RES_ERROR;
            break;

        default:
            res = RES_PARERR;
            break;
    }

    return res;
}

#endif /*_USE_IOCTL*/

void es_fatfs_error_handler(void)
{
    printf("[es_fatfs_error]");
}

void es_usb_msc_fatfs_register(void)
{
    /* Regeister the disk I/O driver */
    if (FATFS_LinkDriverEx(&s_usbh_msc_fatfs_drv, (char *)s_es_fatfs_path, 0) == 0)
    {
        /* Register the file system object to the FatFs module */
        if (f_mount(&s_es_fatfs, (const TCHAR *)s_es_fatfs_path, 1) != FR_OK)
        {
            /* FatFs Initialization Error */
            es_fatfs_error_handler();

            /*nUnlink the spi Flash disk I/O driver */
            FATFS_UnLinkDriverEx((char *)s_es_fatfs_path, 0);
        }
        else
        {
            printf("\r\nfatfs mount succes , path :%s\r\n", s_es_fatfs_path);
        }
    }
}

void es_try_fatfs_unregister(void)
{
    if (f_mount((void *)0, (const TCHAR *)s_es_fatfs_path, 0) != FR_OK)
    {
        /* FatFs Initialization Error */
        es_fatfs_error_handler();
    }

    printf("fatfs unmount succes , path :%s", s_es_fatfs_path);

    /*nUnlink the spi Flash disk I/O driver */
    FATFS_UnLinkDriverEx((char *)s_es_fatfs_path, 0);
}

uint8_t es_iap_program_words(uint32_t addr, uint8_t *buf, uint32_t len, uint8_t en)
{
    uint8_t ret = 0U;

    if (en)
    {
        if (md_msc_code_erase_page(addr, ~addr, 0) == MD_ERROR)
            ret = 1;
    }

    if (md_msc_code_program_words(addr, ~addr, buf, len, 0) != 0)
        ret = 2;

    if (ret)
        printf("iap write error\r\n");

    return ret;
}

void jump2app(void)
{
    __disable_irq();

    ald_usb_int_unregister();

    MD_SYSCFG_UNLOCK();
    md_syscfg_set_cpu_boot_addr(FAT_AND_NEW_APP_FLASH_S);
    MD_SYSCFG_LOCK();

    __enable_irq();

    md_rmu_reset_system();
}

void es_update_app(void)
{
    FIL fp[1];
    FRESULT res;
    FILINFO fno;
    unsigned int get_num;
    const char *new_app_path = "0:/demo_0x10000_crc.bin";
    uint8_t check_pass_flag = 1U;
    uint32_t i, temp_file_read_buf[16];
    uint32_t crc_value;

    res = f_stat(new_app_path, &fno);

    if (res != FR_OK)
        return;

    if (fno.fsize & 0x3)
        return;

    res = f_open(fp, new_app_path, FA_READ);

    if (res == FR_OK)
    {
        res = f_lseek(fp, fno.fsize - 8);

        if (res == FR_OK)
        {
            res = f_read(fp, temp_file_read_buf, 8, &get_num);

            if (res == FR_OK)
            {
                if (temp_file_read_buf[1] != (fno.fsize - 8))
                    check_pass_flag = 0;

                if (fno.fsize & 0x3)
                    check_pass_flag = 0;

                crc_value = temp_file_read_buf[0];

                if (check_pass_flag)
                {
                    check_pass_flag = 0;

                    res = f_lseek(fp, 0);

                    if (res == FR_OK)
                    {
                        i = 0;

                        while (1)
                        {
                            res = f_read(fp, temp_file_read_buf, 64, &get_num);

                            if ((res != FR_OK) || (get_num == 0))
                                break;

                            if (res == FR_OK)
                            {
                                if (memcmp((const char *)temp_file_read_buf, (const char *)(FAT_AND_NEW_APP_FLASH_S + i), ((i + get_num) >= (fno.fsize - 8)) ? (fno.fsize - 8 - i) : (get_num)))
                                    break;

                                i += get_num;
                            }
                        }

                        if (i >= (fno.fsize - 8))
                        {
                            printf("equal app , not need to update app\r\n");

                            jump2app();
                        }
                        else
                            check_pass_flag = 1;

                        if (check_pass_flag)
                        {
                            md_crc_init_t g_crc_init;

                            md_crc_init_struct(&g_crc_init);
                            g_crc_init.mode = MD_CRC_MODE_32;
                            g_crc_init.len = MD_CRC_DATASIZE_32;
                            g_crc_init.order = MD_CRC_BYTORD_LOW;
                            g_crc_init.seed = 0xFFFFFFFF;
                            g_crc_init.chs_inv = ENABLE;
                            g_crc_init.chs_rev = ENABLE;
                            g_crc_init.data_inv = DISABLE;
                            g_crc_init.data_rev = ENABLE;
                            md_crc_init(&g_crc_init);

                            uint32_t i, end;

                            end = (fno.fsize - 8);/*crc value + code length*/

                            res = f_lseek(fp, 0);

                            if (res == FR_OK)
                            {
                                for (i = 0; i < end ; i += 4)
                                {
                                    if ((i & 0x3F) == 0)
                                    {
                                        res = f_read(fp, temp_file_read_buf, 64, &get_num);

                                        if ((res != FR_OK) || (get_num == 0))
                                            break;
                                    }

                                    md_crc_write_data(CRC, temp_file_read_buf[(i & 0x3F) >> 2]);
                                }

                                if (crc_value != md_crc_get_check_result(CRC))
                                {
                                    printf("new app crc err %08X %08X\r\n", crc_value, md_crc_get_check_result(CRC));
                                    check_pass_flag = 0;
                                }
                                else
                                    printf("new app crc ok\r\n");
                            }

                        }

                    }

                    if (check_pass_flag)
                    {
                        printf("get new app\r\n");

                        res = f_lseek(fp, 0);

                        if (res == FR_OK)
                        {
                            i = 0;

                            while (1)
                            {
                                res = f_read(fp, temp_file_read_buf, 64, &get_num);

                                if ((res != FR_OK) || (get_num == 0))
                                    break;

                                if ((i + get_num) > (FAT_AND_NEW_APP_FLASH_E - FAT_AND_NEW_APP_FLASH_S))
                                    get_num = ((FAT_AND_NEW_APP_FLASH_E - FAT_AND_NEW_APP_FLASH_S) - i + 3) & 0xFFFFFFFC;

                                if (res == FR_OK)
                                {
                                    if ((i % DISK_BLOCK_SIZE) == 0)
                                        es_iap_program_words(FAT_AND_NEW_APP_FLASH_S + i, (uint8_t *)temp_file_read_buf, get_num, 1);
                                    else
                                        es_iap_program_words(FAT_AND_NEW_APP_FLASH_S + i, (uint8_t *)temp_file_read_buf, get_num, 0);

                                    i += get_num;
                                }

                                if (i > (FAT_AND_NEW_APP_FLASH_E - FAT_AND_NEW_APP_FLASH_S))
                                    break;
                            }

                            jump2app();
                        }
                        else
                            es_fatfs_error_handler();
                    }
                }
            }
            else
                es_fatfs_error_handler();
        }
        else
            es_fatfs_error_handler();

        res = f_close(fp);

        if (res != FR_OK)
            es_fatfs_error_handler();
    }
    else
        printf("file open fail\r\n");
}

void usbh_msc_run(struct usbh_msc *msc_class)
{
    g_state_msc_usbdev_connect = 1;

    if (s_es_usb_msc_class1 == NULL)
    {
        s_es_usb_msc_class1 = msc_class;
        es_usb_msc_fatfs_register();

        es_update_app();
    }
    else
        printf("only support 1 usb_msc_device");
}

void usbh_msc_stop(struct usbh_msc *msc_class)
{
    es_try_fatfs_unregister();
    s_es_usb_msc_class1 = NULL;
}

/************* (C) COPYRIGHT Eastsoft Microelectronics *****END OF FILE****/
