/**
  *********************************************************************************
  *
  * @file    .c
  * @brief  Source file
  *
  * @version V1.0
  * @date    26 Jun 2019
  * @author  AE Team
  * @note
  *          Change Logs:
  *          Date            Author          Notes
  *          26 Jun 2019     AE Team         The first version
  *
  * Copyright (C) Shanghai Eastsoft Microelectronics Co. Ltd. All rights reserved.
  *
  * SPDX-License-Identifier: Apache-2.0
  *
  * Licensed under the Apache License, Version 2.0 (the License); you may
  * not use this file except in compliance with the License.
  * You may obtain a copy of the License at
  *
  * www.apache.org/licenses/LICENSE-2.0
  *
  * Unless required by applicable law or agreed to in writing, software
  * distributed under the License is distributed on an AS IS BASIS, WITHOUT
  * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
  * See the License for the specific language governing permissions and
  * limitations under the License.
  **********************************************************************************
  */

/* Includes ------------------------------------------------------------------*/
#include "ram_rom_diskio.h"
#include "boot_conf.h"
#include "md_crc.h"
#include "md_msc.h"
#include "md_syscfg.h"
#include "ald_usb.h"
#include <string.h>

/* Private Macros ------------------------------------------------------------ */
#define ES_FATFS_BASE_PATH_MAX_LEN (64)

#ifdef ES_USE_TINY_PRINTF/*Reduce ROM consumption*/
    #include "printf.h"
    #define printf   printf_
#endif /*ES_USE_TINY_PRINTF*/

/* Private Variables ---------------------------------------------------------*/
static FATFS s_es_fatfs;        /* File system object for flash */
static uint8_t s_es_fatfs_path[ES_FATFS_BASE_PATH_MAX_LEN];
static uint8_t s_es_fatfs_new_app_path[ES_FATFS_BASE_PATH_MAX_LEN];
uint8_t g_tmp_buf_page[DISK_BLOCK_SIZE];

/* Private Constants --------------------------------------------------------- */

/* Private function prototypes -----------------------------------------------*/
DSTATUS spi_flash_init(BYTE);
DSTATUS spi_flash_stat(BYTE);
DRESULT spi_flash_read(BYTE, BYTE *, DWORD, UINT);
#if  _USE_WRITE
    DRESULT spi_flash_write(BYTE, const BYTE *, DWORD, UINT);
#endif/*_USE_WRITE*/
#if  _USE_IOCTL
    DRESULT spi_flash_ioctl(BYTE, BYTE, void *);
#endif/*_USE_IOCTL*/
/* Public Variables ---------------------------------------------------------- */
const  Diskio_drvTypeDef g_spi_flash_drv =
{
    spi_flash_init,
    spi_flash_stat,
    spi_flash_read,

#if  _USE_WRITE
    spi_flash_write,
#endif/*_USE_WRITE*/

#if  _USE_IOCTL
    spi_flash_ioctl,
#endif/*_USE_IOCTL*/
};

#if ES_FAT_USE_RAM_PAGE
    uint8_t g_es_fat_ram_page[ES_FAT_RAM_PAGE_NUM * DISK_BLOCK_SIZE];
#endif/*_USE_IOCTL*/
/* Private Functions ---------------------------------------------------------*/
void jump2app(void)
{
    __disable_irq();

    ald_usb_int_unregister();

    MD_SYSCFG_UNLOCK();
    md_syscfg_set_cpu_boot_addr(FAT_AND_NEW_APP_FLASH_S);
    MD_SYSCFG_LOCK();

    __enable_irq();

    md_rmu_reset_system();
}

uint8_t es_iap_program_words(uint32_t addr, uint8_t *buf, uint32_t len, uint8_t en)
{
    uint8_t ret = 0U;
    uint32_t i;

    if (en)
    {
        for (i = 0; i < len; i += DISK_BLOCK_SIZE)
        {
            if (md_msc_code_erase_page((addr + i), ~(addr + i), 0) == MD_ERROR)
            {
                ret = 1;
                break;
            }
        }
    }

    if (md_msc_code_program_words(addr, ~addr, buf, len, 0) != 0)
        ret = 2;

    if (ret)
        printf("iap write error\r\n");

    return ret;
}

/**
  * @brief  Initializes a Drive
  * @param  NONE
  * @retval DSTATUS: Operation status
  */
DSTATUS spi_flash_init(BYTE lun)
{
    return RES_OK;
}

/**
  * @brief  Gets Disk Status
  * @param  parm_num
  * @param  param
  * @retval DSTATUS: Operation status
  */
DSTATUS spi_flash_stat(BYTE lun)
{
    return RES_OK;
}

void fs_flash_read(uint8_t *buf, uint32_t sector, uint32_t num_block)
{
    uint32_t i, j, k, offset, sector_now;

    k = 0;

    for (i = 0; i < num_block; i++)
    {
        sector_now = sector + i;

        if (sector_now < ES_FAT_RAM_PAGE_NUM)
        {
            offset = DISK_BLOCK_SIZE * sector_now;

            for (j = 0; j < DISK_BLOCK_SIZE; j++)
                buf[k++] = g_es_fat_ram_page[offset + j];
        }
        else
        {
            offset = SECTION_ADDR(sector_now - ES_FAT_RAM_PAGE_NUM);

            for (j = 0; j < DISK_BLOCK_SIZE; j++)
                buf[k++] = *((uint8_t *)(offset + j));
        }
    }
}

/**
  * @brief  Reads Data
  * @param  *buff: Data buffer to store read data
  * @param  sector: Sector address (LBA)
  * @param  count: Number of sectors to read (1..128)
  * @retval DRESULT: Operation result
  */
DRESULT spi_flash_read(BYTE lun, BYTE *buff, DWORD sector, UINT count)
{
    fs_flash_read(buff, (uint32_t)(sector), count);

    return RES_OK;
}

void fs_flash_write(uint8_t *buf, uint32_t sector, uint32_t num_block)
{
    uint32_t i, j, k, offset, sector_now;

    k = 0;

    for (i = 0; i < num_block; i++)
    {
        sector_now = sector + i;

        if (sector_now < ES_FAT_RAM_PAGE_NUM)
        {
            offset = DISK_BLOCK_SIZE * sector_now;

            for (j = 0; j < DISK_BLOCK_SIZE; j++)
                g_es_fat_ram_page[offset + j] = buf[k++];
        }
        else
        {
            es_iap_program_words(SECTION_ADDR(sector_now - ES_FAT_RAM_PAGE_NUM), buf + k, DISK_BLOCK_SIZE, ENABLE);
            k += DISK_BLOCK_SIZE;
        }
    }
}

/**
  * @brief  Writes Sector(s)
  * @param  *buff: Data to be written
  * @param  sector: Sector address (LBA)
  * @param  count: Number of sectors to write (1..128)
  * @retval DRESULT: Operation result
  */
#if _USE_WRITE
DRESULT spi_flash_write(BYTE lun, const BYTE *buff, DWORD sector, UINT count)
{
    fs_flash_write((uint8_t *)buff, sector, count);

    return RES_OK;
}
#endif/*_USE_WRITE*/

#if  _USE_IOCTL
/**
 * @brief  I/O control operation
 * @param  cmd: Control code
 * @param  *buff: Buffer to send/receive control data
 * @retval DRESULT: Operation result
 */
DRESULT spi_flash_ioctl(BYTE lun, BYTE cmd, void *buff)
{
    DRESULT res = RES_ERROR;

    switch (cmd)
    {
        /* Make sure that no pending write process */
        case CTRL_SYNC:
            res = RES_OK;
            break;

        /* Get number of sectors on the disk (DWORD) */
        case GET_SECTOR_COUNT :
            *(DWORD *)buff = DISK_BLOCK_NR;
            res = RES_OK;
            break;

        /* Get R/W sector size (WORD) */
        case GET_SECTOR_SIZE:
            *(WORD *)buff = DISK_BLOCK_SIZE;
            res = RES_OK;
            break;

        /* Get erase block size in unit of sector (DWORD) */
        case GET_BLOCK_SIZE:
            *(DWORD *)buff = 1;
            res = RES_OK;
            break;

        /* Informs the data on the block of sectors can be erased. */
        case CTRL_TRIM:
            res = RES_OK;
            break;

        default:
            res = RES_PARERR;
            break;
    }

    return res;
}

#endif/*_USE_IOCTL*/

void app_flash_page_write(uint32_t pBin_start_page)
{
    uint32_t i, page = 0U;

    if (pBin_start_page == 0)
    {
        return;
    }

    if (pBin_start_page == ES_FAT_RAM_PAGE_NUM)
        return;

    page = (FAT_AND_NEW_APP_FLASH_E - FAT_AND_NEW_APP_FLASH_S + DISK_BLOCK_SIZE - 1) / DISK_BLOCK_SIZE;

    if (pBin_start_page > ES_FAT_RAM_PAGE_NUM)
    {
        for (i = 0; i < page; i++)
        {
            fs_flash_read(g_tmp_buf_page, i + pBin_start_page, 1);
            fs_flash_write(g_tmp_buf_page, i + ES_FAT_RAM_PAGE_NUM, 1);
        }
    }

    if (pBin_start_page < ES_FAT_RAM_PAGE_NUM)
    {
        for (i = page - 1; i < 0x3FFFFFFF; i--)
        {
            fs_flash_read(g_tmp_buf_page, i + pBin_start_page, 1);
            fs_flash_write(g_tmp_buf_page, i + ES_FAT_RAM_PAGE_NUM, 1);
        }
    }
}

void es_fatfs_error_handler(void)
{
    printf("[es_fatfs_error]\r\n");
}

static uint32_t read_dir_find_newapp(char *input_fullpath)
{
    char *fullpath = input_fullpath;
    DIR dir;
    FILINFO fno;
    FRESULT res;
    FIL fp[1];
    uint32_t len, index;
    uint8_t temp_flag;
    uint32_t i, j, temp_file_read_buf[16];
    uint32_t crc_value;
    unsigned int get_num;
    uint32_t pBin_start_page;

    res = f_opendir(&dir, fullpath);

    if (res == FR_OK)
    {
        printf("dir: %s\r\n", fullpath);

        while (1)
        {
            res = f_readdir(&dir, &fno);

            if ((res != FR_OK) || (fno.fname[0] == '\0'))
                break;

            printf("%-32s ", fno.fname);

            if ((fno.fattrib) & (AM_DIR))
                printf("<DIR>\r\n");
            else
            {
                printf("%d\r\n", (unsigned int)(fno.fsize));

                temp_flag = 0;
                len = strlen(fullpath);

                if (len < ES_FATFS_BASE_PATH_MAX_LEN - 1)
                {
                    memcpy(s_es_fatfs_new_app_path, fullpath, len);
                    index = len;
                    len = strlen(fno.fname);

                    if (len + index < ES_FATFS_BASE_PATH_MAX_LEN - 1)
                    {
                        memcpy(s_es_fatfs_new_app_path + index, fno.fname, len);
                        s_es_fatfs_new_app_path[len + index] = '\0';
                        temp_flag = 1;
                    }
                }

                if (temp_flag)
                {
                    res = f_stat((const char *)s_es_fatfs_new_app_path, &fno);

                    printf("\r\ncheck file %s\r\n", s_es_fatfs_new_app_path);

                    if (res != FR_OK)
                        temp_flag = 0;

                    if (fno.fsize & 0x3)
                        temp_flag = 0;

                    res = f_open(fp, (const char *)s_es_fatfs_new_app_path, FA_READ);

                    temp_flag = 0;

                    if (res == FR_OK)
                    {
                        res = f_lseek(fp, fno.fsize - 8);

                        if (res == FR_OK)
                        {
                            res = f_read(fp, temp_file_read_buf, 8, &get_num);

                            if (res == FR_OK)
                            {
                                temp_flag = 1;

                                if (temp_file_read_buf[1] != (fno.fsize - 8))
                                    temp_flag = 0;

                                if (fno.fsize & 0x3)
                                    temp_flag = 0;

                                crc_value = temp_file_read_buf[0];

                                if (temp_flag)
                                {
                                    res = f_lseek(fp, 0);

                                    if (res == FR_OK)
                                    {
                                        if (temp_flag)
                                        {
                                            md_crc_init_t g_crc_init;

                                            md_crc_init_struct(&g_crc_init);
                                            g_crc_init.mode = MD_CRC_MODE_32;
                                            g_crc_init.len = MD_CRC_DATASIZE_32;
                                            g_crc_init.order = MD_CRC_BYTORD_LOW;
                                            g_crc_init.seed = 0xFFFFFFFF;
                                            g_crc_init.chs_inv = ENABLE;
                                            g_crc_init.chs_rev = ENABLE;
                                            g_crc_init.data_inv = DISABLE;
                                            g_crc_init.data_rev = ENABLE;
                                            md_crc_init(&g_crc_init);

                                            uint32_t i, end;

                                            end = (fno.fsize - 8);/*crc value + length*/

                                            res = f_lseek(fp, 0);

                                            if (res == FR_OK)
                                            {
                                                for (i = 0; i < end ; i += 4)
                                                {
                                                    if ((i & 0x3F) == 0)
                                                    {
                                                        res = f_read(fp, temp_file_read_buf, 64, &get_num);

                                                        if ((res != FR_OK) || (get_num == 0))
                                                            break;
                                                    }

                                                    md_crc_write_data(CRC, temp_file_read_buf[(i & 0x3F) >> 2]);
                                                }

                                                if (crc_value != md_crc_get_check_result(CRC))
                                                {
                                                    printf("new app crc err\r\n");
                                                    temp_flag = 0;
                                                }
                                                else
                                                    printf("new app crc ok\r\n");
                                            }
                                        }
                                    }

                                    if (temp_flag)
                                    {
                                        printf("get new app\r\n");

                                        /*start to find file true location*/
                                        pBin_start_page = 0;
                                        i = 0;
                                        index = 0;
                                        len = 0;

                                        for (i = 0; i < DISK_BLOCK_NR; i++)
                                        {
                                            fs_flash_read(g_tmp_buf_page, i, 1);
                                            f_lseek(fp, index * DISK_BLOCK_SIZE);

                                            for (j = 0; j < DISK_BLOCK_SIZE; j += 64)
                                            {
                                                f_read(fp, temp_file_read_buf, 64, &get_num);

                                                if (get_num)
                                                {
                                                    res = memcmp(g_tmp_buf_page + j, temp_file_read_buf, get_num);

                                                    if (res)
                                                    {
                                                        j = 0x7FFFFFFF;
                                                        break;
                                                    }
                                                    else
                                                        len += get_num;
                                                }
                                                else
                                                    break;
                                            }

                                            if ((j == 0x7FFFFFFF))
                                            {
                                                pBin_start_page = 0;
                                                index = 0;
                                                len = 0;
                                            }
                                            else
                                            {
                                                index++;

                                                if (pBin_start_page == 0)
                                                    pBin_start_page = i;

                                                if (len == fno.fsize)
                                                    break;
                                            }
                                        }

                                        if (len == fno.fsize)
                                        {
                                            app_flash_page_write(pBin_start_page);

                                            jump2app();
                                        }
                                        else
                                        {
                                            printf("not find app phy_addr\r\n");
                                        }
                                    }
                                }
                            }
                            else
                                es_fatfs_error_handler();
                        }
                        else
                            es_fatfs_error_handler();

                        res = f_close(fp);

                        if (res != FR_OK)
                            es_fatfs_error_handler();
                    }
                    else
                        printf("file open fail\r\n");
                }
            }
        }

        res = f_closedir(&dir);
    }
    else
    {
        es_fatfs_error_handler();
    }

    return 0;
}

void es_boot_fatfs_register(void)
{
    /* Regeister the disk I/O driver */
    if (FATFS_LinkDriverEx(&g_spi_flash_drv, (char *)s_es_fatfs_path, 0) == 0)
    {
        /* Register the file system object to the FatFs module */
        if (f_mount(&s_es_fatfs, (const TCHAR *)s_es_fatfs_path, 1) != FR_OK)
        {
            /* Create FAT volume on the logical drive 0 */
            printf("[err]not find fatfs\r\n");

            /* FatFs Initialization Error */
            es_fatfs_error_handler();
        }
        else
        {
            printf("\r\nfatfs mount succes , path :%s\r\n", s_es_fatfs_path);

            read_dir_find_newapp((char *)s_es_fatfs_path);
        }

        if (f_mount((void *)0, (const TCHAR *)s_es_fatfs_path, 0) != FR_OK)
        {
            /* FatFs Initialization Error */
            es_fatfs_error_handler();
        }

        printf("fatfs unmount succes , path :%s", s_es_fatfs_path);

        /*nUnlink the spi Flash disk I/O driver */
        FATFS_UnLinkDriverEx((char *)s_es_fatfs_path, 0);
    }
}

/************* (C) COPYRIGHT Eastsoft Microelectronics *****END OF FILE****/
