/**
  **************************************************************************************
  * @file    iap.c
  * @brief   iap operation functions.
  * @version V1.0
  * @date    26 Jun 2019
  * @author  AE Team
  * @note
  *          Change Logs:
  *          Date            Author          Notes
  *          26 Jun 2019     AE Team         The first version
  *
  * Copyright (C) Shanghai Eastsoft Microelectronics Co. Ltd. All rights reserved.
  *
  * SPDX-License-Identifier: Apache-2.0
  *
  * Licensed under the Apache License, Version 2.0 (the License); you may
  * not use this file except in compliance with the License.
  * You may obtain a copy of the License at
  *
  * www.apache.org/licenses/LICENSE-2.0
  *
  * Unless required by applicable law or agreed to in writing, software
  * distributed under the License is distributed on an AS IS BASIS, WITHOUT
  * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
  * See the License for the specific language governing permissions and
  * limitations under the License.
  **********************************************************************************
  */

/* Includes-------------------------------------------------------------------*/
#include "es_hid_lab.h"
#include "app_update.h"
#include "md_syscfg.h"

/* Public Variables ---------------------------------------------------------- */
uint8_t g_jump2boot_flag = 0U;
/* Private Variables --------------------------------------------------------- */
/* Private Constants --------------------------------------------------------- */
static const char s_cmd_incalid_ack[14]  = "CMD_INVALID\r\n\0";
static const char s_param_error[14]      = "PARAM_ERROR\r\n\0";
static const char s_check_error_ack[14]  = "CHECK_ERROR\r\n\0";
static const char s_prog_failed_ack[14]  = "PROG_FAILED\r\n\0";
/*static const char count_error_ack[14]  = "COUNT_ERROR\r\n\0";
static const char addr_error_ack[13]   = "ADDR_ERROR\r\n\0";
static const char busy_ack[7]          = "BUSY\r\n\0";
static const char wait_ack[4]          = "*\r\n\0";
static const char erase_failed_ack[15] = "ERASE_FAILED\r\n\0";*/
/* Private Macros ------------------------------------------------------------*/
#define OK           0
#define CMD_INVALID  1
#define PARAM_ERROR  2
#define COUNT_ERROR  3
#define ADDR_ERROR   4
#define BUSY         5
#define WAIT         6
#define CHECK_ERROR  7
#define ERASE_FAILED 8
#define PROG_FAILED  9
/* Private function prototypes -----------------------------------------------*/
/* Private Functions ---------------------------------------------------------*/

/**
  * @brief ok ack.
  * @param request is pointer to request data
  * @param response is pointer to response data
  * @retval the length of the ok ack in bytes
  */
static uint32_t ok_ack(uint8_t *request, uint8_t *response)
{
    int i;
    uint32_t len;

    if (strstr((char *)request, "\r\n") == NULL)
        return 0;

    len = strstr((char *)request, "\r\n") - (char *)request;
    memcpy(response, request, len);

    *(response) = '!';
    *(response + len + 0) = 'O';
    *(response + len + 1) = 'K';
    *(response + len + 2) = '\r';
    *(response + len + 3) = '\n';

    for (i = 0; i <= (len + 3); i++)
        *(response + len + 4) += (uint8_t)response[i];

    *(response + len + 5) = '\0';
    return (len + 6);
}
/**
  * @brief error ack.
  * @param error type
  * @param response is pointer to response data
  * @retval the length of the error ack in bytes
  */
static uint32_t error_ack(uint8_t error_type, uint8_t *response)
{
    int i;
    const char *pstr;
    uint32_t size;

    switch (error_type)
    {
        case CMD_INVALID:
            pstr = s_cmd_incalid_ack;
            size = sizeof(s_cmd_incalid_ack);
            break;

        case PARAM_ERROR:
            pstr = s_param_error;
            size = sizeof(s_param_error);
            break;

        case PROG_FAILED:
            pstr = s_prog_failed_ack;
            size = sizeof(s_prog_failed_ack);
            break;

        case CHECK_ERROR:
            pstr = s_check_error_ack;
            size = sizeof(s_check_error_ack);
            break;

        default:
            pstr = s_cmd_incalid_ack;
            size = sizeof(s_cmd_incalid_ack);
            break;
    }

    memcpy(response, pstr, size);
    response[size - 1] = 0;

    for (i = 0; i < (size - 1); i++)
        response[size - 1] += response[i];

    response[size] = '\0';
    return OK;
}

/**
  * @brief es lab commands process.
  * @param request is pointer to request data
  * @param response is pointer to response data
  * @retval none
  */
uint32_t es_lab_commands_process(uint8_t *request, uint8_t *response)
{
    uint8_t result = OK;
    uint32_t ack_len;

    if (*request != '?')
    {
        result = CMD_INVALID;
        goto __ACK;
    }

    switch (*(request + 1))
    {
        case 'G':
            g_jump2boot_flag = 0;

            if (memcmp(request + 2, " BOOT", 5))
                result = PARAM_ERROR;

            if (result == OK)
            {
                g_jump2boot_flag = 1;
                app_flash_erase_area(ADDR_CHECK_APP_FLASH_CRC_LEN & 0xFFFFF800, 2048);
            }

            break;

        default:
            result = CMD_INVALID;
            break;
    }

__ACK:

    memset(response, 0, 64);

    if (result == OK)
    {
        ack_len = ok_ack(request, response);
    }
    else
    {
        ack_len = error_ack(result, response);
    }

    return  ack_len;
}

/******************* (C) COPYRIGHT Eastsoft Microelectronics END OF FILE ****/
