/**
  **************************************************************************************
  * @file    iap.c
  * @brief   iap operation functions.
  * @version V1.0
  * @date    26 Jun 2019
  * @author  AE Team
  * @note
  *          Change Logs:
  *          Date            Author          Notes
  *          26 Jun 2019     AE Team         The first version
  *
  * Copyright (C) Shanghai Eastsoft Microelectronics Co. Ltd. All rights reserved.
  *
  * SPDX-License-Identifier: Apache-2.0
  *
  * Licensed under the Apache License, Version 2.0 (the License); you may
  * not use this file except in compliance with the License.
  * You may obtain a copy of the License at
  *
  * www.apache.org/licenses/LICENSE-2.0
  *
  * Unless required by applicable law or agreed to in writing, software
  * distributed under the License is distributed on an AS IS BASIS, WITHOUT
  * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
  * See the License for the specific language governing permissions and
  * limitations under the License.
  **********************************************************************************
  */

/* Includes-------------------------------------------------------------------*/
#include "es_hid_lab.h"
#include "app_update.h"
#include "md_syscfg.h"

/* Private Variables --------------------------------------------------------- */
static uint32_t s_r_cmd_start = 0U;
static uint32_t s_r_cmd_len = 0U;
static uint32_t s_w_cmd_start = 0U;
static uint32_t s_w_cmd_len = 0U;
static uint32_t s_prog_addr;
static uint32_t s_app_flash_start_addr;
static uint32_t s_app_flash_size;
/* Public Variables ---------------------------------------------------------- */
/* Private Constants --------------------------------------------------------- */
static const char s_cmd_incalid_ack[14]  = "CMD_INVALID\r\n\0";
static const char s_param_error[14]      = "PARAM_ERROR\r\n\0";
static const char s_check_error_ack[14]  = "CHECK_ERROR\r\n\0";
static const char s_prog_failed_ack[14]  = "PROG_FAILED\r\n\0";
/*static const char count_error_ack[14]  = "COUNT_ERROR\r\n\0";
static const char addr_error_ack[13]   = "ADDR_ERROR\r\n\0";
static const char busy_ack[7]          = "BUSY\r\n\0";
static const char wait_ack[4]          = "*\r\n\0";
static const char erase_failed_ack[15] = "ERASE_FAILED\r\n\0";*/
/* Private Macros ------------------------------------------------------------*/
#define OK           0
#define CMD_INVALID  1
#define PARAM_ERROR  2
#define COUNT_ERROR  3
#define ADDR_ERROR   4
#define BUSY         5
#define WAIT         6
#define CHECK_ERROR  7
#define ERASE_FAILED 8
#define PROG_FAILED  9
/* Private function prototypes -----------------------------------------------*/
/* Private Functions ---------------------------------------------------------*/
/**
  * @brief The string is converted to the hex data
  * @param s: is point to the string .
  * @param data: is point to the data.
  * @param length: is the data length.
  * @retval checksum
  */
static uint32_t string_to_hex(char *s, uint32_t *data, uint32_t length)
{
    uint32_t i;
    uint32_t size;
    char *posend;
    char *pos;
    char str[11];

    if ((posend = strchr(s, ' ')) == NULL)
        return 1;

    if (strstr(s, "\r\n") == NULL)
        return 1;

    for (i = 0; i < length; i++)
    {
        pos = posend + 1;
        memset(str, 0, sizeof(str));

        if ((posend = strchr(pos, ' ')) == NULL)
            return 1;

        size = posend - pos;

        if (size > 10)
            return PARAM_ERROR;

        memcpy(str, pos, size);

#if 0

        if (sscanf(str, "%x", data) == 0)
            return 1;

#else
        uint8_t j;

        *data = 0;

        if ((str[0] == '0') && (str[1] == 'x'))
            j = 2;
        else
            j = 0;

        for (; j < size ; j ++)
        {
            *data = (*data) << 4;

            if ((str[j] >= '0') && (str[j] <= '9'))
                *data += str[j] - '0';
            else if ((str[j] >= 'a') && (str[j] <= 'f'))
                *data += str[j] - 'a' + 10;
            else if ((str[j] >= 'A') && (str[j] <= 'F'))
                *data += str[j] - 'A' + 10;
            else
                return 11;
        }

#endif /*CDK sscanf occupies too much ROM*/

        data++;
    }

    return 0;
}

/**
  * @brief ok ack.
  * @param request is pointer to request data
  * @param response is pointer to response data
  * @retval the length of the ok ack in bytes
  */
static uint32_t ok_ack(uint8_t *request, uint8_t *response)
{
    int i;
    uint32_t len;

    if (strstr((char *)request, "\r\n") == NULL)
        return 0;

    len = strstr((char *)request, "\r\n") - (char *)request;
    memcpy(response, request, len);

    *(response) = '!';
    *(response + len + 0) = 'O';
    *(response + len + 1) = 'K';
    *(response + len + 2) = '\r';
    *(response + len + 3) = '\n';

    for (i = 0; i <= (len + 3); i++)
        *(response + len + 4) += (uint8_t)response[i];

    if (s_r_cmd_len)
    {
        for (i = 0; i < s_r_cmd_len; i++)
        {
            *(response + len + 5 + i) = *((uint8_t *)(s_r_cmd_start + i));
            *(response + len + 5 + s_r_cmd_len) += *((uint8_t *)(s_r_cmd_start + i));
        }

        len += (s_r_cmd_len + 1);
        s_r_cmd_len = 0;
    }

    *(response + len + 5) = '\0';
    return (len + 6);
}
/**
  * @brief error ack.
  * @param error type
  * @param response is pointer to response data
  * @retval the length of the error ack in bytes
  */
static uint32_t error_ack(uint8_t error_type, uint8_t *response)
{
    int i;
    const char *pstr;
    uint32_t size;

    switch (error_type)
    {
        case CMD_INVALID:
            pstr = s_cmd_incalid_ack;
            size = sizeof(s_cmd_incalid_ack);
            break;

        case PARAM_ERROR:
            pstr = s_param_error;
            size = sizeof(s_param_error);
            break;

        case PROG_FAILED:
            pstr = s_prog_failed_ack;
            size = sizeof(s_prog_failed_ack);
            break;

        case CHECK_ERROR:
            pstr = s_check_error_ack;
            size = sizeof(s_check_error_ack);
            break;

        default:
            pstr = s_cmd_incalid_ack;
            size = sizeof(s_cmd_incalid_ack);
            break;
    }

    memcpy(response, pstr, size);
    response[size - 1] = 0;

    for (i = 0; i < (size - 1); i++)
        response[size - 1] += response[i];

    response[size] = '\0';
    return OK;
}


/**
  * @brief Calculate the checksum of the app flash
  * @param addr is the flash start address
  * @param size is the flash start size
  * @retval checksum
  */
static uint32_t app_flash_checksum(uint32_t addr, uint32_t size)
{
    uint32_t data = 0U;
    uint32_t sum  = 0U;
    uint32_t i    = 0U;

    for (i = 0; i < size; i += 4)
    {
        data = *(volatile uint32_t *)(addr + i);
        sum += data ;
    }

    return sum;
}

/**
  * @brief es lab commands process.
  * @param request is pointer to request data
  * @param response is pointer to response data
  * @retval none
  */
uint32_t es_lab_commands_process(uint8_t *request, uint8_t *response)
{
    uint8_t result = OK;
    uint32_t ack_len;
    uint32_t param[2];

    uint32_t check_sum;

    char *ptr;

    if (*request != '?')
    {
        result = CMD_INVALID;
        goto __ACK;
    }

    switch (*(request + 1))
    {
        case 'G':
            result = OK;

            if (memcmp(request + 2, " APP", 4))
                result = PARAM_ERROR;

            if (check_app_crc())
                result = CHECK_ERROR;

            if (result == OK)
                g_boot_complete = TRUE;

            break;

        case 'R':
            if (string_to_hex((char *)request, param, 2) != 0)
            {
                result = PARAM_ERROR;
                break;
            }

            if (param[1] < 33U)
            {
                s_r_cmd_start = param[0];
                s_r_cmd_len = param[1];
            }

            result = OK;
            break;

        case 'W':
            if (string_to_hex((char *)request, param, 2) != 0)
            {
                result = PARAM_ERROR;
                break;
            }

            app_flash_info(&s_app_flash_start_addr, &s_app_flash_size);

            if ((param[0] < s_app_flash_start_addr) && \
                    ((param[0] + param[1]) < (s_app_flash_start_addr + s_app_flash_size)))
            {
                result = PARAM_ERROR;
                break;
            }

            s_prog_addr = param[0];

            s_w_cmd_start = param[0];
            s_w_cmd_len = param[1];

            if (app_flash_erase_area(s_w_cmd_start, s_w_cmd_len) != TRUE)
            {
                result = PARAM_ERROR;
                break;
            }

            result = OK;
            break;

        case 'D':
            if (string_to_hex((char *)request, param, 2) != 0)
            {
                result = PARAM_ERROR;
                break;
            }

            if (param[1] == 0x00)
            {

            }
            else if (param[1] == 0xff)
            {
                if ((ptr = strstr((char *)request, "\r\n")) == NULL)
                {
                    result = PARAM_ERROR;
                    break;
                }

                ptr += 3;
                check_sum = (ptr[3] << 24) | (ptr[2] << 16) | (ptr[1] << 8) | (ptr[0]);

                __disable_irq();

                if (check_sum != app_flash_checksum(s_w_cmd_start, s_w_cmd_len))
                {
                    app_flash_erase_area(s_w_cmd_start, s_w_cmd_len);
                    result = CHECK_ERROR;
                    break;
                }
                else
                {

                    if ((s_w_cmd_start <= ADDR_CHECK_APP_FLASH_CRC_LEN) && (ADDR_CHECK_APP_FLASH_CRC_LEN <= (s_w_cmd_start + s_w_cmd_len)))
                        g_boot_complete = TRUE;
                }

                __enable_irq();

            }
            else
            {
                if ((ptr = strstr((char *)request, "\r\n")) == NULL)
                {
                    result = PARAM_ERROR;
                    break;
                }

                ptr += 3;

                if (app_flash_write(s_prog_addr, (uint8_t *)ptr, param[1]) != TRUE)
                {
                    result = PROG_FAILED;
                    break;
                }

                s_prog_addr += param[1];

            }

            result = OK;
            break;

        default:
            result = CMD_INVALID;
            break;
    }

__ACK:

    memset(response, 0, 64);

    if (result == OK)
    {
        ack_len = ok_ack(request, response);
    }
    else
    {
        ack_len = error_ack(result, response);
    }

    return  ack_len;
}

/******************* (C) COPYRIGHT Eastsoft Microelectronics END OF FILE ****/
