/**
  *********************************************************************************
  * @file   boot_flash.c
  * @brief  Write update to flash or memory.
  *
  * @version V1.0
  * @date    28 Feb 2023
  * @author  AE Team
  * @note
  *          Change Logs:
  *          Date            Author          Notes
  *          31 Dec 2019     AE Team         The first version
  *          28 Feb 2023     AE Team         Port to ES32VF2264
  *
  * Copyright (C) Shanghai Eastsoft Microelectronics Co. Ltd. All rights reserved.
  *
  * SPDX-License-Identifier: Apache-2.0
  *
  * Licensed under the Apache License, Version 2.0 (the License); you may
  * not use this file except in compliance with the License.
  * You may obtain a copy of the License at
  *
  * www.apache.org/licenses/LICENSE-2.0
  *
  * Unless required by applicable law or agreed to in writing, software
  * distributed under the License is distributed on an AS IS BASIS, WITHOUT
  * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
  * See the License for the specific language governing permissions and
  * limitations under the License.
  **********************************************************************************
  */
/* Includes ------------------------------------------------------------------ */

#include <string.h>
#include <stdio.h>
#include "md_conf.h"
#include "boot_flash.h"
#include "boot_frame.h"
#include "boot_shell.h"

/* Private Macros ------------------------------------------------------------ */

/* Private Constants --------------------------------------------------------- */

/* Private function prototypes ----------------------------------------------- */

/* Private Variables --------------------------------------------------------- */

/* Public Variables ---------------------------------------------------------- */

/** @addtogroup Bootloader
  * @{
  */

/** @defgroup Flash Flash
  * @brief    Bootloader Flash Module
  * @{
  */

/* Private Function ---------------------------------------------------------- */

/** @defgroup Flash_Public_Functions Public Functions
  * @brief    Flash Public Functions
  * @{
  */
/**
  * @brief  Get page of data flash
  * @retval Number of page
  */
uint32_t flash_usr_page_get(void)
{
    uint32_t *p = NULL;
    uint32_t tmp = 0U;

    p = (uint32_t *)APP_FLASH_S;

    while (*p != 0xFFFFFFFF)
    {
        tmp++;
		p+=1024;
    }

    return tmp;
}

/**
  * @brief  Erase usr flash
  * @retval 0-success, other value indicates failed.
  */
int flash_usr_page_erase(void)
{
    uint32_t addr = APP_FLASH_S;
	int status=0;
		
    while (addr < APP_FLASH_E)
    {
		int need_erase=0;
		for (int i=0;i<512;i+=4)
		{
			if (*(uint32_t*)(addr+i)!=0xFFFFFFFF)
			{
				need_erase=1;
				break;
			}
		}
		
        if (need_erase&&md_msc_code_erase_page(addr, ~addr, 0) )
        {
			status=-1;
			break;
		}
        addr += 512;
    }
		
    return status;
}

/**
  * @brief  write data to flash at specified address.
  * @param  page: specific page.
  * @param  data: write data
  * @param  crc: data crc.
  * @retval 0-success, other value indicates failed.
  */
int write_data_to_flash(uint8_t page, uint8_t *data, uint16_t crc)
{
    uint8_t *p;
    uint16_t _crc;
	int status=0;

    p = (uint8_t *)USR_FLASH_ADDER(page);
	
    if (md_msc_code_program_words(USR_FLASH_ADDER(page),~USR_FLASH_ADDER(page), data, 1024, FALSE))
    {
		status=-1;
	}
	
	if (status)
	{
		return status;
	}
	
    MD_CRC_RESET();
	
	for (int i=0;i<1024;i++)
	{
		md_crc_write_data(g_h_crc, p[i]);
	}
    _crc = md_crc_get_check_result(g_h_crc);
    if (_crc != crc)
        status = -2;

    return status;
}
/**
  * @}
  */
/**
  * @}
  */
/**
  * @}
  */
