/**
  *********************************************************************************
  * @file   boot.c
  * @brief  Base functions.
  *
  * @version V1.0
  * @date    28 Feb 2023
  * @author  AE Team
  * @note
  *          Change Logs:
  *          Date            Author          Notes
  *          31 Dec 2019     AE Team         The first version
  *          28 Feb 2023     AE Team         Port to ES32VF2264
  *
  * Copyright (C) Shanghai Eastsoft Microelectronics Co. Ltd. All rights reserved.
  *
  * SPDX-License-Identifier: Apache-2.0
  *
  * Licensed under the Apache License, Version 2.0 (the License); you may
  * not use this file except in compliance with the License.
  * You may obtain a copy of the License at
  *
  * www.apache.org/licenses/LICENSE-2.0
  *
  * Unless required by applicable law or agreed to in writing, software
  * distributed under the License is distributed on an AS IS BASIS, WITHOUT
  * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
  * See the License for the specific language governing permissions and
  * limitations under the License.
  **********************************************************************************
  */
  
/* Includes ------------------------------------------------------------------ */

#include <string.h>
#include <stdio.h>
#include "md_conf.h"
#include "main.h"
#include "boot.h"
#include "boot_flash.h"
#include "boot_frame.h"
#include "boot_shell.h"

/* Private Macros ------------------------------------------------------------ */

/* Private Constants --------------------------------------------------------- */

/* Private function prototypes ----------------------------------------------- */

/* Private Variables --------------------------------------------------------- */

/** @defgroup Bootloader Bootloader
  * @brief    Eastsoft Bootloader
  * @{
  */

/** @defgroup Base Base
  * @brief    Bootloader Base
  * @{
  */
/** @defgroup Base_Public_Variables Public Variable
  * @brief    Base Public Variable
  * @{
  */
  
/* Public Variables ---------------------------------------------------------- */

timer_env_t g_timer_env = {0};
/**
  * @}
  */

/** @defgroup Base_Private_Functions Private Functions
  * @brief    Base Private_Functions
  * @{
  */
  
/* Private Function ---------------------------------------------------------- */

/**
  * @brief  Systick callback function
  * @retval None
  */
__isr__ void boot_systick_irq_cbk(void)
{
    if (!g_timer_env.state)
        return;

    if ((--g_timer_env.tick == 0) && (g_timer_env.cbk != NULL))
        g_timer_env.cbk();
}

/**
  * @brief  Initialize timer.
  * @retval None
  */
void timer_init(void)
{
    memset(&g_timer_env, 0x0, sizeof(timer_env_t));
    return;
}

/**
  * @brief  Start timer.
  * @param  tick: Timeout in milliseconds.
  * @param  cbk: callback function for timeout.
  * @retval None
  */
void timer_start(uint32_t tick, timer_cbk_t cbk)
{
    tick = tick == 0 ? 1 : tick;

    __disable_irq();
    g_timer_env.tick  = tick;
    g_timer_env.cbk   = cbk;
    g_timer_env.state = 1;
    __enable_irq();

    return;
}

/**
  * @brief  Stop timer.
  * @retval None
  */
void timer_stop(void)
{
    __disable_irq();
    memset(&g_timer_env, 0x0, sizeof(timer_env_t));
    __enable_irq();

    return;
}

/**
  * @brief  Callback function for timeout.
  * @retval None
  */
void boot_timer_cbk(void)
{
    g_timer_env.timeout = 1;
}
/**
  * @brief  Check app CRC
  * @param  page: total page of flash data.
  * @retval 0-success, other value indicates failed.
  */
uint8_t check_app_crc(void)
{
    uint32_t begin, end;
    md_crc_init_t g_crc_init;
	uint32_t applen;
	uint32_t appcrc;
	appcrc=*((uint32_t*)(APP_FLASH_INFO_S+0));
	applen=*((uint32_t*)(APP_FLASH_INFO_S+4));
    begin = ADDR_CHECK_APP_FLASH_CRC_START;
    end = ADDR_CHECK_APP_FLASH_CRC_START + applen;
	if (!VALID_APP_ADDR(end))
	{
		return 1;
	}

    if ((begin & 0x3) || (end & 0x3) || (end > (APP_FLASH_S + APP_FLASH_SIZE)))
        return 2;

    /* Init CRC */
    md_crc_init_struct(&g_crc_init);
    g_crc_init.mode = MD_CRC_MODE_32;
    g_crc_init.len = MD_CRC_DATASIZE_32;
    g_crc_init.order = MD_CRC_BYTORD_LOW;
    g_crc_init.seed = 0xFFFFFFFF;
    g_crc_init.chs_inv = ENABLE;
    g_crc_init.chs_rev = ENABLE;
    g_crc_init.data_inv = DISABLE;
    g_crc_init.data_rev = ENABLE;
    md_crc_init(&g_crc_init);

    for (uint32_t i=begin; i < end ; i += 4)
    {
        md_crc_write_data(CRC, *((uint32_t *)(i)));
    }

    if (appcrc != md_crc_get_check_result(CRC))
        return 1;

    return 0;
}

/**
  * @brief  Copy data from flash to sram.
  * @retval 0-success, other value indicates failed.
  */
int copy_data_to_sram()
{
    uint32_t *p_flash = NULL;
    uint32_t *p_ram   = NULL;
    uint32_t tmp = 0U;
	uint32_t app_len=(*((uint32_t *)(ADDR_CHECK_APP_FLASH_CRC_LEN)));

    if (app_len > (APP_RUN_RAM_E - APP_RUN_RAM_S))
    {
        printf_e("Copy data to ram error, Exceeded RAM size!!!\r\n");
        printf_e("SRAM = %u , APP = %u\r\n",(APP_RUN_RAM_E - APP_RUN_RAM_S),app_len);
        return -1;
    }

    p_flash = (uint32_t *)APP_FLASH_S;
    p_ram   = (uint32_t *)(APP_RUN_RAM_S);

    for (tmp = 0; tmp < app_len/4; tmp++)
    {
        *p_ram++ = *p_flash++;
    }

    return 0;
}

void reset_and_run(uint32_t app_addr)
{
    __disable_irq();
	
    MD_SYSCFG_UNLOCK();
    md_syscfg_set_cpu_boot_addr(app_addr);
    MD_SYSCFG_LOCK();

    __enable_irq();

    md_rmu_reset_system();
}

/**
  * @brief  Run in flash
  * @retval None.
  */
void run_in_flash(void)
{
	reset_and_run(APP_FLASH_S);

    return;
}

/**
  * @brief  Run in ram
  * @retval None.
  */
void run_in_sram(void)
{
    /*
    __disable_irq();
    MD_SYSCFG_UNLOCK();
    md_rmu_enable_cuart1_reset();
    md_rmu_enable_gpio_reset();
	md_rmu_enable_crc_reset();
	
    md_cmu_disable_perh_all();
    MD_SYSCFG_LOCK();
    __enable_irq();
	((void(*)(void))(APP_RUN_RAM_S))();
    */
    
    /* Not implemented yet */
    return;
}

/**
  * @brief  command for "version".
  * @retval None
  */
void cmd_version(void)
{
    printf_e("Bootload: 01-00-00-00\r\n");
    return;
}

/**
  * @brief  command for "reboot".
  * @retval None
  */
void cmd_reboot(void)
{
    md_rmu_reset_system();
    return;
}

static void uart_irq_ctrl(type_func_t status)
{
#ifdef BOOT_ES32VF2
	if(status==ENABLE)
	{
		md_uart_enable_it_rfnempty(BOOT_UARTX);
		csi_vic_enable_irq(BOOT_UARTX_IRQ);
	}
	else 
	{
		md_uart_disable_it_rfnempty(BOOT_UARTX);
		csi_vic_disable_irq(BOOT_UARTX_IRQ);
	}
#endif /* BOOT_ES32VF2 */
    return;
}

/**
  * @brief  command for "download_ram".
  * @retval None
  */
void cmd_download(void)
{
    uart_irq_ctrl(DISABLE);
    printf_e("************* Update Program *************\r\n");
    printf_e("Strat updating afer 3 second !\r\n");
    printf_e("Don't input anything!!!\r\n");
    md_delay_1ms(3000);
	
    flash_usr_page_erase();
	
    boot_image_update();
	
    boot_update_info();
    uart_irq_ctrl(ENABLE);

    return;
}

int can_run_app()
{
    if (!flash_usr_page_get())
    {
        printf_e("The APP flash is empty, Please Update first!\r\n");
        return -1;
    }
	#ifdef __USE_CRC32
	if (check_app_crc())
	{
        printf_e("The APP flash cannot be verified, Please Update first!\r\n");
        return -2;
	}
	#endif /* __USE_CRC32 */
	return 0;
}
/**
  * @brief  command for "run_flash".
  * @retval None
  */
void run_flash_cmd(void)
{
	if (can_run_app())
		return;
    printf_e("\r\n");
    run_in_flash();
	md_delay_1ms(1);
    return;
}

/**
  * @brief  command for "run_ram".
  * @retval None
  */
void run_sram_cmd(void)
{
	if (can_run_app())
		return;

    if (copy_data_to_sram())
        return;

    printf_e("\r\n");
    run_in_sram();
    return;
}
/**
  * @}
  */
/** @defgroup Base_Public_Functions Public functions
  * @brief    Base Public Functions
  * @{
  */
/**
  * @brief  Enter bootloader.
  * @retval None
  */
void boot_enter(UART_TypeDef*hperh)
{
    timer_init();
    boot_frame_init(hperh);
    shell_init(hperh);
    shell_cmd_insert("version", cmd_version, 0);
    shell_cmd_insert("reboot", cmd_reboot, 0);
    shell_cmd_insert("update", cmd_download, 0);
    shell_cmd_insert("run_flash", run_flash_cmd, 0);
    /* shell_cmd_insert("run_sram", run_sram_cmd, 0); */
	
    if (!flash_usr_page_get())
    {
        printf_e("The APP flash is empty, Please Update first!\r\n");
    }
    else
    {
		#ifdef __USE_CRC32
		if (check_app_crc())
		{
			printf_e("The APP flash is unempty, but verify failed.\r\n");
			printf_e("Please Update first\r\n");
		}
		else
		#endif /* __USE_CRC32 */
		{
			printf_e("The APP flash is unempty, You can type the following cmd:\r\n");
			printf_e("update, run_flash, run_sram:\r\n");
			printf_e("Otherwise auto run after 3 seconds:\r\n");
			timer_start(3000, boot_timer_cbk);
		}
    }
	
    printf_e("\r\nes#");

    while (1)
    {
        if (shell_rx_flag_get())
        {
            shell_rx_flag_clear();
            shell_task_func(NULL);

            if (g_timer_env.state)
                timer_stop();
        }

        if (g_timer_env.timeout)
        {
            g_timer_env.timeout = 0;

            if (RUN_MODE_DEFAULT == RUN_FROM_FLASH)
                run_in_flash();
            else
                run_sram_cmd();
        }
    }
}
/**
  * @}
  */
/**
  * @}
  */
/**
  * @}
  */
