/**********************************************************************************
 *
 * @file    .c
 * @brief   Source file
 *
 * @author  AE Team
 * @note
 *          Change Logs:
 *          Date               Author          Notes
 *          2023-02-23         shiwa           UART Bootloader for ES32VF2264
 *
 * Copyright (C) Shanghai Eastsoft Microelectronics Co. Ltd. All rights reserved.
 *
 * SPDX-License-Identifier: Apache-2.0
 *
 * Licensed under the Apache License, Version 2.0 (the License); you may
 * not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an AS IS BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 *
 **********************************************************************************
 */

/* Includes-------------------------------------------------------------------*/
#include "main.h"
#include "boot.h"
/* Private Macros ------------------------------------------------------------ */
/* Private Variables --------------------------------------------------------- */
/* Public Variables ---------------------------------------------------------- */
/* Private Constants --------------------------------------------------------- */
/* Private function prototypes ----------------------------------------------- */
/* Private Function ---------------------------------------------------------- */
/**
  * @brief  Initialize the UART
  * @retval None
  */
void uart_init(void)
{
    md_gpio_init_t gpio_init;
    md_gpio_init_struct(&gpio_init);
	
	/* Initialize tx pin */
    gpio_init.mode  = MD_GPIO_MODE_OUTPUT;
    gpio_init.odos  = MD_GPIO_PUSH_PULL;
    gpio_init.pupd  = MD_GPIO_PUSH_UP;
    gpio_init.odrv  = MD_GPIO_OUT_DRIVE_NORMAL;
    gpio_init.flt   = MD_GPIO_FILTER_DISABLE;
    gpio_init.type  = MD_GPIO_TYPE_CMOS;
    gpio_init.func  = MD_GPIO_FUNC_2;
	md_gpio_init(BOOT_UART_TX_PORT, BOOT_UART_TX_PIN, &gpio_init);

	/* Initialize rx pin */
    gpio_init.mode  = MD_GPIO_MODE_OUTPUT;
    gpio_init.odos  = MD_GPIO_PUSH_PULL;
    gpio_init.pupd  = MD_GPIO_PUSH_UP;
    gpio_init.odrv  = MD_GPIO_OUT_DRIVE_NORMAL;
    gpio_init.flt   = MD_GPIO_FILTER_DISABLE;
    gpio_init.type  = MD_GPIO_TYPE_CMOS;
    gpio_init.func  = MD_GPIO_FUNC_2;
	md_gpio_init(BOOT_UART_RX_PORT, BOOT_UART_RX_PIN, &gpio_init);

    md_uart_init_t uart_init;
    md_uart_init_struct(&uart_init);

    /* Initialize UART */
    uart_init.baud        = 115200;
    uart_init.word_length = MD_UART_WORD_LENGTH_8B;
    uart_init.stop_bits   = MD_UART_STOP_BITS_1;
    uart_init.parity      = MD_UART_PARITY_NONE;
    uart_init.fctl        = MD_UART_FLOW_CTL_DISABLE;
    uart_init.mode        = MD_UART_MODE;
    md_uart_init(BOOT_UARTX, &uart_init);
	
	md_uart_enable_it_rfnempty(BOOT_UARTX);
	
    csi_vic_set_prio(BOOT_UARTX_IRQ, 3);
    csi_vic_enable_sirq(BOOT_UARTX_IRQ);
	
	return;
}

/**
  * @brief  UART send string
  * @param  s: string to send
  * @param  len: string length
  * @param  timeout: tiemout(ms)
  * @retval 0-success 1-timeout
  */
uint32_t uart_send_str(const char *s, uint32_t len, uint32_t timeout)
{
	uint32_t status=0U;
	for (uint32_t i=0U;i<len;i++)
	{
		uint32_t tick=md_get_tick()+timeout;
		while (!md_uart_is_active_flag_tfempty(BOOT_UARTX)&&md_get_tick()<tick);
		if (md_get_tick()>=tick)
		{
			status=1;
			break;
		}
		md_uart_set_send_data8(BOOT_UARTX, (uint8_t)s[i]);
	}
	return status;
}

/**
  * @brief  UART receive string
  * @param  s: buffer to receive
  * @param  len: string length
  * @param  timeout: tiemout(ms)
  * @retval 0-success 1-timeout
  */
uint32_t uart_recv_str(char *s, uint32_t len, uint32_t timeout)
{
	uint32_t status=0U;
	for (uint32_t i=0U;i<len;i++)
	{
		uint32_t tick=md_get_tick()+timeout;
		while (!md_uart_is_active_flag_rfnempty(BOOT_UARTX)&&md_get_tick()<tick);
		if (md_get_tick()>=tick)
		{
			status=1;
			break;
		}
		s[i]=(char)md_uart_recv_data8(BOOT_UARTX);
	}
	return status;
}

/**
  * @brief main.
  * @param none
  * @retval none
  */
int main(void)
{
    md_cmu_pll_config(MD_CMU_PLL_INPUT_HOSC8M, MD_CMU_PLL_OUTPUT_72M);
    md_cmu_clock_config(MD_CMU_CLOCK_PLL, 72000000);

    MD_SYSCFG_UNLOCK();
    md_cmu_enable_perh_all();
    MD_SYSCFG_LOCK();
	
    __enable_irq();
    md_init_1ms_tick();
	uart_init();
	
	boot_enter(CUART1);
	return 0;
}

/******************* (C) COPYRIGHT Eastsoft Microelectronics END OF FILE ****/
