/**
  *********************************************************************************
  * @file   boot_frame.c
  * @brief  boot frames module.
  *
  * @version V1.0
  * @date    20 Mar 2023
  * @author  AE Team
  * @note
  *          Change Logs:
  *          Date            Author          Notes
  *          31 Dec 2019     AE Team         The first version
  *          28 Feb 2023     AE Team         Port to ES32VF2264
  *          20 Mar 2023     AE Team         Support backup app
  *
  * Copyright (C) Shanghai Eastsoft Microelectronics Co. Ltd. All rights reserved.
  *
  * SPDX-License-Identifier: Apache-2.0
  *
  * Licensed under the Apache License, Version 2.0 (the License); you may
  * not use this file except in compliance with the License.
  * You may obtain a copy of the License at
  *
  * www.apache.org/licenses/LICENSE-2.0
  *
  * Unless required by applicable law or agreed to in writing, software
  * distributed under the License is distributed on an AS IS BASIS, WITHOUT
  * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
  * See the License for the specific language governing permissions and
  * limitations under the License.
  **********************************************************************************
  */
/* Includes ------------------------------------------------------------------ */

#include <string.h>
#include <stdio.h>
#include "boot_frame.h"
#include "boot_flash.h"
#include "boot_shell.h"
#include "md_conf.h"
#include "uartstdio.h"

/* Private Macros ------------------------------------------------------------ */

/* Private Constants --------------------------------------------------------- */

/* Private function prototypes ----------------------------------------------- */

/* Private Variables --------------------------------------------------------- */


/** @addtogroup Bootloader
  * @{
  */
/** @defgroup Frame Frame
  * @brief    Bootloader Frame Module
  * @{
  */
/** @defgroup Frame_Public_Variables Public Variable
  * @brief    Frame Public Variable
  * @{
  */
/* Public Variables ---------------------------------------------------------- */

boot_frame_env_t g_frame_env;
CRC_TypeDef* g_h_crc;
/**
  * @}
  */

/** @defgroup Frame_Private_Functions Private Functions
  * @brief    Frame Private Functions
  * @{
  */
/**
  * @brief  Send command to tramsmitor
  * @param  cmd: Command.
  * @retval 0-success, other value indicates failed.
  */
/* Private Function ---------------------------------------------------------- */

static int xmod_cmd_send(xmod_tx_cmd_t cmd)
{
    uint8_t _cmd = cmd;

    if ((uart_send_str((char*)&_cmd, 1, 100)))
    {
        g_frame_env.state = UART_SEND_ERROR;
        return -1;
    }

    return 0;
}

/**
  * @brief  Check frame index.
  * @param  idx: frame index.
  * @retval 0-success, other value indicates failed.
  */
static int check_idx(uint8_t idx)
{
    uint8_t _idx  = 0U;
    uint8_t next_idx = 0U;

    _idx = g_frame_env.idx;
    next_idx = _idx == 0xFF ? 0 : _idx + 1;

    if ((next_idx != idx) && (idx != _idx))
        return 1;

    return 0;
}

#ifdef XMODE_1K
/**
  * @brief  Receive data by 1K_XMODE frame.
  * @retval 0-success, other value indicates failed.
  */
static int xmod_1K_recv(void)
{
    uint16_t crc, _crc;

    if ((uart_recv_str((char*)&g_frame_env.soh, 1, TIME_OUT)))
    {
        g_frame_env.state = FRAME_RECEIVE_TIMEOUT;
        return -1;
    }
	
    if (g_frame_env.soh == XMODE_EOT)
    {
        g_frame_env.state = FRAME_SUCCESS;
        return 1;
    }
	
	if (g_frame_env.soh != XMODE_STX)
	{
        g_frame_env.state = FRAME_INDEX_ERROR;
        return -2;
	}
    if ((uart_recv_str((char*)g_frame_env.data, 1028, TIME_OUT)))
    {
        g_frame_env.state = FRAME_RECEIVE_TIMEOUT;
        return -1;
    }

    if ((g_frame_env.data[0] ^ g_frame_env.data[1]) != 0xFF)
    {
        g_frame_env.state = FRAME_INDEX_ERROR;
        return -2;
    }

    if ((check_idx(g_frame_env.data[0])) != 0)
    {
        g_frame_env.state = FRAME_INDEX_ERROR;
        return -2;
    }

    MD_CRC_RESET();
	for (int i=0;i<1024;i++)
	{
		md_crc_write_data(g_h_crc, g_frame_env.data[2+i]);
	}
    _crc = md_crc_get_check_result(g_h_crc);
    crc = (g_frame_env.data[1026] << 8) | g_frame_env.data[1027];

    if (crc != _crc)
    {
        g_frame_env.state = FRAME_VERIFY_ERROR;
        return -3;
    }

    if (write_data_to_flash(g_frame_env.idx, &g_frame_env.data[2], crc))
    {
        g_frame_env.state = FLASH_WRITE_ERROR;
        return -4;
    }

    ++g_frame_env.idx;
    return 0;
}
#endif /* XMODE_1K */

/**
  * @brief  Receive a frame.
  * @retval 0-success, other value indicates failed.
  */
static int xmod_frame_recv(void)
{
    int result = 0;

#ifdef XMODE_1K
    result = xmod_1K_recv();
#endif /* XMODE_1K */
    return result;
}

/**
  * @}
  */

/** @defgroup Frame_Public_Functions Public Functions
  * @brief    Frame Public Functions
  * @{
  */

/**
  * @brief  Show update info
  * @retval none.
  */
void boot_update_info(void)
{
    if (g_frame_env.state == FRAME_SUCCESS)
    {
        printf_e("Update program sucess !\r\n");
    }
    else if (g_frame_env.state == FRAME_INDEX_ERROR)
    {
        printf_e("Update program error: FRAME_INDEX_ERROR !!!\r\n");
    }
    else if (g_frame_env.state == FRAME_RECEIVE_TIMEOUT)
    {
        printf_e("Update program error: FRAME_RECEIVE_TIMEOUT !!!\r\n");
    }
    else if (g_frame_env.state == FRAME_VERIFY_ERROR)
    {
        printf_e("Update program error: FRAME_VERIFY_ERROR !!!\r\n");
    }
    else if (g_frame_env.state == UART_SEND_ERROR)
    {
        printf_e("Update program error: UART_SEND_ERROR !!!\r\n");
    }
    else if (g_frame_env.state == FLASH_WRITE_ERROR)
    {
        printf_e("Update program error: FLASH_WRITE_ERROR !!!\r\n");
    }
    else if (g_frame_env.state ==APP_VERIFY_ERROR)
    {
        printf_e("Update program error: APP_VERIFY_ERROR !!!\r\n");
    }
    else
    {
    }

    return;
}

void xmodem_crc_init()
{
	md_crc_init_t crc_init;
    /* Clear crc_handle_t structure */
    md_crc_init_struct(&crc_init);

    /* Initialize CRC */
    crc_init.mode = MD_CRC_MODE_CCITT;
    crc_init.len = MD_CRC_DATASIZE_8;
    crc_init.order = MD_CRC_BYTORD_LOW;
    crc_init.seed = 0;
    crc_init.chs_inv = DISABLE;
    crc_init.chs_rev = DISABLE;
    crc_init.data_inv = DISABLE;
    crc_init.data_rev = DISABLE;
	md_crc_init(&crc_init);
}
/**
  * @brief  Initialize frame module
  * @param  hperh: Pointer to uart_handle_t structure.
  * @retval None
  */
void boot_frame_init(UART_TypeDef *hperh)
{
    if (hperh == NULL)
	{
		return;
	}
	
	g_h_crc=CRC;
    memset(&g_frame_env, 0, sizeof(boot_frame_env_t));
    g_frame_env.h_uart = hperh;

    return;
}

/**
  * @brief  Check app flash and copy info data
  * @retval 0 if success, others if error
  */
int check_app()
{
	/* Check app */
	uint32_t page=0U;
	uint8_t* data;
	uint32_t dlen=0U;
	uint32_t dcrc;
	uint32_t*info_data;
    
	#ifdef XMODE_1K
		page=g_frame_env.idx-1;
	#endif /* XMODE_1K */
    
	data=(uint8_t*)USR_FLASH_ADDER(page);
	if (!VALID_APP_ADDR((uint32_t)(data+1024-4)))
	{
		return -1; /*User app page invalid (receive error)*/
	}
	info_data=(uint32_t*)(data+INFO_ADDR_IN_PAGE);
	dcrc=info_data[0];
	dlen=info_data[1];
	if (!VALID_APP_ADDR(NEW_APP_FLASH_S+dlen-1))
	{
		return -3; /*User app length invalid */
	}
	
	md_msc_code_program_word(UPDATE_INFO_CRC_VALUE,~UPDATE_INFO_CRC_VALUE,dcrc);
	md_msc_code_program_word(UPDATE_INFO_CRC_LEN,~UPDATE_INFO_CRC_LEN,dlen);
	md_msc_code_program_word(UPDATE_INFO_NEW_APP_ADDR,~UPDATE_INFO_NEW_APP_ADDR,NEW_APP_FLASH_S);
	
	return 0;
}
/**
  * @brief  download bootloader program.
  * @retval 0-success, other value indicates failed.
  */
int  boot_image_update(void)
{
    int ret;
	
	xmodem_crc_init();
    g_frame_env.idx   = 0;
    g_frame_env.cnt   = 0;
    g_frame_env.state = FRAME_SUCCESS;
    xmod_cmd_send(XMOD_TX_CMD_START);

    while (1)
    {
        ret = xmod_frame_recv();

        if ((ret < 0) && (++g_frame_env.cnt < 10) &&g_frame_env.state==FRAME_RECEIVE_TIMEOUT)
        {
            if (g_frame_env.idx == 0)
                xmod_cmd_send(XMOD_TX_CMD_START);
            else
                xmod_cmd_send(XMOD_TX_CMD_NCK);
            continue;
        }

        if (ret < 0)
        {
            xmod_cmd_send(XMOD_TX_CMD_TERM);
            return ret;
        }

        if (ret == 1)
        {
            xmod_cmd_send(XMOD_TX_CMD_ACK);
            break;
        }

        xmod_cmd_send(XMOD_TX_CMD_ACK);
    }
#ifdef __USE_CRC32
	if (g_frame_env.state==FRAME_SUCCESS)
	{
		if (check_app())
		{
			g_frame_env.state=APP_VERIFY_ERROR;
			return -5;
		}
	}
#endif /* __USE_CRC32 */
	return 0;
}
/**
  * @}
  */
/**
  * @}
  */
/**
  * @}
  */
