/**
  *********************************************************************************
  * @file   boot.c
  * @brief  Base functions.
  *
  * @version V1.1
  * @date    20 Mar 2023
  * @author  AE Team
  * @note
  *          Change Logs:
  *          Date            Author          Notes
  *          31 Dec 2019     AE Team         The first version
  *          28 Feb 2023     AE Team         Port to ES32VF2264
  *          20 Mar 2023     AE Team         Delete run_to_xxx, reboot now reboot to boot(0x0)
  *
  * Copyright (C) Shanghai Eastsoft Microelectronics Co. Ltd. All rights reserved.
  *
  * SPDX-License-Identifier: Apache-2.0
  *
  * Licensed under the Apache License, Version 2.0 (the License); you may
  * not use this file except in compliance with the License.
  * You may obtain a copy of the License at
  *
  * www.apache.org/licenses/LICENSE-2.0
  *
  * Unless required by applicable law or agreed to in writing, software
  * distributed under the License is distributed on an AS IS BASIS, WITHOUT
  * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
  * See the License for the specific language governing permissions and
  * limitations under the License.
  **********************************************************************************
  */
  
/* Includes ------------------------------------------------------------------ */

#include <string.h>
#include <stdio.h>
#include "md_conf.h"
#include "main.h"
#include "boot.h"
#include "boot_flash.h"
#include "boot_frame.h"
#include "boot_shell.h"

/* Private Macros ------------------------------------------------------------ */

/* Private Constants --------------------------------------------------------- */

/* Private function prototypes ----------------------------------------------- */

/* Private Variables --------------------------------------------------------- */

/** @defgroup Bootloader Bootloader
  * @brief    Eastsoft Bootloader
  * @{
  */

/** @defgroup Base Base
  * @brief    Bootloader Base
  * @{
  */
/** @defgroup Base_Public_Variables Public Variable
  * @brief    Base Public Variable
  * @{
  */
  
/* Public Variables ---------------------------------------------------------- */

timer_env_t g_timer_env = {0};
/**
  * @}
  */

/** @defgroup Base_Private_Functions Private Functions
  * @brief    Base Private_Functions
  * @{
  */
  
/* Private Function ---------------------------------------------------------- */

/**
  * @brief  Systick callback function
  * @retval None
  */
__isr__ void boot_systick_irq_cbk(void)
{
    if (!g_timer_env.state)
        return;

    if ((--g_timer_env.tick == 0) && (g_timer_env.cbk != NULL))
        g_timer_env.cbk();
}

/**
  * @brief  Initialize timer.
  * @retval None
  */
void timer_init(void)
{
    memset(&g_timer_env, 0x0, sizeof(timer_env_t));
    return;
}

/**
  * @brief  Start timer.
  * @param  tick: Timeout in milliseconds.
  * @param  cbk: callback function for timeout.
  * @retval None
  */
void timer_start(uint32_t tick, timer_cbk_t cbk)
{
    tick = (tick == 0) ? 1 : tick;

    __disable_irq();
    g_timer_env.tick  = tick;
    g_timer_env.cbk   = cbk;
    g_timer_env.state = 1;
    __enable_irq();

    return;
}

/**
  * @brief  Stop timer.
  * @retval None
  */
void timer_stop(void)
{
    __disable_irq();
    memset(&g_timer_env, 0x0, sizeof(timer_env_t));
    __enable_irq();

    return;
}

/**
  * @brief  Callback function for timeout.
  * @retval None
  */
void boot_timer_cbk(void)
{
    g_timer_env.timeout = 1;
}

void reset_and_run(uint32_t app_addr)
{
    __disable_irq();
	
    MD_SYSCFG_UNLOCK();
    md_syscfg_set_cpu_boot_addr(app_addr);
    MD_SYSCFG_LOCK();

    __enable_irq();

    csi_cpu_reset(MEXSTATUS_RESET_CORE);
}

/**
  * @brief  command for "version".
  * @retval None
  */
void cmd_version(void)
{
    printf_e("Bootload: 01-00-00-00\r\n");
	printf_e("Compile:" __DATE__ " " __TIME__ "\r\n");
    return;
}

/**
  * @brief  command for "reboot".
  * @retval None
  */
void cmd_reboot(void)
{
    __disable_irq();
	
    MD_SYSCFG_UNLOCK();
	/* make sure reboot to bootloader */
    md_syscfg_set_cpu_boot_addr(0);
    MD_SYSCFG_LOCK();
	
    __enable_irq();
	
    md_rmu_reset_system();
}

static void uart_irq_ctrl(type_func_t status)
{
#ifdef BOOT_ES32VF2
	if(status==ENABLE)
	{
		md_uart_enable_it_rfnempty(BOOT_UARTX);
		csi_vic_enable_irq(BOOT_UARTX_IRQ);
	}
	else 
	{
		md_uart_disable_it_rfnempty(BOOT_UARTX);
		csi_vic_disable_irq(BOOT_UARTX_IRQ);
	}
#endif /* BOOT_ES32VF2 */
    return;
}

/**
  * @brief  command for "download_ram".
  * @retval None
  */
void cmd_download(void)
{
    uart_irq_ctrl(DISABLE);
    printf_e("************* Update Program *************\r\n");
    printf_e("Strat updating afer 3 second !\r\n");
    printf_e("Don't input anything!!!\r\n");
    md_delay_1ms(3000);
	
    flash_usr_page_erase();
	
    boot_image_update();
	
    boot_update_info();
    uart_irq_ctrl(ENABLE);

    return;
}

/**
  * @}
  */
/** @defgroup Base_Public_Functions Public functions
  * @brief    Base Public Functions
  * @{
  */
/**
  * @brief  Enter bootloader.
  * @retval None
  */
void boot_enter(UART_TypeDef*hperh)
{
    timer_init();
    boot_frame_init(hperh);
    shell_init(hperh);
    shell_cmd_insert("version", cmd_version, 0);
    shell_cmd_insert("reboot", cmd_reboot, 0);
    shell_cmd_insert("update", cmd_download, 0);
	
    printf_e("\r\nes#");

    while (1)
    {
        if (shell_rx_flag_get())
        {
            shell_rx_flag_clear();
            shell_task_func(NULL);

            if (g_timer_env.state)
                timer_stop();
        }
    }
}
/**
  * @}
  */
/**
  * @}
  */
/**
  * @}
  */
