/*
 * FreeRTOS Kernel V10.3.1
 * Copyright (C) 2020 Amazon.com, Inc. or its affiliates.  All Rights Reserved.
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy of
 * this software and associated documentation files (the "Software"), to deal in
 * the Software without restriction, including without limitation the rights to
 * use, copy, modify, merge, publish, distribute, sublicense, and/or sell copies of
 * the Software, and to permit persons to whom the Software is furnished to do so,
 * subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in all
 * copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY, FITNESS
 * FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE AUTHORS OR
 * COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER
 * IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN
 * CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.
 *
 * http://www.FreeRTOS.org
 * http://aws.amazon.com/freertos
 *
 * 1 tab == 4 spaces!
 */

/*
 * The FreeRTOS kernel's RISC-V port is split between the the code that is
 * common across all currently supported RISC-V chips (implementations of the
 * RISC-V ISA), and code which tailors the port to a specific RISC-V chip:
 *
 * + The code that is common to all RISC-V chips is implemented in
 *   FreeRTOS\Source\portable\GCC\RISC-V-RV32\portASM.S.  There is only one
 *   portASM.S file because the same file is used no matter which RISC-V chip is
 *   in use.
 *
 * + The code that tailors the kernel's RISC-V port to a specific RISC-V
 *   chip is implemented in freertos_risc_v_chip_specific_extensions.h.  There
 *   is one freertos_risc_v_chip_specific_extensions.h that can be used with any
 *   RISC-V chip that both includes a standard CLINT and does not add to the
 *   base set of RISC-V registers.  There are additional
 *   freertos_risc_v_chip_specific_extensions.h files for RISC-V implementations
 *   that do not include a standard CLINT or do add to the base set of RISC-V
 *   registers.
 *
 * CARE MUST BE TAKEN TO INCLDUE THE CORRECT
 * freertos_risc_v_chip_specific_extensions.h HEADER FILE FOR THE CHIP
 * IN USE.  To include the correct freertos_risc_v_chip_specific_extensions.h
 * header file ensure the path to the correct header file is in the assembler's
 * include path.
 *
 * This freertos_risc_v_chip_specific_extensions.h is for use on RISC-V chips
 * that include a standard CLINT and do not add to the base set of RISC-V
 * registers.
 *
 */
#if __riscv_xlen == 64
	#define portWORD_SIZE 8
	#define store_x sd
	#define load_x ld
#elif __riscv_xlen == 32
	#define store_x sw
	#define load_x lw
	#define portWORD_SIZE 4
#else
	#error Assembler did not define __riscv_xlen
#endif

#ifdef __riscv_32e
	/* x1 x3-x15 pc*/
    #define portCONTEXT_SIZE ( 15 * portWORD_SIZE )
    #define portMEPC_OFFSET  14
#else

#ifndef USE_FPU
	/* x1 x3-x31 pc*/
    #define portCONTEXT_SIZE ( 31 * portWORD_SIZE )
    #define portMEPC_OFFSET  30
#else
	/* x1 x3-x31 pc f0-f31*/
    #define portCONTEXT_SIZE ( 63 * portWORD_SIZE )
    #define portMEPC_OFFSET  30
    #define portFREG_OFFSET  31
#endif
#endif

#include "freertos_risc_v_chip_specific_extensions.h"

/* Check the freertos_risc_v_chip_specific_extensions.h and/or command line
definitions. */
#if defined( portasmHAS_CLINT ) && defined( portasmHAS_MTIME )
	#error The portasmHAS_CLINT constant has been deprecated.  Please replace it with portasmHAS_MTIME.  portasmHAS_CLINT and portasmHAS_MTIME cannot both be defined at once.  See https://www.freertos.org/Using-FreeRTOS-on-RISC-V.html
#endif

#ifndef portasmHAS_MTIME
	#error freertos_risc_v_chip_specific_extensions.h must define portasmHAS_MTIME to either 1 (MTIME clock present) or 0 (MTIME clock not present).  See https://www.freertos.org/Using-FreeRTOS-on-RISC-V.html
#endif


.global xPortStartFirstTask
.global freertos_risc_v_trap_handler
.global tspend_handler
.global pxPortInitialiseStack
.extern pxCurrentTCB
.extern ulPortTrapHandler
.extern vTaskSwitchContext
.extern xTaskIncrementTick
.extern Timer_IRQHandler
.extern pullMachineTimerCompareRegister
.extern pullNextTime
.extern uxTimerIncrementsForOneTick /* size_t type so 32-bit on 32-bit core and 64-bits on 64-bit core. */
.extern xISRStackTop
.extern portasmHANDLE_INTERRUPT

/*-----------------------------------------------------------*/
/* Enable interrupts when returning from the handler */
#define MSTATUS_PRV1 0x1880

/********************************************************************
 * Functions: vPortYield
 *
 ********************************************************************/
.global vPortYield
.type   vPortYield, %function
vPortYield:
    li      t0, 0xE080100C
    lb      t1, (t0)
    li      t2, 0x01
    or      t1, t1, t2
    sb      t1, (t0)

    ret


.align 8
.func
.globl MACHINE_MODE_SOFT_Handler

freertos_risc_v_trap_handler:
    addi    sp,sp,-portCONTEXT_SIZE
    sw      x1,0*portWORD_SIZE (sp)
    sw      x3,1*portWORD_SIZE (sp)
    sw      x4,2*portWORD_SIZE (sp)
    sw      x5,3*portWORD_SIZE (sp)
    sw      x6,4*portWORD_SIZE (sp)
    sw      x7,5*portWORD_SIZE (sp)
    sw      x8,6*portWORD_SIZE (sp)
    sw      x9,7*portWORD_SIZE (sp)
    sw      x10,8*portWORD_SIZE (sp)
    sw      x11,9*portWORD_SIZE (sp)
    sw      x12,10*portWORD_SIZE (sp)
    sw      x13,11*portWORD_SIZE (sp)
    sw      x14,12*portWORD_SIZE (sp)
    sw      x15,13*portWORD_SIZE (sp)
#ifndef __riscv_32e
    sw      x16,14*portWORD_SIZE (sp)
    sw      x17,15*portWORD_SIZE (sp)
    sw      x18,16*portWORD_SIZE (sp)
    sw      x19,17*portWORD_SIZE (sp)
    sw      x20,18*portWORD_SIZE (sp)
    sw      x21,19*portWORD_SIZE (sp)
    sw      x22,20*portWORD_SIZE (sp)
    sw      x23,21*portWORD_SIZE (sp)
    sw      x24,22*portWORD_SIZE (sp)
    sw      x25,23*portWORD_SIZE (sp)
    sw      x26,24*portWORD_SIZE (sp)
    sw      x27,25*portWORD_SIZE (sp)
    sw      x28,26*portWORD_SIZE (sp)
    sw      x29,27*portWORD_SIZE (sp)
    sw      x30,28*portWORD_SIZE (sp)
    sw      x31,29*portWORD_SIZE (sp)
#endif /* __riscv_32e */
    csrr    t0, mepc
    sw      t0, portMEPC_OFFSET*portWORD_SIZE(sp)
#ifdef USE_FPU
    sw      f0,(0+portFREG_OFFSET)*portWORD_SIZE (sp)
    sw      f1,(1+portFREG_OFFSET)*portWORD_SIZE (sp)
    sw      f2,(2+portFREG_OFFSET)*portWORD_SIZE (sp)
    sw      f3,(3+portFREG_OFFSET)*portWORD_SIZE (sp)
    sw      f4,(4+portFREG_OFFSET)*portWORD_SIZE (sp)
    sw      f5,(5+portFREG_OFFSET)*portWORD_SIZE (sp)
    sw      f6,(6+portFREG_OFFSET)*portWORD_SIZE (sp)
    sw      f7,(7+portFREG_OFFSET)*portWORD_SIZE (sp)
    sw      f8,(8+portFREG_OFFSET)*portWORD_SIZE (sp)
    sw      f9,(9+portFREG_OFFSET)*portWORD_SIZE (sp)
    sw      f10,(10+portFREG_OFFSET)*portWORD_SIZE (sp)
    sw      f11,(11+portFREG_OFFSET)*portWORD_SIZE (sp)
    sw      f12,(12+portFREG_OFFSET)*portWORD_SIZE (sp)
    sw      f13,(13+portFREG_OFFSET)*portWORD_SIZE (sp)
    sw      f14,(14+portFREG_OFFSET)*portWORD_SIZE (sp)
    sw      f15,(15+portFREG_OFFSET)*portWORD_SIZE (sp)
    sw      f16,(16+portFREG_OFFSET)*portWORD_SIZE (sp)
    sw      f17,(17+portFREG_OFFSET)*portWORD_SIZE (sp)
    sw      f18,(18+portFREG_OFFSET)*portWORD_SIZE (sp)
    sw      f19,(19+portFREG_OFFSET)*portWORD_SIZE (sp)
    sw      f20,(20+portFREG_OFFSET)*portWORD_SIZE (sp)
    sw      f21,(21+portFREG_OFFSET)*portWORD_SIZE (sp)
    sw      f22,(22+portFREG_OFFSET)*portWORD_SIZE (sp)
    sw      f23,(23+portFREG_OFFSET)*portWORD_SIZE (sp)
    sw      f24,(24+portFREG_OFFSET)*portWORD_SIZE (sp)
    sw      f25,(25+portFREG_OFFSET)*portWORD_SIZE (sp)
    sw      f26,(26+portFREG_OFFSET)*portWORD_SIZE (sp)
    sw      f27,(27+portFREG_OFFSET)*portWORD_SIZE (sp)
    sw      f28,(28+portFREG_OFFSET)*portWORD_SIZE (sp)
    sw      f29,(29+portFREG_OFFSET)*portWORD_SIZE (sp)
    sw      f30,(30+portFREG_OFFSET)*portWORD_SIZE (sp)
    sw      f31,(31+portFREG_OFFSET)*portWORD_SIZE (sp)
#endif /* USE_FPU*/
	
    la      a1, pxCurrentTCB
    lw      a1, (a1)
    sw      sp, (a1)

    jal     vTaskSwitchContext

    la      a1, pxCurrentTCB
    lw      a1, (a1)
    lw      sp, (a1)

    /* Run in machine mode */
    li      t0, MSTATUS_PRV1
    csrs    mstatus, t0

    lw      x1,0*portWORD_SIZE (sp)
    lw      x3,1*portWORD_SIZE (sp)
    lw      x4,2*portWORD_SIZE (sp)
    lw      x5,3*portWORD_SIZE (sp)
    lw      x6,4*portWORD_SIZE (sp)
    lw      x7,5*portWORD_SIZE (sp)
    lw      x8,6*portWORD_SIZE (sp)
    lw      x9,7*portWORD_SIZE (sp)
    lw      x10,8*portWORD_SIZE (sp)
    lw      x11,9*portWORD_SIZE (sp)
    lw      x12,10*portWORD_SIZE (sp)
    lw      x13,11*portWORD_SIZE (sp)
    lw      x14,12*portWORD_SIZE (sp)
    lw      x15,13*portWORD_SIZE (sp)
#ifndef __riscv_32e
    lw      x16,14*portWORD_SIZE (sp)
    lw      x17,15*portWORD_SIZE (sp)
    lw      x18,16*portWORD_SIZE (sp)
    lw      x19,17*portWORD_SIZE (sp)
    lw      x20,18*portWORD_SIZE (sp)
    lw      x21,19*portWORD_SIZE (sp)
    lw      x22,20*portWORD_SIZE (sp)
    lw      x23,21*portWORD_SIZE (sp)
    lw      x24,22*portWORD_SIZE (sp)
    lw      x25,23*portWORD_SIZE (sp)
    lw      x26,24*portWORD_SIZE (sp)
    lw      x27,25*portWORD_SIZE (sp)
    lw      x28,26*portWORD_SIZE (sp)
    lw      x29,27*portWORD_SIZE (sp)
    lw      x30,28*portWORD_SIZE (sp)
    lw      x31,29*portWORD_SIZE (sp)
#endif /* __riscv_32e */
    lw      t0, portMEPC_OFFSET*portWORD_SIZE(sp)
    csrw    mepc,t0
#ifdef USE_FPU
    lw      f0,(0+portFREG_OFFSET)*portWORD_SIZE (sp)
    lw      f1,(1+portFREG_OFFSET)*portWORD_SIZE (sp)
    lw      f2,(2+portFREG_OFFSET)*portWORD_SIZE (sp)
    lw      f3,(3+portFREG_OFFSET)*portWORD_SIZE (sp)
    lw      f4,(4+portFREG_OFFSET)*portWORD_SIZE (sp)
    lw      f5,(5+portFREG_OFFSET)*portWORD_SIZE (sp)
    lw      f6,(6+portFREG_OFFSET)*portWORD_SIZE (sp)
    lw      f7,(7+portFREG_OFFSET)*portWORD_SIZE (sp)
    lw      f8,(8+portFREG_OFFSET)*portWORD_SIZE (sp)
    lw      f9,(9+portFREG_OFFSET)*portWORD_SIZE (sp)
    lw      f10,(10+portFREG_OFFSET)*portWORD_SIZE (sp)
    lw      f11,(11+portFREG_OFFSET)*portWORD_SIZE (sp)
    lw      f12,(12+portFREG_OFFSET)*portWORD_SIZE (sp)
    lw      f13,(13+portFREG_OFFSET)*portWORD_SIZE (sp)
    lw      f14,(14+portFREG_OFFSET)*portWORD_SIZE (sp)
    lw      f15,(15+portFREG_OFFSET)*portWORD_SIZE (sp)
    lw      f16,(16+portFREG_OFFSET)*portWORD_SIZE (sp)
    lw      f17,(17+portFREG_OFFSET)*portWORD_SIZE (sp)
    lw      f18,(18+portFREG_OFFSET)*portWORD_SIZE (sp)
    lw      f19,(19+portFREG_OFFSET)*portWORD_SIZE (sp)
    lw      f20,(20+portFREG_OFFSET)*portWORD_SIZE (sp)
    lw      f21,(21+portFREG_OFFSET)*portWORD_SIZE (sp)
    lw      f22,(22+portFREG_OFFSET)*portWORD_SIZE (sp)
    lw      f23,(23+portFREG_OFFSET)*portWORD_SIZE (sp)
    lw      f24,(24+portFREG_OFFSET)*portWORD_SIZE (sp)
    lw      f25,(25+portFREG_OFFSET)*portWORD_SIZE (sp)
    lw      f26,(26+portFREG_OFFSET)*portWORD_SIZE (sp)
    lw      f27,(27+portFREG_OFFSET)*portWORD_SIZE (sp)
    lw      f28,(28+portFREG_OFFSET)*portWORD_SIZE (sp)
    lw      f29,(29+portFREG_OFFSET)*portWORD_SIZE (sp)
    lw      f30,(30+portFREG_OFFSET)*portWORD_SIZE (sp)
    lw      f31,(31+portFREG_OFFSET)*portWORD_SIZE (sp)
#endif /* USE_FPU*/
    addi    sp,sp,portCONTEXT_SIZE

    mret
.endfunc


.align 8
.func
xPortStartFirstTask:
    la      a0, pxCurrentTCB
    lw      a0, (a0)
    lw      sp, (a0)

    /* Run in machine mode */
    li      t0, MSTATUS_PRV1
    csrs    mstatus, t0

    lw      x1,0*portWORD_SIZE (sp)
    lw      x3,1*portWORD_SIZE (sp)
    lw      x4,2*portWORD_SIZE (sp)
    lw      x5,3*portWORD_SIZE (sp)
    lw      x6,4*portWORD_SIZE (sp)
    lw      x7,5*portWORD_SIZE (sp)
    lw      x8,6*portWORD_SIZE (sp)
    lw      x9,7*portWORD_SIZE (sp)
    lw      x10,8*portWORD_SIZE (sp)
    lw      x11,9*portWORD_SIZE (sp)
    lw      x12,10*portWORD_SIZE (sp)
    lw      x13,11*portWORD_SIZE (sp)
    lw      x14,12*portWORD_SIZE (sp)
    lw      x15,13*portWORD_SIZE (sp)
#ifndef __riscv_32e
    lw      x16,14*portWORD_SIZE (sp)
    lw      x17,15*portWORD_SIZE (sp)
    lw      x18,16*portWORD_SIZE (sp)
    lw      x19,17*portWORD_SIZE (sp)
    lw      x20,18*portWORD_SIZE (sp)
    lw      x21,19*portWORD_SIZE (sp)
    lw      x22,20*portWORD_SIZE (sp)
    lw      x23,21*portWORD_SIZE (sp)
    lw      x24,22*portWORD_SIZE (sp)
    lw      x25,23*portWORD_SIZE (sp)
    lw      x26,24*portWORD_SIZE (sp)
    lw      x27,25*portWORD_SIZE (sp)
    lw      x28,26*portWORD_SIZE (sp)
    lw      x29,27*portWORD_SIZE (sp)
    lw      x30,28*portWORD_SIZE (sp)
    lw      x31,29*portWORD_SIZE (sp)
#endif /* __riscv_32e */
    lw      t0, portMEPC_OFFSET*portWORD_SIZE(sp)
    csrw    mepc,t0
    addi    sp,sp,30*portWORD_SIZE

    mret
.endfunc
/*-----------------------------------------------------------*/

/*
 * Unlike other ports pxPortInitialiseStack() is written in assembly code as it
 * needs access to the portasmADDITIONAL_CONTEXT_SIZE constant.  The prototype
 * for the function is as per the other ports:
 * StackType_t *pxPortInitialiseStack( StackType_t *pxTopOfStack, TaskFunction_t pxCode, void *pvParameters );
 *
 * As per the standard RISC-V ABI pxTopcOfStack is passed in in a0, pxCode in
 * a1, and pvParameters in a2.  The new top of stack is passed out in a0.
 *
 * RISC-V maps registers to ABI names as follows (X1 to X31 integer registers
 * for the 'I' profile, X1 to X15 for the 'E' profile, currently I assumed).
 *
 * Register		ABI Name	Description						Saver
 * x0			zero		Hard-wired zero					-
 * x1			ra			Return address					Caller
 * x2			sp			Stack pointer					Callee
 * x3			gp			Global pointer					-
 * x4			tp			Thread pointer					-
 * x5-7			t0-2		Temporaries						Caller
 * x8			s0/fp		Saved register/Frame pointer	Callee
 * x9			s1			Saved register					Callee
 * x10-11		a0-1		Function Arguments/return values Caller
 * x12-17		a2-7		Function arguments				Caller
 * x18-27		s2-11		Saved registers					Callee
 * x28-31		t3-6		Temporaries						Caller
 *
 * The RISC-V context is saved t FreeRTOS tasks in the following stack frame,
 * where the global and thread pointers are currently assumed to be constant so
 * are not saved:
 *
 * mstatus
 * x31
 * x30
 * x29
 * x28
 * x27
 * x26
 * x25
 * x24
 * x23
 * x22
 * x21
 * x20
 * x19
 * x18
 * x17
 * x16
 * x15
 * x14
 * x13
 * x12
 * x11
 * pvParameters
 * x9
 * x8
 * x7
 * x6
 * x5
 * portTASK_RETURN_ADDRESS
 * [chip specific registers go here]
 * pxCode
 */

/*-----------------------------------------------------------*/
