#include "usbh_core.h"
#include "usbh_hub.h"
#include "ald_usb.h"
#include "md_syscfg.h"
#include "md_rmu.h"

#define ES_USB_PERH_EP_MAX_INDEX     (4U)
#define ES_USB_HOST_DEFAULT_FIFO_SIZE (1024U)
#define ES_USB_HOST_DEFAULT_FIFO_SIZE_PARA ALD_USB_FIFO_SZ_1024
#define ES_USB_HOST_FIFO_STATIC_USED  (ES_USB_PERH_EP_MAX_INDEX * ES_USB_HOST_DEFAULT_FIFO_SIZE)

typedef enum {
    USB_EP0_STATE_SETUP = 0x0,      /**< SETUP DATA */
    USB_EP0_STATE_IN_DATA = 0x1,    /**< IN DATA */
    USB_EP0_STATE_OUT_DATA = 0x3,   /**< OUT DATA */
    USB_EP0_STATE_IN_STATUS = 0x4,  /**< IN status */
    USB_EP0_STATE_OUT_STATUS = 0x5, /**< OUT status */
} ep0_state_t;

struct musb_pipe {
    uint8_t dev_addr;
    uint8_t ep_addr;
    uint8_t ep_type;
    uint8_t ep_interval;
    uint8_t speed;
    uint16_t ep_mps;
    bool inuse;
    uint32_t xfrd;
    volatile bool waiter;
    usb_osal_sem_t waitsem;
    struct usbh_hubport *hport;
    struct usbh_urb *urb;
};

struct musb_hcd {
    volatile bool port_csc;
    volatile bool port_pec;
    volatile bool port_pe;
    struct musb_pipe pipe_pool[ES_USB_PERH_EP_MAX_INDEX + 1][2]; /* Support Bidirectional ep */
} g_musb_hcd;

static volatile uint8_t usb_ep0_state = USB_EP0_STATE_SETUP;

uint16_t es_usbh_get_pipe_info_ep_mps(struct musb_pipe * p)
{
	if(p)
		return p->ep_mps;
	else
		return 0;
}

__weak void usb_hc_low_level_init(void)
{
}

__WEAK void usb_hc_low_level_deinit(void)
{
}

static void es_usbd_ep_write_packet_8bit(uint8_t ep_idx, uint8_t *buffer, uint16_t len)
{
    uint32_t i;
    uint8_t* buf8 = (uint8_t*)(((uint32_t)(&(USB->EP0FIFO))) + ((ep_idx) << 2));

    for(i = 0;i < len;i++)
        *buf8 = buffer[i];
}

static void es_usbd_ep_read_packet_8bit(uint8_t ep_idx, uint8_t *buffer, uint16_t len)
{
    uint32_t i;
    uint8_t* buf8 = (uint8_t*)(((uint32_t)(&(USB->EP0FIFO))) + ((ep_idx) << 2));

    for(i = 0;i < len;i++)
        buffer[i] = *buf8;
}

/* get current active ep */
static inline uint8_t musb_get_active_ep(void)
{
    return USB->INDEX;
}

/* set the active ep */
static inline void musb_set_active_ep(uint8_t ep_index)
{
    USB->INDEX = ep_index;
}

static inline uint32_t es_usbd_ep_tx_ready_state(uint8_t ep_addr)
{
    if (ep_addr == 0U) 
    {
		if (USB->CSR0L_TXCSRL & ALD_USB_CSR0L_TXRDY)
			return 1;
	}
	else 
    {
		if (USB->CSR0L_TXCSRL & ALD_USB_TXCSRL_TXRDY)
			return 2;
	}
    
    return 0;
}

int usb_ep_in_data_avail(uint8_t ep_addr)
{
    uint16_t old_ep_idx, length;
    uint8_t ep_idx = USB_EP_GET_IDX(ep_addr);
    
    old_ep_idx = musb_get_active_ep();
    musb_set_active_ep(ep_idx);
    
    if(es_usbd_ep_tx_ready_state(ep_idx))
        length = 0;
    else
    {
        if(ep_idx == 0)
            length = 64;
        else
            length = USB->TXMAXP;
    }
    
    musb_set_active_ep(old_ep_idx);
    return length;
}

void musb_control_pipe_init(struct musb_pipe *pipe, struct usb_setup_packet *setup, uint8_t *buffer, uint32_t buflen)
{
    uint8_t old_ep_index;

    old_ep_index = musb_get_active_ep();
    musb_set_active_ep(0);
	
	ald_usb_host_addr_set(0, pipe->dev_addr, ALD_USB_EP_HOST_OUT);
	ald_usb_host_ep_config(0, 64, 0, 0, (ALD_USB_EP_MODE_CTRL | ALD_USB_EP_HOST_OUT));

    es_usbd_ep_write_packet_8bit(0, (uint8_t *)setup, 8);
    USB->CSR0L_TXCSRL = (USB_CSR0L_TXRDY_MSK | USB_CSR0L_SETUPPKT_MSK);
    musb_set_active_ep(old_ep_index);
}

void musb_bulk_pipe_init(struct musb_pipe *pipe, uint8_t *buffer, uint32_t buflen)
{
    uint8_t ep_idx;
    uint8_t old_ep_index;

    ep_idx = pipe->ep_addr & 0x7f;
    old_ep_index = musb_get_active_ep();
    musb_set_active_ep(ep_idx);
	
	ald_usb_host_addr_set(ep_idx, pipe->dev_addr, 0);
	
    if (pipe->ep_addr & 0x80) 
	{
		ald_usb_host_ep_config(ep_idx, pipe->ep_mps, pipe->ep_interval, ep_idx, (ALD_USB_EP_MODE_BULK | ALD_USB_EP_HOST_IN));
        USB->RXCSRL |= USB_RXCSRL_REQPKT_MSK;
    } 
	else 
	{
		ald_usb_host_ep_config(ep_idx, pipe->ep_mps, pipe->ep_interval, ep_idx, (ALD_USB_EP_MODE_BULK | ALD_USB_EP_HOST_OUT));

        if (buflen > pipe->ep_mps)
            buflen = pipe->ep_mps;

        es_usbd_ep_write_packet_8bit(ep_idx, buffer, buflen);
		USB->CSR0L_TXCSRL = USB_TXCSRL_TXRDY_MSK;
    }
	
    musb_set_active_ep(old_ep_index);
}

void musb_intr_pipe_init(struct musb_pipe *pipe, uint8_t *buffer, uint32_t buflen)
{
    uint8_t ep_idx;
    uint8_t old_ep_index;

    ep_idx = pipe->ep_addr & 0x7f;
    old_ep_index = musb_get_active_ep();
    musb_set_active_ep(ep_idx);
	
	ald_usb_host_addr_set(ep_idx, pipe->dev_addr, 0);

    if (pipe->ep_addr & 0x80) 
	{
		ald_usb_host_ep_config(ep_idx, pipe->ep_mps, pipe->ep_interval, ep_idx, (ALD_USB_EP_MODE_INT | ALD_USB_EP_HOST_IN));
        USB->RXCSRL |= ALD_USB_RXCSRL_REQPKT;
    } 
	else 
	{
		ald_usb_host_ep_config(ep_idx, pipe->ep_mps, pipe->ep_interval, ep_idx, (ALD_USB_EP_MODE_INT | ALD_USB_EP_HOST_OUT));

        if (buflen > pipe->ep_mps)
            buflen = pipe->ep_mps;

        es_usbd_ep_write_packet_8bit(ep_idx, buffer, buflen);
        USB->CSR0L_TXCSRL = USB_TXCSRL_TXRDY_MSK;
    }
    
    musb_set_active_ep(old_ep_index);
}

void musb_iso_pipe_init(struct musb_pipe *pipe, uint8_t *buffer, uint32_t buflen)
{
    uint8_t ep_idx;
    uint8_t old_ep_index;

    ep_idx = pipe->ep_addr & 0x7f;
    old_ep_index = musb_get_active_ep();
    musb_set_active_ep(ep_idx);
	
	ald_usb_host_addr_set(ep_idx, pipe->dev_addr, 0);

    if (pipe->ep_addr & 0x80) 
	{
		ald_usb_host_ep_config(ep_idx, pipe->ep_mps, pipe->ep_interval, ep_idx, (ALD_USB_EP_MODE_ISOC | ALD_USB_EP_HOST_IN));
        USB->RXCSRL |= ALD_USB_RXCSRL_REQPKT;
    } 
	else 
	{
		ald_usb_host_ep_config(ep_idx, pipe->ep_mps, pipe->ep_interval, ep_idx, (ALD_USB_EP_MODE_ISOC | ALD_USB_EP_HOST_OUT));

        if (buflen > pipe->ep_mps)
            buflen = pipe->ep_mps;

        es_usbd_ep_write_packet_8bit(ep_idx, buffer, buflen);
        USB->CSR0L_TXCSRL = USB_TXCSRL_TXRDY_MSK;
    }
    
    musb_set_active_ep(old_ep_index);
}

static int usbh_reset_port(const uint8_t port)
{
    g_musb_hcd.port_pe = 0;
    ald_usb_host_reset(1);
    usb_osal_msleep(20);
    ald_usb_host_reset(0);
    usb_osal_msleep(20);
    g_musb_hcd.port_pe = 1;
    return 0;
}

static uint8_t usbh_get_port_speed(const uint8_t port)
{
    uint8_t speed = USB_SPEED_UNKNOWN;
	
	if (USB->DEVCON & ALD_USB_DEVCON_FSDEV)
		return USB_SPEED_FULL;

	if (USB->DEVCON & ALD_USB_DEVCON_LSDEV)
		return USB_SPEED_LOW;

	return speed;
}

static void es_usbh_config_usb_perh(void)
{
	volatile uint32_t i;
	uint16_t fifo_sddr = 0U;
	
	/* Disable usb interrupt*/
	ald_usb_int_unregister();
	
	MD_SYSCFG_UNLOCK();
	md_rmu_enable_usb_reset();
	MD_SYSCFG_LOCK();
	
	for(i = 0;i < 0x3FFFF;)
		i++;
	
	/* Config EP0 */
	ald_usb_host_ep_config(ALD_USB_EP_0, 64, 0, 0, (ALD_USB_EP_MODE_CTRL | ALD_USB_EP_SPEED_FULL | ALD_USB_EP_HOST_OUT));
	
	/* Clear interrupts */
	ald_usb_int_status_get();
	/* Init interrupts */
	ald_usb_int_enable(ALD_USB_INTCTRL_SESSION | ALD_USB_INTCTRL_DISCONNECT | ALD_USB_INTCTRL_CONNECT |
				ALD_USB_INTCTRL_BABBLE | ALD_USB_INTCTRL_RESUME);
	ald_usb_int_enable_ep(ALD_USB_INTEP_ALL);
    
	/* Enable PHY power */
	ald_usb_host_pwr_enable();
	/* clear hnp session */
	ald_usb_otg_session_request(false);

	/* set vbus control mode and threshold value */
	ald_usb_swvbus_sigctl_set(1);
	ald_usb_swvbus_sesendth_set(1);
	ald_usb_swvbus_sesvalth_set(1);
	ald_usb_swvbus_valth_set(1);
	/* Pull down DP and DM */
	ald_usb_dppud_set(ALD_USB_DPDM_PUSH_DOWN);
	ald_usb_dmpud_set(ALD_USB_DPDM_PUSH_DOWN);
	
	/* software control CID */
	ald_usb_swcid_cidctrl(1);
	/* force to host mode */
	ald_usb_swcid_host(0);
	
	/* start host request */
	ald_usb_mode_host_req();
	/* Start hnp */
	ald_usb_otg_session_request(true);
	
	/*分配ep的默认USB_FIFO*/
	for(i = 1;i <= ES_USB_PERH_EP_MAX_INDEX;i++)
	{
		ald_usb_fifo_config_set(i , fifo_sddr , ES_USB_HOST_DEFAULT_FIFO_SIZE_PARA  , ALD_USB_EP_HOST_OUT);
		ald_usb_fifo_config_set(i , fifo_sddr , ES_USB_HOST_DEFAULT_FIFO_SIZE_PARA  , ALD_USB_EP_HOST_IN);
		fifo_sddr += ES_USB_HOST_DEFAULT_FIFO_SIZE;
	}
	
	/* Enable usb interrupt*/
	ald_usb_int_register();
}

int usb_hc_init(void)
{
    memset(&g_musb_hcd, 0, sizeof(struct musb_hcd));

    for (uint8_t i = 0; i < ES_USB_PERH_EP_MAX_INDEX; i++) {
        g_musb_hcd.pipe_pool[i][0].waitsem = usb_osal_sem_create(0);
        g_musb_hcd.pipe_pool[i][1].waitsem = usb_osal_sem_create(0);
    }

    usb_hc_low_level_init();

	es_usbh_config_usb_perh();
	
    return 0;
}

int usb_hc_deinit(void)
{
    return 0;
}

int usbh_roothub_control(struct usb_setup_packet *setup, uint8_t *buf)
{
    uint8_t nports;
    uint8_t port;
    uint32_t status;

    nports = CONFIG_USBHOST_MAX_RHPORTS;
    port = setup->wIndex;
    if (setup->bmRequestType & USB_REQUEST_RECIPIENT_DEVICE) {
        switch (setup->bRequest) {
            case HUB_REQUEST_CLEAR_FEATURE:
                switch (setup->wValue) {
                    case HUB_FEATURE_HUB_C_LOCALPOWER:
                        break;
                    case HUB_FEATURE_HUB_C_OVERCURRENT:
                        break;
                    default:
                        return -EPIPE;
                }
                break;
            case HUB_REQUEST_SET_FEATURE:
                switch (setup->wValue) {
                    case HUB_FEATURE_HUB_C_LOCALPOWER:
                        break;
                    case HUB_FEATURE_HUB_C_OVERCURRENT:
                        break;
                    default:
                        return -EPIPE;
                }
                break;
            case HUB_REQUEST_GET_DESCRIPTOR:
                break;
            case HUB_REQUEST_GET_STATUS:
                memset(buf, 0, 4);
                break;
            default:
                break;
        }
    } else if (setup->bmRequestType & USB_REQUEST_RECIPIENT_OTHER) {
        switch (setup->bRequest) {
            case HUB_REQUEST_CLEAR_FEATURE:
                if (!port || port > nports) {
                    return -EPIPE;
                }

                switch (setup->wValue) {
                    case HUB_PORT_FEATURE_ENABLE:
                        break;
                    case HUB_PORT_FEATURE_SUSPEND:
                    case HUB_PORT_FEATURE_C_SUSPEND:
                        break;
                    case HUB_PORT_FEATURE_POWER:
                        break;
                    case HUB_PORT_FEATURE_C_CONNECTION:
                        g_musb_hcd.port_csc = 0;
                        break;
                    case HUB_PORT_FEATURE_C_ENABLE:
                        g_musb_hcd.port_pec = 0;
                        break;
                    case HUB_PORT_FEATURE_C_OVER_CURREN:
                        break;
                    case HUB_PORT_FEATURE_C_RESET:
                        break;
                    default:
                        return -EPIPE;
                }
                break;
            case HUB_REQUEST_SET_FEATURE:
                if (!port || port > nports) {
                    return -EPIPE;
                }

                switch (setup->wValue) {
                    case HUB_PORT_FEATURE_SUSPEND:
                        break;
                    case HUB_PORT_FEATURE_POWER:
                        break;
                    case HUB_PORT_FEATURE_RESET:
                        usbh_reset_port(port);
                        break;

                    default:
                        return -EPIPE;
                }
                break;
            case HUB_REQUEST_GET_STATUS:
                if (!port || port > nports) {
                    return -EPIPE;
                }

                status = 0;
                if (g_musb_hcd.port_csc) {
                    status |= (1 << HUB_PORT_FEATURE_C_CONNECTION);
                }
                if (g_musb_hcd.port_pec) {
                    status |= (1 << HUB_PORT_FEATURE_C_ENABLE);
                }

                if (g_musb_hcd.port_pe) {
                    status |= (1 << HUB_PORT_FEATURE_CONNECTION);
                    status |= (1 << HUB_PORT_FEATURE_ENABLE);
                    if (usbh_get_port_speed(port) == USB_SPEED_LOW) {
                        status |= (1 << HUB_PORT_FEATURE_LOWSPEED);
                    } else if (usbh_get_port_speed(port) == USB_SPEED_HIGH) {
                        status |= (1 << HUB_PORT_FEATURE_HIGHSPEED);
                    }
                }

                memcpy(buf, &status, 4);
                break;
            default:
                break;
        }
    }
    return 0;
}

int usbh_ep_pipe_reconfigure(usbh_pipe_t pipe, uint8_t dev_addr, uint8_t ep_mps, uint8_t mult)
{
    struct musb_pipe *ppipe = (struct musb_pipe *)pipe;

    ppipe->dev_addr = dev_addr;
    ppipe->ep_mps = ep_mps;

    return 0;
}

int usbh_pipe_alloc(usbh_pipe_t *pipe, const struct usbh_endpoint_cfg *ep_cfg)
{
    struct musb_pipe *ppipe;
    uint8_t old_ep_index;
    uint8_t ep_idx;
    usb_osal_sem_t waitsem;

    ep_idx = ep_cfg->ep_addr & 0x7f;

    if (ep_idx > ES_USB_PERH_EP_MAX_INDEX) {
        return -ENOMEM;
    }

    old_ep_index = musb_get_active_ep();
    musb_set_active_ep(ep_idx);

    if (ep_cfg->ep_addr & 0x80) {
        ppipe = &g_musb_hcd.pipe_pool[ep_idx][1];
    } else {
        ppipe = &g_musb_hcd.pipe_pool[ep_idx][0];
    }

    /* store variables */
    waitsem = ppipe->waitsem;

    memset(ppipe, 0, sizeof(struct musb_pipe));

    ppipe->ep_addr = ep_cfg->ep_addr;
    ppipe->ep_type = ep_cfg->ep_type;
    ppipe->ep_mps = ep_cfg->ep_mps;
    ppipe->ep_interval = ep_cfg->ep_interval;
    ppipe->speed = ep_cfg->hport->speed;
    ppipe->dev_addr = ep_cfg->hport->dev_addr;
    ppipe->hport = ep_cfg->hport;

    if (ep_cfg->ep_type == USB_ENDPOINT_TYPE_CONTROL) 
	{
		
    } 
	else 
	{
		if((ep_cfg->ep_mps) > ES_USB_HOST_DEFAULT_FIFO_SIZE)
		{
			USB_LOG_ERR("please change usb_fifo");
		}
		else
		{
			
		}
			
        if (ppipe->ep_addr & 0x80) 
            USB->RXIER = (1 << ep_idx);
        else
            USB->TXIER = (1 << ep_idx);
    }
    /* restore variable */
    ppipe->inuse = true;
    ppipe->waitsem = waitsem;

    musb_set_active_ep(old_ep_index);
    *pipe = (usbh_pipe_t)ppipe;
    return 0;
}

int usbh_pipe_free(usbh_pipe_t pipe)
{
    struct musb_pipe *ppipe;
    struct usbh_urb *urb;

    ppipe = (struct musb_pipe *)pipe;

    if (!ppipe) {
        return -EINVAL;
    }

    urb = ppipe->urb;

    if (urb) {
        usbh_kill_urb(urb);
    }

    return 0;
}

int usbh_submit_urb(struct usbh_urb *urb)
{
    struct musb_pipe *pipe;
    size_t flags;
    int ret = 0;

    if (!urb) {
        return -EINVAL;
    }

    pipe = urb->pipe;

    if (!pipe) {
        return -EINVAL;
    }

    if (!pipe->hport->connected) {
        return -ENODEV;
    }

    if (pipe->urb) {
        return -EBUSY;
    }

    flags = usb_osal_enter_critical_section();

    pipe->waiter = false;
    pipe->xfrd = 0;
    pipe->urb = urb;
    urb->errorcode = -EBUSY;
    urb->actual_length = 0;

    if (urb->timeout > 0) {
        pipe->waiter = true;
    }
    usb_osal_leave_critical_section(flags);

    switch (pipe->ep_type) {
        case USB_ENDPOINT_TYPE_CONTROL:
            usb_ep0_state = USB_EP0_STATE_SETUP;
            musb_control_pipe_init(pipe, urb->setup, urb->transfer_buffer, urb->transfer_buffer_length);
            break;
        case USB_ENDPOINT_TYPE_BULK:
            musb_bulk_pipe_init(pipe, urb->transfer_buffer, urb->transfer_buffer_length);
            break;
        case USB_ENDPOINT_TYPE_INTERRUPT:
            musb_intr_pipe_init(pipe, urb->transfer_buffer, urb->transfer_buffer_length);
            break;
        case USB_ENDPOINT_TYPE_ISOCHRONOUS:
            musb_iso_pipe_init(pipe, urb->transfer_buffer, urb->transfer_buffer_length);            break;
        default:
            break;
    }
    if (urb->timeout > 0) {
        /* wait until timeout or sem give */
        ret = usb_osal_sem_take(pipe->waitsem, urb->timeout);
        if (ret < 0) {
            goto errout_timeout;
        }

        ret = urb->errorcode;
    }
    return ret;
errout_timeout:
    pipe->waiter = false;
    usbh_kill_urb(urb);
    return ret;
}

int usbh_kill_urb(struct usbh_urb *urb)
{
    struct musb_pipe *pipe;

    pipe = urb->pipe;

    if (!urb || !pipe) {
        return -EINVAL;
    }

    if (pipe->waiter) {
        pipe->waiter = false;
        urb->errorcode = -ESHUTDOWN;
        usb_osal_sem_give(pipe->waitsem);
    }

    return 0;
}

static inline void musb_pipe_waitup(struct musb_pipe *pipe)
{
    struct usbh_urb *urb;

    urb = pipe->urb;
    pipe->urb = NULL;

    if (pipe->waiter) {
        pipe->waiter = false;
        usb_osal_sem_give(pipe->waitsem);
    }

    if (urb->complete) {
        if (urb->errorcode < 0) {
            urb->complete(urb->arg, urb->errorcode);
        } else {
            urb->complete(urb->arg, urb->actual_length);
        }
    }
}
void handle_ep0(void)
{
    struct musb_pipe *pipe;
    struct usbh_urb *urb;
    uint32_t size;

    pipe = (struct musb_pipe *)&g_musb_hcd.pipe_pool[0][0];
    urb = pipe->urb;
    if (urb == NULL) {
        return;
    }

    musb_set_active_ep(0);
	
    if ((USB->CSR0L_TXCSRL) & USB_CSR0L_STALLED_MSK) 
	{
        USB->CSR0L_TXCSRL &= ~USB_CSR0L_STALLED_MSK;
        usb_ep0_state = USB_EP0_STATE_SETUP;
        urb->errorcode = -EPERM;
        musb_pipe_waitup(pipe);
        return;
    }
    if ((USB->CSR0L_TXCSRL) & USB_CSR0L_ERROR_MSK) 
	{
        USB->CSR0L_TXCSRL &= ~USB_CSR0L_ERROR_MSK;
        ald_usb_fifo_flush(0,0);
        usb_ep0_state = USB_EP0_STATE_SETUP;
        urb->errorcode = -EIO;
        musb_pipe_waitup(pipe);
        return;
    }

    switch (usb_ep0_state) {
        case USB_EP0_STATE_SETUP:
            urb->actual_length += 8;
            if (urb->transfer_buffer_length) {
                if (urb->setup->bmRequestType & 0x80) {
                    usb_ep0_state = USB_EP0_STATE_IN_DATA;
                    USB->CSR0L_TXCSRL = USB_CSR0L_REQPKT_MSK;
                } else {
                    usb_ep0_state = USB_EP0_STATE_OUT_DATA;
                    size = urb->transfer_buffer_length;
                    if (size > pipe->ep_mps) {
                        size = pipe->ep_mps;
                    }

                    es_usbd_ep_write_packet_8bit(0, urb->transfer_buffer, size);
                    USB->CSR0L_TXCSRL = USB_CSR0L_TXRDY_MSK;

                    urb->transfer_buffer += size;
                    urb->transfer_buffer_length -= size;
                    urb->actual_length += size;
                }
            } 
			else {
                usb_ep0_state = USB_EP0_STATE_IN_STATUS;
                USB->CSR0L_TXCSRL = (USB_CSR0L_REQPKT_MSK | USB_CSR0L_STATUSPKT_MSK);
            }
            break;
        case USB_EP0_STATE_IN_DATA:
            if ((USB->CSR0L_TXCSRL) & USB_CSR0L_RXRDY_MSK) 
			{
                size = urb->transfer_buffer_length;
                if (size > pipe->ep_mps) {
                    size = pipe->ep_mps;
				}

                size = MIN(size, (ald_usb_ep_data_avail(0)));
                es_usbd_ep_read_packet_8bit(0, urb->transfer_buffer, size);
                USB->CSR0L_TXCSRL &= ~USB_CSR0L_RXRDY_MSK;
                urb->transfer_buffer += size;
                urb->transfer_buffer_length -= size;
                urb->actual_length += size;

                if ((size < pipe->ep_mps) || (urb->transfer_buffer_length == 0)) {
                    usb_ep0_state = USB_EP0_STATE_OUT_STATUS;
                    USB->CSR0L_TXCSRL = (USB_CSR0L_TXRDY_MSK | USB_CSR0L_STATUSPKT_MSK);
                } else {
                    USB->CSR0L_TXCSRL = USB_CSR0L_REQPKT_MSK;
                }
            }
            break;
        case USB_EP0_STATE_OUT_DATA:
            if (urb->transfer_buffer_length > 0) {
                size = urb->transfer_buffer_length;
                if (size > pipe->ep_mps) {
                    size = pipe->ep_mps;
                }

                es_usbd_ep_write_packet_8bit(0, urb->transfer_buffer, size);
                USB->CSR0L_TXCSRL |= USB_CSR0L_TXRDY_MSK;

                urb->transfer_buffer += size;
                urb->transfer_buffer_length -= size;
                urb->actual_length += size;
            } else {
                usb_ep0_state = USB_EP0_STATE_IN_STATUS;
				USB->CSR0L_TXCSRL = (USB_CSR0L_REQPKT_MSK | USB_CSR0L_STATUSPKT_MSK);
            }
            break;
        case USB_EP0_STATE_OUT_STATUS:
            urb->errorcode = 0;
            musb_pipe_waitup(pipe);
            break;
        case USB_EP0_STATE_IN_STATUS:
            if ((USB->CSR0L_TXCSRL) & (USB_CSR0L_RXRDY_MSK | USB_CSR0L_STATUSPKT_MSK)) 
			{
                USB->CSR0L_TXCSRL &= ~(USB_CSR0L_RXRDY_MSK | USB_CSR0L_STATUSPKT_MSK);
                urb->errorcode = 0;
                musb_pipe_waitup(pipe);
            }
            break;
    }
}

const uint8_t __lowest_bit_bitmap[] =
{
    /* 00 */ 0, 0, 1, 0, 2, 0, 1, 0, 3, 0, 1, 0, 2, 0, 1, 0,
    /* 10 */ 4, 0, 1, 0, 2, 0, 1, 0, 3, 0, 1, 0, 2, 0, 1, 0,
    /* 20 */ 5, 0, 1, 0, 2, 0, 1, 0, 3, 0, 1, 0, 2, 0, 1, 0,
    /* 30 */ 4, 0, 1, 0, 2, 0, 1, 0, 3, 0, 1, 0, 2, 0, 1, 0,
    /* 40 */ 6, 0, 1, 0, 2, 0, 1, 0, 3, 0, 1, 0, 2, 0, 1, 0,
    /* 50 */ 4, 0, 1, 0, 2, 0, 1, 0, 3, 0, 1, 0, 2, 0, 1, 0,
    /* 60 */ 5, 0, 1, 0, 2, 0, 1, 0, 3, 0, 1, 0, 2, 0, 1, 0,
    /* 70 */ 4, 0, 1, 0, 2, 0, 1, 0, 3, 0, 1, 0, 2, 0, 1, 0,
    /* 80 */ 7, 0, 1, 0, 2, 0, 1, 0, 3, 0, 1, 0, 2, 0, 1, 0,
    /* 90 */ 4, 0, 1, 0, 2, 0, 1, 0, 3, 0, 1, 0, 2, 0, 1, 0,
    /* A0 */ 5, 0, 1, 0, 2, 0, 1, 0, 3, 0, 1, 0, 2, 0, 1, 0,
    /* B0 */ 4, 0, 1, 0, 2, 0, 1, 0, 3, 0, 1, 0, 2, 0, 1, 0,
    /* C0 */ 6, 0, 1, 0, 2, 0, 1, 0, 3, 0, 1, 0, 2, 0, 1, 0,
    /* D0 */ 4, 0, 1, 0, 2, 0, 1, 0, 3, 0, 1, 0, 2, 0, 1, 0,
    /* E0 */ 5, 0, 1, 0, 2, 0, 1, 0, 3, 0, 1, 0, 2, 0, 1, 0,
    /* F0 */ 4, 0, 1, 0, 2, 0, 1, 0, 3, 0, 1, 0, 2, 0, 1, 0
};

void __attribute__((interrupt)) USB_Handler(void)
{
    uint8_t old_ep_idx;
    uint8_t ep_idx;
    uint8_t status;
    struct musb_pipe *pipe;
    struct usbh_urb *urb;

	status = USB->IFM & 0x7F;
	USB->ICR |= status;
    
    old_ep_idx = musb_get_active_ep();
	
    if (status & USB_IFM_DISCONIFM_MSK) 
    {
        g_musb_hcd.port_csc = 1;
        g_musb_hcd.port_pec = 1;
        g_musb_hcd.port_pe = 0;
        usbh_roothub_thread_wakeup(1);
		
		es_usbh_config_usb_perh();
    }
                                
    if (status & USB_IFM_CONIFM_MSK) 
    {
        g_musb_hcd.port_csc = 1;
        g_musb_hcd.port_pec = 1;
        g_musb_hcd.port_pe = 1;
        usbh_roothub_thread_wakeup(1);
    }


    if (status & USB_IFM_BABIFM_MSK) 
    {
    }

    if (status & (USB_IFM_SOFIFM_MSK)) 
    {
    }

    if (status & USB_IFM_RESIFM_MSK) 
    {
    }

    if (status & USB_IFM_SESREQIFM_MSK) 
    {
    }
	
    while (USB->TXIFM) 
    {                                      
        ep_idx = __lowest_bit_bitmap[USB->TXIFM];
        USB->TXICR = 1U << ep_idx;  
        musb_set_active_ep(ep_idx);
        
        if(ep_idx == 0U)
		{
			USB->TXIER = USB_TXIDR_EP0ID_MSK;
            handle_ep0();
		}
        else
        {
            pipe = &g_musb_hcd.pipe_pool[ep_idx][0];
            urb = pipe->urb;
			
			if ((USB->CSR0L_TXCSRL) & USB_CSR0L_ERROR_MSK) 
			{
                USB->CSR0L_TXCSRL &= ~USB_CSR0L_ERROR_MSK;
                urb->errorcode = -EIO;
            } 
			else if ((USB->CSR0L_TXCSRL) & USB_CSR0L_NAKTO_MSK) 
			{
                USB->CSR0L_TXCSRL &= ~USB_CSR0L_NAKTO_MSK;
                urb->errorcode = -EBUSY;
            } 
			else if ((USB->CSR0L_TXCSRL) & USB_TXCSRL_STALLED_MSK) 
			{
                USB->CSR0L_TXCSRL &= ~USB_TXCSRL_STALLED_MSK;
                urb->errorcode = -EPERM;
            }
			else
			{
                uint32_t size = urb->transfer_buffer_length;

                if (size > pipe->ep_mps) {
                    size = pipe->ep_mps;
                }

                urb->transfer_buffer += size;
                urb->transfer_buffer_length -= size;
                urb->actual_length += size;

                if (urb->transfer_buffer_length == 0) 
				{
                    urb->errorcode = 0;
                } 
				else 
				{
					es_usbd_ep_write_packet_8bit(ep_idx, urb->transfer_buffer, size);
                    USB->CSR0L_TXCSRL |= USB_TXCSRL_TXRDY_MSK;
                }
			}
			
			if(((urb->errorcode) != (-EBUSY))||((urb->transfer_buffer_length) == 0))
				musb_pipe_waitup(pipe);
        }
    }

    while (USB->RXIFM) 
    {                                  
        ep_idx = __lowest_bit_bitmap[USB->RXIFM];  
        USB->RXICR = 1U << ep_idx;  
        musb_set_active_ep(ep_idx);
		
		pipe = &g_musb_hcd.pipe_pool[ep_idx][1];
		urb = pipe->urb;
		
		if((USB->RXCSRL) & USB_RXCSRL_ERROR_MSK)
		{
			USB->RXCSRL &= ~USB_RXCSRL_ERROR_MSK;
			urb->errorcode = -EIO;
		}
		else if((USB->RXCSRL) & USB_RXCSRL_NAKTO_MSK)
		{
			USB->RXCSRL &= ~USB_RXCSRL_NAKTO_MSK;
			urb->errorcode = -EBUSY;
		}
		else if((USB->RXCSRL) & USB_RXCSRL_STALLED_MSK)
		{
			USB->RXCSRL &= ~USB_RXCSRL_STALLED_MSK;
			urb->errorcode = -EPERM;
		}
		else
		{
			uint32_t size = ald_usb_ep_data_avail(ep_idx);
			
			if (size > pipe->ep_mps)
				size = pipe->ep_mps;
				
			size = MIN(size, (urb->transfer_buffer_length));
			
			es_usbd_ep_read_packet_8bit(ep_idx, urb->transfer_buffer, size);
			
			USB->RXCSRL &= ~( USB_RXCSRL_ERROR_MSK | USB_RXCSRL_STALLED_MSK);
			USB->RXCSRL &= ~USB_RXCSRL_RXRDY_MSK;
			
			urb->transfer_buffer += size;
			urb->transfer_buffer_length -= size;
			urb->actual_length += size;
			
			if ((size < pipe->ep_mps) || ((urb->transfer_buffer_length) < (pipe->ep_mps))) 
				urb->errorcode = 0;
			else 
				USB->RXCSRL |= USB_RXCSRL_REQPKT_MSK;
		}
		
		if(((urb->errorcode) != (-EBUSY))||((urb->transfer_buffer_length) < (pipe->ep_mps)))
			musb_pipe_waitup(pipe);
    }

    musb_set_active_ep(old_ep_idx);
}
