/**********************************************************************************
 *
 * @file    micro_boot.c
 * @brief   description for microboot state machine
 *
 * @version V1.0
 * @date    22 Feb  2023
 * @author  AE Team
 * @note
 *          Change Logs:
 *          Date            Author          Notes
 *          22 Feb  2023    shicc           version:1.0.1   Notes:Migrate the enumeration types and structure types to the header file micro_boot
 *
 * Copyright (C) Shanghai Eastsoft Microelectronics Co. Ltd. All rights reserved.
 *
 * SPDX-License-Identifier: Apache-2.0
 *
 * Licensed under the Apache License, Version 2.0 (the License); you may
 * not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an AS IS BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 **********************************************************************************
 */

/* Includes ----------------------------------------------------------------- */

#include "micro_boot.h"

/* Private Macros ----------------------------------------------------------- */
#define     ES_CMD_ERR_HANDLE1  {return;}

/* Public Variables ---------------------------------------------------------- */
isp_data_t g_isp_data;
fsm_data_type_t g_fsm_data;
/* Private Variables --------------------------------------------------------- */
static uint8_t s_cmd_handle_func_addr_index;
static uint8_t s_cmd_handle_func_step;

static void es_handle_cmd_read_m(void);
static void es_handle_cmd_go(void);
static void es_handle_cmd_write_m(void);
static void es_handle_cmd_ex_erase(void);
static void es_handle_cmd_get_crc32(void);
static void es_handle_cmd_check_empty(void);

void (* const s_cmd_handle_func_addr_buf[])(void) =
{
#if ES_MICRO_BOOT_CONFIG__SUPPORT_CMD_READ_M
    es_handle_cmd_read_m,
#endif/*ES_MICRO_BOOT_CONFIG__SUPPORT_CMD_READ_M*/
#if ES_MICRO_BOOT_CONFIG__SUPPORT_CMD_GO
    es_handle_cmd_go,
#endif/*ES_MICRO_BOOT_CONFIG__SUPPORT_CMD_GO*/
#if ES_MICRO_BOOT_CONFIG__SUPPORT_CMD_WRITE_M
    es_handle_cmd_write_m,
#endif/*ES_MICRO_BOOT_CONFIG__SUPPORT_CMD_WRITE_M*/
#if ES_MICRO_BOOT_CONFIG__SUPPORT_CMD_EX_ERASE
    es_handle_cmd_ex_erase,
#endif/*ES_MICRO_BOOT_CONFIG__SUPPORT_CMD_EX_ERASE*/
#if ES_MICRO_BOOT_CONFIG__SUPPORT_CMD_GET_CRC32
    es_handle_cmd_get_crc32,
#endif/*ES_MICRO_BOOT_CONFIG__SUPPORT_CMD_GET_CRC32*/
#if ES_MICRO_BOOT_CONFIG__SUPPORT_CMD_CHECK_EMPTY
    es_handle_cmd_check_empty,
#endif/*ES_MICRO_BOOT_CONFIG__SUPPORT_CMD_CHECK_EMPTY*/
};

typedef enum
{
    ES_INDEX_CMD_NONE = 0,
#if ES_MICRO_BOOT_CONFIG__SUPPORT_CMD_READ_M
    ES_INDEX_CMD_READ_M,
#endif/*ES_MICRO_BOOT_CONFIG__SUPPORT_CMD_READ_M*/
#if ES_MICRO_BOOT_CONFIG__SUPPORT_CMD_GO
    ES_INDEX_CMD_GO,
#endif/*ES_MICRO_BOOT_CONFIG__SUPPORT_CMD_GO*/
#if ES_MICRO_BOOT_CONFIG__SUPPORT_CMD_WRITE_M
    ES_INDEX_CMD_WRITE_M,
#endif/*ES_MICRO_BOOT_CONFIG__SUPPORT_CMD_WRITE_M*/
#if ES_MICRO_BOOT_CONFIG__SUPPORT_CMD_EX_ERASE
    ES_INDEX_CMD_EX_ERASE,
#endif/*ES_MICRO_BOOT_CONFIG__SUPPORT_CMD_EX_ERASE*/
#if ES_MICRO_BOOT_CONFIG__SUPPORT_CMD_GET_CRC32
    ES_INDEX_CMD_GET_CRC32,
#endif/*ES_MICRO_BOOT_CONFIG__SUPPORT_CMD_GET_CRC32*/
#if ES_MICRO_BOOT_CONFIG__SUPPORT_CMD_CHECK_EMPTY
    ES_INDEX_CMD_CHECK_EMPTY,
#endif/*ES_MICRO_BOOT_CONFIG__SUPPORT_CMD_CHECK_EMPTY*/
} es_cmd_index_step_t;

static const uint8_t s_cmd_handle_func_addr_buf_size = (sizeof(s_cmd_handle_func_addr_buf)) >> 2;

/* Private Constants --------------------------------------------------------- */

/* Private function prototypes ----------------------------------------------- */

/* Private Function---------------------------------------------------------- */

/**
  * @brief  reverse the oder of array
  * @param  None
  * @retval None
  */
static void reverse_fill_byte_arr(u_b32b08_t *para, uint8_t deeplength)
{
    uint8_t i;

    para->b32 = 0;

    for (i = 0; i < deeplength; i++)
    {
        para->b08[i] = g_fsm_data.rx_buf[deeplength - i - 1];
    }

    return;
}

/**
  * @brief  calculate the xor sum
  * @param
  *        start: the start address of an array
  *        len:   the number of elements which'll take part in calculating
  * @retval return a unsigned char result
  */
static uint8_t xor_sum_calculate(uint8_t *start, uint32_t len)
{
    uint8_t sum = 0x00U;

    while (len)
    {
        sum ^= *start++;
        len--;
    }

    return sum;
}

/**
  * @brief  feedback ACK to upper
  * @param  None
  * @retval None
  */
void feedback_ack(void)
{
    g_fsm_data.tx_len = 1;
    g_fsm_data.tx_buf[0] = POS_ACK;
    (*(g_isp_data.p_send_func))(g_fsm_data.tx_buf, g_fsm_data.tx_len);
}

/**
  * @brief  feedback NACK to upper
  * @param  None
  * @retval None
  */
void feedback_nack(void)
{
    g_fsm_data.tx_len = 1;
    g_fsm_data.tx_buf[0] = NEG_ACK;
    (*(g_isp_data.p_send_func))(g_fsm_data.tx_buf, g_fsm_data.tx_len);
    s_cmd_handle_func_step = 0xFF;
    s_cmd_handle_func_addr_index = 0xFF;
}

/**
  * @brief  change .step to IN_SEND_OVER and .result to FAIL due to various failed procedures
            and prepare for receive another cmd
  * @param  None
  * @retval None
  */
void fail_restart(void)
{
    g_fsm_data.step = IN_FSM_STEP_OTHER;
    g_fsm_data.trigger = 1;
}

/**
  * @brief  init state machine parameters
  * @param  None
  * @retval None
  */
static void fsm_init(void)
{
    g_fsm_data.step = IN_START_RX;
    g_fsm_data.cmd = NONE;
    g_fsm_data.trigger = 0x00;
    g_fsm_data.para1.b32 = 0x00;
    g_fsm_data.para2.b32 = 0x00;
    g_fsm_data.rx_len = 0;
    g_fsm_data.tx_len = 0;
    s_cmd_handle_func_step = 0xFF;
    s_cmd_handle_func_addr_index = 0xFF;
    return;
}

/* Public Function---------------------------------------------------------- */

/**
  * @brief  function to init parameters of state machine, it can only be used after
  *         void fsm_comm_func_init(void) and void fsm_exec_func_init(void) called..
  * @param  None
  * @retval None
  */
void fsm_para_init(void)
{
    /* init the main parameters if state machine */
    g_fsm_data.syn       = 0;
    g_fsm_data.step      = IN_START_HANDLE;
    g_fsm_data.cmd       = NONE;
    g_fsm_data.trigger   = 0;
    s_cmd_handle_func_step = 0xFF;
    s_cmd_handle_func_addr_index = 0xFF;

    /* sync cmd is 0x3F, which byte length is one byte */
    g_fsm_data.rx_len  = 1;
    /* init the state machine receive function and prepare for receive 1 byte to sync state machine */
    (*(g_isp_data.p_recv_func))(g_fsm_data.rx_buf, g_fsm_data.rx_len);
}

#if ES_MICRO_BOOT_CONFIG__SUPPORT_CMD_READ_M
static void es_handle_cmd_read_m(void)
{
    switch (s_cmd_handle_func_step)
    {
        case ES_CMD_STEP_0:
        {
            g_fsm_data.rx_len = 5;
            (*(g_isp_data.p_recv_func))(g_fsm_data.rx_buf, g_fsm_data.rx_len);

            s_cmd_handle_func_step++;
            break;
        }

        case ES_CMD_STEP_1:
        {
            /* using exclusive-or verification method to check the received 1th parameter(generally, the start address) */
            if (g_fsm_data.rx_buf[g_fsm_data.rx_len - 1] == xor_sum_calculate(g_fsm_data.rx_buf, g_fsm_data.rx_len - 1))
            {
                reverse_fill_byte_arr(&g_fsm_data.para1, g_fsm_data.rx_len - 1);
                feedback_ack();
            }
            else
            {
                feedback_nack();
                ES_CMD_ERR_HANDLE1
            }

            s_cmd_handle_func_step++;
            break;
        }

        case ES_CMD_STEP_2:
        {
            /* waitting for receiving 2 byte data which contain 1 byte data indicates the byte length that
            upper computer wanna read (less 1 than the real amount) and 1 byte checksum using exclusive-or */
            g_fsm_data.rx_len = 2;
            (*(g_isp_data.p_recv_func))(g_fsm_data.rx_buf, g_fsm_data.rx_len);

            s_cmd_handle_func_step++;
            break;
        }

        case ES_CMD_STEP_3:
        {
            if (g_fsm_data.rx_buf[0] == (uint8_t)(~g_fsm_data.rx_buf[1]))
            {
                /* verification of byte number which upper computer wanna read is successful */
                g_fsm_data.para2.b32 = g_fsm_data.rx_buf[0] + 1;
                feedback_ack();
            }
            else
            {
                feedback_nack();
                ES_CMD_ERR_HANDLE1
            }

            s_cmd_handle_func_step++;
            break;
        }

        case ES_CMD_STEP_4:
        {
            (*(g_isp_data.p_read_memo))(g_fsm_data.tx_buf, (void *)g_fsm_data.para1.b32, g_fsm_data.para2.b32);
            g_fsm_data.tx_len = g_fsm_data.para2.b32;
            (*(g_isp_data.p_send_func))(g_fsm_data.tx_buf, g_fsm_data.tx_len);

            s_cmd_handle_func_step = 0xFF;
            break;
        }

        default:
        {
            feedback_nack();
            ES_CMD_ERR_HANDLE1
        }
    }
}
#endif/*ES_MICRO_BOOT_CONFIG__SUPPORT_CMD_READ_M*/

#if ES_MICRO_BOOT_CONFIG__SUPPORT_CMD_GO
static void es_handle_cmd_go(void)
{
    switch (s_cmd_handle_func_step)
    {
        case ES_CMD_STEP_0:
        {
            g_fsm_data.rx_len = 5;
            (*(g_isp_data.p_recv_func))(g_fsm_data.rx_buf, g_fsm_data.rx_len);

            s_cmd_handle_func_step++;
            break;
        }

        case ES_CMD_STEP_1:
        {
            /* using exclusive-or verification method to check the received 1th parameter(generally, the start address) */
            if (g_fsm_data.rx_buf[g_fsm_data.rx_len - 1] == xor_sum_calculate(g_fsm_data.rx_buf, g_fsm_data.rx_len - 1))
            {
                reverse_fill_byte_arr(&g_fsm_data.para1, g_fsm_data.rx_len - 1);
                feedback_ack();
            }
            else
            {
                feedback_nack();
                ES_CMD_ERR_HANDLE1
            }

            s_cmd_handle_func_step++;
            break;
        }

        case ES_CMD_STEP_2:
        {
            feedback_ack();
            s_cmd_handle_func_step++;
            break;
        }

        case ES_CMD_STEP_3:
        {
            (*(g_isp_data.p_go))(g_fsm_data.para1.b32);
            s_cmd_handle_func_step = 0xFF;
            break;
        }

        default:
        {
            feedback_nack();
            ES_CMD_ERR_HANDLE1
        }
    }
}
#endif/*ES_MICRO_BOOT_CONFIG__SUPPORT_CMD_GO*/

#if ES_MICRO_BOOT_CONFIG__SUPPORT_CMD_WRITE_M
static void es_handle_cmd_write_m(void)
{
    switch (s_cmd_handle_func_step)
    {
        case ES_CMD_STEP_0:
        {
            g_fsm_data.rx_len = 5;
            (*(g_isp_data.p_recv_func))(g_fsm_data.rx_buf, g_fsm_data.rx_len);

            s_cmd_handle_func_step++;
            break;
        }

        case ES_CMD_STEP_1:
        {
            /* using exclusive-or verification method to check the received 1th parameter(generally, the start address) */
            if (g_fsm_data.rx_buf[g_fsm_data.rx_len - 1] == xor_sum_calculate(g_fsm_data.rx_buf, g_fsm_data.rx_len - 1))
            {
                reverse_fill_byte_arr(&g_fsm_data.para1, 4);
                feedback_ack();
            }
            else
            {
                feedback_nack();
                ES_CMD_ERR_HANDLE1
            }

            s_cmd_handle_func_step++;
            break;
        }

        case ES_CMD_STEP_2:
        {
            /* waitting for receiving 1 byte data which indicates the byte length that upper computer will flash
             (less 1 than the real amount)*/
            g_fsm_data.rx_len = 1;
            (*(g_isp_data.p_recv_func))(g_fsm_data.rx_buf, g_fsm_data.rx_len);

            s_cmd_handle_func_step++;
            break;
        }

        case ES_CMD_STEP_3:
        {
            /* store byte number which upper computer wanna flash to parameter 2, waiting for receiving the rest of
            N+1 byte data which flashed to MCU and 1 byte exclusive-or checksum, and store the (N+1)+1 bytes to memory
            from address buf+1 */
            g_fsm_data.para2.b32 = g_fsm_data.rx_buf[0] + 1;
            g_fsm_data.rx_len = (g_fsm_data.rx_buf[0] + 1) + 1;
            (*(g_isp_data.p_recv_func))(g_fsm_data.rx_buf + 1, g_fsm_data.rx_len);

            s_cmd_handle_func_step++;
            break;
        }

        case ES_CMD_STEP_4:
        {
            /* conduct the exclusive-or checksum for the 1 byte data indicates the data length and N+1 byte
            data which upper computer wanna flash */
            if (g_fsm_data.rx_buf[g_fsm_data.rx_len] == xor_sum_calculate(g_fsm_data.rx_buf, g_fsm_data.rx_len))
            {
#if ES_MICRO_BOOT_CONFIG__WRITE_M_CMD_OPTION__SUPPORT_RAM

                if (g_fsm_data.para1.b32 >= ES_MICRO_BOOT_CONFIG__BOOT_RAM_START_ADDR)
                {
                    /* if address is on sram then memcpy directly */
                    memcpy((void *)g_fsm_data.para1.b32, g_fsm_data.rx_buf + 1, g_fsm_data.para2.b32);
                }
                else
#endif/*ES_MICRO_BOOT_CONFIG__WRITE_M_CMD_OPTION__SUPPORT_RAM*/
                {
                    /* stop frame timer temporarily */
                    g_isp_data.u_frame_timer = 0;

                    if (IAP_FAIL == ((*(g_isp_data.p_words_progrm))(g_fsm_data.para1.b32, (uint32_t)(g_fsm_data.rx_buf + 1), g_fsm_data.para2.b32)))
                    {
                        feedback_nack();
                        ES_CMD_ERR_HANDLE1
                    }

#if ES_MICRO_BOOT_CONFIG__WRITE_M_CMD_OPTION__CHECK_MEM
                    else
                    {
                        /* IAP successfully, read out data from address and check */
                        (*(g_isp_data.p_read_memo))(g_fsm_data.rx_buf + 1, (void *)g_fsm_data.para1.b32, g_fsm_data.para2.b32);

                        if (g_fsm_data.rx_buf[g_fsm_data.rx_len] != xor_sum_calculate(g_fsm_data.rx_buf, g_fsm_data.rx_len))
                        {
                            feedback_nack();
                            ES_CMD_ERR_HANDLE1
                        }
                    }

#endif/*ES_MICRO_BOOT_CONFIG__WRITE_M_CMD_OPTION__CHECK_MEM*/
                    /* stop frame timer temporarily */
                    g_isp_data.u_frame_timer = FRAME_INTERVAL;
                }
            }
            else
            {
                feedback_nack();
                ES_CMD_ERR_HANDLE1
            }

            s_cmd_handle_func_step = 0xFF;
            feedback_ack();
            break;
        }

        default:
        {
            feedback_nack();
            ES_CMD_ERR_HANDLE1
        }
    }
}
#endif/*ES_MICRO_BOOT_CONFIG__SUPPORT_CMD_WRITE_M*/

#if ES_MICRO_BOOT_CONFIG__SUPPORT_CMD_EX_ERASE
static void es_handle_cmd_ex_erase(void)
{
    uint32_t n;

    switch (s_cmd_handle_func_step)
    {
        case ES_CMD_STEP_0:
        {
            g_fsm_data.rx_len = 2;
            (*(g_isp_data.p_recv_func))(g_fsm_data.rx_buf, g_fsm_data.rx_len);

            s_cmd_handle_func_step++;
            break;
        }

        case ES_CMD_STEP_1:
        {
            if (g_fsm_data.rx_buf[0] == 0xFF)
            {
                feedback_nack();
                ES_CMD_ERR_HANDLE1
            }
            else
            {
                /* erase N+1 page */
                reverse_fill_byte_arr(&g_fsm_data.para1, 2);

                /* page erase:
                waitting for receiving the rest of 2*(N+1)+1 bytes indicate the page code and store them in
                memory from the start address buf+2 */
                g_fsm_data.rx_len = (g_fsm_data.para1.b32 + 1) * 2 + 1;

                if (g_fsm_data.rx_len > (BUF_SIZE - 2))
                {
                    feedback_nack();
                    ES_CMD_ERR_HANDLE1
                }
                else
                {
                    (*(g_isp_data.p_recv_func))(g_fsm_data.rx_buf + 2, g_fsm_data.rx_len);
                }
            }

            s_cmd_handle_func_step++;
            break;
        }

        case ES_CMD_STEP_2:
        {
            /* compare xor sum with received, calculate from rx_buf[0] to rx_buf[(2 + rx_len) - 1] */
            if (g_fsm_data.rx_buf[g_fsm_data.rx_len + 1] == xor_sum_calculate(g_fsm_data.rx_buf, g_fsm_data.rx_len + 1))
            {
                /* page amount waitting for erase */
                n = g_fsm_data.para1.b32 + 1;

                while (n)
                {
                    /* each 2 byte represent a page code, the LSB of page code is (2 + 2 * i + 1), and the MSB
                    is (2 + 2 * i), it need to be considerred, the first 2 byte data which represent page amount
                    and the state machine take the strategy that erase in an reversed order */
                    g_fsm_data.para2.b32 = (uint32_t)g_fsm_data.rx_buf[n * 2 + 1] + ((uint32_t)g_fsm_data.rx_buf[n * 2] << 8);

                    if (IAP_FAIL == ((*(g_isp_data.p_page_erase))(g_fsm_data.para2.b32)))
                    {
                        feedback_nack();
                        ES_CMD_ERR_HANDLE1
                    }

                    n--;
                }
            }
            else
            {
                feedback_nack();
                ES_CMD_ERR_HANDLE1
            }

            s_cmd_handle_func_step = 0xFF;
            feedback_ack();
            break;
        }

        default:
        {
            feedback_nack();
            ES_CMD_ERR_HANDLE1
        }
    }
}
#endif/*ES_MICRO_BOOT_CONFIG__SUPPORT_CMD_EX_ERASE*/

#if ES_MICRO_BOOT_CONFIG__SUPPORT_CMD_GET_CRC32
static void es_handle_cmd_get_crc32(void)
{
    uint32_t buf32;

    switch (s_cmd_handle_func_step)
    {
        case ES_CMD_STEP_0:
        {
            g_fsm_data.rx_len = 5;
            (*(g_isp_data.p_recv_func))(g_fsm_data.rx_buf, g_fsm_data.rx_len);

            s_cmd_handle_func_step++;
            break;
        }

        case ES_CMD_STEP_1:
        {
            /* using exclusive-or verification method to check the received 1th parameter(generally, the start address) */
            if (g_fsm_data.rx_buf[g_fsm_data.rx_len - 1] == xor_sum_calculate(g_fsm_data.rx_buf, g_fsm_data.rx_len - 1))
            {
                reverse_fill_byte_arr(&g_fsm_data.para1, 4);

                if ((g_fsm_data.para1.b32 & 0x3) == 0)
                {
                    feedback_ack();
                }
                else
                {
                    feedback_nack();
                    ES_CMD_ERR_HANDLE1
                }
            }
            else
            {
                feedback_nack();
                ES_CMD_ERR_HANDLE1
            }

            s_cmd_handle_func_step++;
            break;
        }

        case ES_CMD_STEP_2:
        {
            /* waitting for receiving 5 byte data which contains 4 byte data indicate the amount of byte which
            upper computer wanna check (less 1 than the real amount) and 1 byte exclusive-or checksum */
            g_fsm_data.rx_len = 5;
            (*(g_isp_data.p_recv_func))(g_fsm_data.rx_buf, g_fsm_data.rx_len);

            s_cmd_handle_func_step++;
            break;
        }

        case ES_CMD_STEP_3:
        {
            if (g_fsm_data.rx_buf[g_fsm_data.rx_len - 1] == xor_sum_calculate(g_fsm_data.rx_buf, g_fsm_data.rx_len - 1))
            {
                reverse_fill_byte_arr(&g_fsm_data.para2, 4);

                if (((g_fsm_data.para2.b32 + 1) & 0x3) == 0)
                {
                    feedback_ack();
                }
                else
                {
                    feedback_nack();
                    ES_CMD_ERR_HANDLE1
                }
            }
            else
            {
                feedback_nack();
                ES_CMD_ERR_HANDLE1
            }

            s_cmd_handle_func_step++;
            break;
        }

        case ES_CMD_STEP_4:
        {
            buf32 = (*(g_isp_data.p_get_crc32))((uint32_t *)g_fsm_data.para1.b32, g_fsm_data.para2.b32 + 1);

            g_fsm_data.tx_len = 4;
            memcpy(g_fsm_data.tx_buf, (const void *)&buf32, g_fsm_data.tx_len);
            (*(g_isp_data.p_send_func))(g_fsm_data.tx_buf, g_fsm_data.tx_len);

            s_cmd_handle_func_step = 0xFF;
            break;
        }

        default:
        {
            feedback_nack();
            ES_CMD_ERR_HANDLE1
        }
    }
}
#endif/*ES_MICRO_BOOT_CONFIG__SUPPORT_CMD_GET_CRC32*/

#if ES_MICRO_BOOT_CONFIG__SUPPORT_CMD_CHECK_EMPTY
static void es_handle_cmd_check_empty(void)
{
    switch (s_cmd_handle_func_step)
    {
        case ES_CMD_STEP_0:
        {
            g_fsm_data.rx_len = 5;
            (*(g_isp_data.p_recv_func))(g_fsm_data.rx_buf, g_fsm_data.rx_len);

            s_cmd_handle_func_step++;
            break;
        }

        case ES_CMD_STEP_1:
        {
            /* using exclusive-or verification method to check the received 1th parameter(generally, the start address) */
            if (g_fsm_data.rx_buf[g_fsm_data.rx_len - 1] == xor_sum_calculate(g_fsm_data.rx_buf, g_fsm_data.rx_len - 1))
            {
                reverse_fill_byte_arr(&g_fsm_data.para1, 4);

                if ((g_fsm_data.para1.b32 & 0x3) == 0)
                {
                    feedback_ack();
                }
                else
                {
                    feedback_nack();
                    ES_CMD_ERR_HANDLE1
                }
            }
            else
            {
                feedback_nack();
                ES_CMD_ERR_HANDLE1
            }

            s_cmd_handle_func_step++;
            break;
        }

        case ES_CMD_STEP_2:
        {
            /* waitting for receiving 5 byte data which contains 4 byte data indicate the amount of byte which
            upper computer wanna check (less 1 than the real amount) and 1 byte exclusive-or checksum */
            g_fsm_data.rx_len = 5;
            (*(g_isp_data.p_recv_func))(g_fsm_data.rx_buf, g_fsm_data.rx_len);

            s_cmd_handle_func_step++;
            break;
        }

        case ES_CMD_STEP_3:
        {
            if (g_fsm_data.rx_buf[g_fsm_data.rx_len - 1] == xor_sum_calculate(g_fsm_data.rx_buf, g_fsm_data.rx_len - 1))
            {
                reverse_fill_byte_arr(&g_fsm_data.para2, 4);

                if (((g_fsm_data.para2.b32 + 1) & 0x3) == 0)
                {
                    feedback_ack();
                }
                else
                {
                    feedback_nack();
                    ES_CMD_ERR_HANDLE1
                }
            }
            else
            {
                feedback_nack();
                ES_CMD_ERR_HANDLE1
            }

            s_cmd_handle_func_step++;
            break;
        }

        case ES_CMD_STEP_4:
        {
            if ((*(g_isp_data.p_check_empty))((uint32_t *)g_fsm_data.para1.b32, g_fsm_data.para2.b32 + 1) == PASS)
            {
                feedback_ack();
            }
            else
            {
                feedback_nack();
                ES_CMD_ERR_HANDLE1
            }

            s_cmd_handle_func_step = 0xFF;
            break;
        }

        default:
        {
            feedback_nack();
            ES_CMD_ERR_HANDLE1
        }
    }
}
#endif/*ES_MICRO_BOOT_CONFIG__SUPPORT_CMD_CHECK_EMPTY*/

static void es_handle_cmd(uint8_t cmd)
{
    s_cmd_handle_func_addr_index = 0xFF;
    s_cmd_handle_func_step = 0;

    switch (cmd)
    {
#if ES_MICRO_BOOT_CONFIG__SUPPORT_CMD_READ_M

        case  READ_M:
            s_cmd_handle_func_addr_index = ES_INDEX_CMD_READ_M;
            break;
#endif/*ES_MICRO_BOOT_CONFIG__SUPPORT_CMD_READ_M*/

#if ES_MICRO_BOOT_CONFIG__SUPPORT_CMD_GO

        case  GO:
            s_cmd_handle_func_addr_index = ES_INDEX_CMD_GO;
            break;
#endif/*ES_MICRO_BOOT_CONFIG__SUPPORT_CMD_GO*/

#if ES_MICRO_BOOT_CONFIG__SUPPORT_CMD_WRITE_M

        case  WRITE_M:
            s_cmd_handle_func_addr_index = ES_INDEX_CMD_WRITE_M;
            break;
#endif/*ES_MICRO_BOOT_CONFIG__SUPPORT_CMD_WRITE_M*/

#if ES_MICRO_BOOT_CONFIG__SUPPORT_CMD_EX_ERASE

        case  EX_ERASE:
            s_cmd_handle_func_addr_index = ES_INDEX_CMD_EX_ERASE;
            break;
#endif/*ES_MICRO_BOOT_CONFIG__SUPPORT_CMD_EX_ERASE*/

#if ES_MICRO_BOOT_CONFIG__SUPPORT_CMD_GET_CRC32

        case  GET_CRC32:
            s_cmd_handle_func_addr_index = ES_INDEX_CMD_GET_CRC32;
            break;
#endif/*ES_MICRO_BOOT_CONFIG__SUPPORT_CMD_GET_CRC32*/

#if ES_MICRO_BOOT_CONFIG__SUPPORT_CMD_CHECK_EMPTY

        case  CHECK_EMPTY:
            s_cmd_handle_func_addr_index = ES_INDEX_CMD_CHECK_EMPTY;
            break;
#endif/*ES_MICRO_BOOT_CONFIG__SUPPORT_CMD_GET_CRC32*/

        case  GET:
        case  GET_V:
        case  GET_ID:
        default:
            g_fsm_data.cmd = NONE;
            feedback_nack();
            break;
    }

    if (g_fsm_data.cmd != NONE)
    {
        feedback_ack();
        g_fsm_data.step = IN_CMD_HANDLE_FUNC;
    }
}


/**
  * @brief  main logic of state machine operation
  * @param  None
  * @retval None
  */
void proc_fsm(void)
{
    do
    {
        g_fsm_data.trigger = 0;

        if ((s_cmd_handle_func_addr_index != 0) && (s_cmd_handle_func_addr_index <= s_cmd_handle_func_addr_buf_size))
            s_cmd_handle_func_addr_buf[s_cmd_handle_func_addr_index - 1]();
        else
            s_cmd_handle_func_step = 0xFF;

        switch (g_fsm_data.step)
        {
            case  IN_START_RX:
            {
                g_fsm_data.step = IN_START_HANDLE;

                /* keep resetting frame timer before receive valid cmd */
                g_isp_data.u_frame_timer = 0x00U;

                /* waitting for receiving the first byte for cmd */
                g_fsm_data.rx_len = 1;
                (*(g_isp_data.p_recv_func))(g_fsm_data.rx_buf, g_fsm_data.rx_len);
                break;
            }

            case  IN_START_HANDLE:
            {
                g_fsm_data.step = IN_START_RX;

                if (g_fsm_data.rx_buf[0] == START)
                {
                    /* keep resetting frame timer before receive valid cmd */
                    g_isp_data.u_frame_timer = 0x00U;

                    /* change .syn to 1 if sync success */
                    g_fsm_data.syn  = 0x01;

                    feedback_ack();
                }
                else
                {
                    if (g_fsm_data.syn == 0x01)
                    {
                        g_fsm_data.step = IN_START_CMD_HANDLE;/* if receive a byte of valid or invalid cmd, change .step to WAIT_REV_INS */

                        /* waitting for receiving the second byte of cmd and put it in rx_buf+1 */
                        g_fsm_data.rx_len = 1;
                        (*(g_isp_data.p_recv_func))(g_fsm_data.rx_buf + 1, g_fsm_data.rx_len);
                    }
                    else
                    {
                        feedback_nack();
                    }
                }

                break;
            }

            case IN_START_CMD_HANDLE:
            {
                g_fsm_data.step = IN_START_RX;

                if (g_fsm_data.rx_buf[0] == (uint8_t)(~g_fsm_data.rx_buf[1]))/* verify if valid format */
                {
                    g_fsm_data.cmd = (fsm_cmd_t)g_fsm_data.rx_buf[0];
                    es_handle_cmd(g_fsm_data.cmd);
                }
                else
                {
                    feedback_nack();
                }

                break;
            }

            case IN_CMD_HANDLE_FUNC:
            {
                if (s_cmd_handle_func_step == 0xFF)
                {
                    g_isp_data.u_frame_timer = 0x00U;
                    s_cmd_handle_func_addr_index = 0xFF;
                    g_fsm_data.step = IN_START_RX;
                    g_fsm_data.cmd = NONE;
                }

                break;
            }

            default:    /* if unknown state then jump to the state of waitting for cmd */
            {
                fsm_init();

                /* execution over, stop frame timer before next execution */
                g_isp_data.u_frame_timer = 0x00U;
                g_fsm_data.trigger = 1;

                break;
            }
        }
    }
    while (g_fsm_data.trigger);
}

/**
  * @brief  function to process the subsequent logic of state machine when receive enough bytes,
            it must be called when it has received the specified byte length.
  * @param  None
  * @retval None
  */
void fsm_recv_over(void)
{
    while ((uint8_t)(*(g_isp_data.p_is_recv_over))());

    g_fsm_data.trigger = 1;
}

/**
  * @brief  function to process the subsequent logic of state machine when send enough bytes,
            it must be called when it has sent specified byte length.
  * @param  None
  * @retval None
  */
void fsm_send_over(void)
{
    while ((uint8_t)(*(g_isp_data.p_is_send_over))());

    g_fsm_data.trigger = 1;
}

/**
  * @brief  function to detect whether state machine is triggered, it'll return 1 if triggered, else 0.
  * @param  None
  * @retval
  */
uint8_t fsm_is_trigger(void)
{
    return g_fsm_data.trigger;
}

/************* (C) COPYRIGHT Eastsoft Microelectronics *****END OF FILE****/
