/**********************************************************************************
 *
 * @file    es32_dsp.c
 * @brief   ES32 fixed point dsp
 *
 * @date    09 Feb  2023
 * @author  AE Team
 * @note
 *          Change Logs:
 *          Date            Author          Notes
 *          09 Feb  2023    AE Team         the first version
 *
 * Copyright (C) Shanghai Eastsoft Microelectronics Co. Ltd. All rights reserved.
 *
 * SPDX-License-Identifier: Apache-2.0
 *
 * Licensed under the Apache License, Version 2.0 (the License); you may
 * not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an AS IS BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 *
 **********************************************************************************
 */
#ifndef _ES32_DSP_H_
#define _ES32_DSP_H_

/* Includes ------------------------------------------------------------------ */
#include <stdint.h>

/* Exported Macros ----------------------------------------------------------- */
/**
  * @brief  IIR 16-bit fixed points
  */
#define IIR_FIXPTS 12
/* Exported Types ------------------------------------------------------------ */
/** 
  * @brief PID state
  */
typedef struct
{
	int16_t Kp;
	int16_t Ki;
	int16_t Kd;
	int16_t prev_err;
	int32_t integral;
}pid_state_t;
/* Exported Variables -------------------------------------------------------- */

/* Exported Constants -------------------------------------------------------- */

/* Exported Functions -------------------------------------------------------- */
/**
  * @brief  Calculate 64 points complex fft with 
  * @param  output: output data buffer
  * @param  input:  input data 
  * @param  N:		input data len (must be 64)
  * @retval None
  */
void cfft_r4_64(int16_t *output,const int16_t*input,int N);


/**
  * @brief  Calculate 256 points complex fft
  * @param  output: output data buffer
  * @param  input:  input data 
  * @param  N:		input data len (must be 256)
  * @retval None
  */
void cfft_r4_256(int16_t *output,const int16_t*input,int N);


/**
  * @brief  Calculate 1024 points complex fft
  * @param  output: output data buffer
  * @param  input:  input data 
  * @param  N:		input data len (must be 1024)
  * @retval None
  */
void cfft_r4_1024(int16_t *output,const int16_t*input,int N);

/**
  * @brief  FIR 16-bit filter in assembly
  * @param  output: output data buffer
  * @param  input:  input data 
  * @param  N:		input data len (multiply of 4)
  * @param  coeff:	coefficents data
  * @param  Ncoeff:	coefficents data len (multiply of 4)
  * @retval None
  */
void fir_16by16(int *output,const int16_t *input, uint32_t N,const int16_t *coeff,uint32_t Ncoeff);

/**
  * @brief  IIR 16-bit filter
  * @param  y:			output
  * @param  x:			input
  * @param  IIRCoeff:	coeff data(20 shorts)
  * @param  ny:			number of output
  * @retval None
  */
  
void iir_biquad(uint16_t *y,const uint16_t *x,const int16_t *IIRCoeff, uint16_t ny);
/**
  * @brief  IIR 16-bit filter
  * @param  y:	output (length == ny+4)
  * @param  x:	input (length == ny+4)
  * @param  h2:	AutoRegressive part Filter Coefficients(5 shorts)
  * @param  h1:	Moving Average part Filter Coefficients(5 shorts)
  * @param  ny:	number of output (>=2)
  * @retval None
  */
void iir_arma(uint16_t *y,const uint16_t *x, uint16_t *h2, uint16_t *h1, uint16_t ny);

/**
  * @brief  IIR 16-bit fixed point filter
  * @param  y:			output
  * @param  x:			input
  * @param  IIRCoeff:	coeff data(20 shorts)
  * @param  ny:			number of output
  * @retval None
  */
void iir_biquad_pts(int16_t *y,const int16_t *x,const int16_t *IIRCoeff, uint16_t ny);

/**
  * @brief  Do pid in asm, Recommended
  * @param  Error: difference between reference and measured value
  * @param  state: PID state,includes PID params(Kp, Ki, Kd) and state data(prev_error, Intergral)
  * @retval PID Output
  */
int do_pid_state(int16_t error,pid_state_t*state);

/**
  * @brief  Do pid in asm, state data stored internally
  * @param  Error: difference between reference and measured value
  * @param  coeff: PID params(Kp, Ki, Kd)
  * @retval PID Output
  */
int do_pid_asm(int16_t error,int16_t *coeff);

/**
  * @brief  Do pid in C, state data stored internally
  * @param  Error: difference between reference and measured value
  * @param  coeff: PID params(Kp, Ki, Kd)
  * @retval PID Output
  */
int do_pid_c(int16_t error, int16_t *coeff);

/**
  * @brief  Do pid in C, state data stored internally
  * @param  in:  input value
  * @param  ref: reference value
  * @param  coeff: PID params(Kp, Ki, Kd)
  * @retval PID Output
  */
int do_full_pid(int16_t in, int16_t ref, int16_t *coeff);

/** 
  * @brief  Set internal integral and prev_error for DoPID and DoFullPID
  * @param  Intgral: new integral value
  * @param  prev_error: new prev_error value
  * @retval None
  */
void reset_pid_c(uint16_t integral,uint16_t prev_error);


/** 
  * @brief  Set internal integral and prev_error for do_pid
  * @param  Intgral: new integral value
  * @param  prev_error: new prev_error value
  * @retval None
  */
void reset_pid_asm(int integral,int prev_error);

#endif /* _ES32_DSP_H_ */