/**********************************************************************************
 *
 * @file    c_pid.c
 * @brief   PID in C
 *
 * @date    09 Feb  2023
 * @author  AE Team
 * @note
 *          Change Logs:
 *          Date            Author          Notes
 *          09 Feb  2023    AE Team         the first version
 *
 * Copyright (C) Shanghai Eastsoft Microelectronics Co. Ltd. All rights reserved.
 *
 * SPDX-License-Identifier: Apache-2.0
 *
 * Licensed under the Apache License, Version 2.0 (the License); you may
 * not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an AS IS BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 *
 **********************************************************************************
 */
/* Includes ------------------------------------------------------------------*/
#include <stdint.h>

/* Private Macros ------------------------------------------------------------ */

/* Private Constants --------------------------------------------------------- */

/* Private function prototypes ----------------------------------------------- */

/* Private Variables --------------------------------------------------------- */

/* Public Variables ---------------------------------------------------------- */
int32_t g_int_term=0,g_prev_error=0;
int32_t g_int_term_c=0,g_prev_error_c=0;

/* Private Function ---------------------------------------------------------- */

/**
  * @brief  PID in C, error computed outside the routine
  * @param ror: difference between reference and measured value
  *   coeff: pointer to PID params
  * @retval : PID output (command)
  */
int do_pid_c(int16_t error, int16_t *coeff)
{
  int32_t Kp, Ki, Kd, output;

  Kp = coeff[0];
  Ki = coeff[1];
  Kd = coeff[2];

  g_int_term_c += Ki*error;
  output = Kp * error;
  output += g_int_term_c;
  output += Kd * (error - g_prev_error_c);

  g_prev_error_c = error;

  return (output);
}

/**
  * @brief  PID in C, error computed inside the routine
  * @param : Input (measured value)
  *   ref: reference (target value)
  *   coeff: pointer to PID params
  * @retval : PID output (command)
  */
int do_full_pid(int16_t input, int16_t ref, uint16_t *coeff)
{
  int32_t Kp, Ki, Kd, output, error;

  error = ref - input;
  Kp = coeff[0];
  Ki = coeff[1];
  Kd = coeff[2];

  g_int_term_c += Ki*error;
  output = Kp * error;
  output += g_int_term_c;
  output += Kd * (error - g_prev_error_c);

  g_prev_error_c = error;

  return (output);
}

void reset_pid_c(uint16_t integral,uint16_t prev_err)
{
	g_int_term_c=integral;
	g_prev_error_c=prev_err;
}

void reset_pid_asm(int integral,int prev_err)
{
	g_int_term=integral;
	g_prev_error=prev_err;
}